﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/spy/spy_Config.h>

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

#include <limits>

#include <nn/audio.h>
#include <nn/util/util_BytePtr.h>
#include <nn/spy/spy_SpyController.h>
#include <nn/spy/audio/detail/spy_WaveformModule.h>

namespace nn {
namespace spy {
namespace audio {
namespace detail {

NN_DEFINE_STATIC_CONSTANT(const uint32_t WaveformPacketData::PacketVersion);
NN_DEFINE_STATIC_CONSTANT(const int WaveformPacketData::MaxChannelCount);
NN_DEFINE_STATIC_CONSTANT(const int WaveformPacketData::MaxSamplePerFrame);

namespace {

    static const ::nn::audio::AudioDeviceName DeviceNameAudioTvOutput = { "AudioTvOutput" };

    NN_STATIC_ASSERT(
        static_cast<int>(WaveformPacketData::SampleFormat_PcmInt16) ==
        static_cast<int>(nn::audio::SampleFormat_PcmInt16));

#if defined(NN_DETAIL_ENABLE_SDK_ASSERT)
    bool CheckChannelTypes(const WaveformModule::ChannelType array[], int count)
    {
        uint32_t bits = 0;

        for (int i = 0; i < count; ++i)
        {
            // 列挙値は範囲内か。
            if (array[i] > WaveformModule::ChannelType::ChannelType_Max)
            {
                return false;
            }

            // 同じ値が指定されていないか。
            if (bits & (1 << array[i]))
            {
                return false;
            }
            bits |= 1 << array[i];
        }

        return true;
    }
#endif

}

//----------------------------------------------------------
WaveformModule::WaveformModule() NN_NOEXCEPT
    : SpyModule("NnAudioWaveform", WaveformPacketData::PacketVersion)
    , m_WaveformMetadataPacket({})
    , m_DeviceVolume(1.0f)
    , m_IsDataRequested(false)
    , m_IsAudioTvOutputAvailable(false)
    , m_IsInitialized(false)
{
    m_WaveformMetadataPacket.packetType = WaveformPacketData::PacketType_WaveformMetadata;
}

bool WaveformModule::Initialize(const InitializeArg& initializeArg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    NN_SDK_REQUIRES(!IsRegistered());
    NN_SDK_REQUIRES_NOT_NULL(initializeArg.pAudioRendererParameter);
    NN_SDK_REQUIRES_MINMAX(initializeArg.waveformChannelCount, 0, WaveformPacketData::MaxChannelCount);
    NN_SDK_REQUIRES(initializeArg.waveformChannelCount == 0 || initializeArg.waveformSampleFormat != nn::audio::SampleFormat_Invalid);
    NN_SDK_REQUIRES(initializeArg.waveformChannelCount == 0 || initializeArg.pWaveformChannelTypes != nullptr);
    NN_SDK_REQUIRES(initializeArg.waveformChannelCount == 0 || CheckChannelTypes(initializeArg.pWaveformChannelTypes, initializeArg.waveformChannelCount));
    NN_SDK_REQUIRES(initializeArg.waveformChannelCount == 0 || initializeArg.pAudioRendererParameter->sampleCount > 0);

    m_IsInitialized = true;

    m_WaveformMetadataPacket.channelCount = static_cast<uint8_t>(initializeArg.waveformChannelCount);
    if (initializeArg.waveformChannelCount == 0)
    {
        return true;
    }

    NN_SDK_ASSERT(initializeArg.pAudioRendererParameter->sampleCount <= WaveformPacketData::MaxSamplePerFrame);
    NN_SDK_ASSERT(initializeArg.waveformSampleFormat == nn::audio::SampleFormat_PcmInt16); // 他は未実装

    m_WaveformMetadataPacket.sampleFormat = static_cast<uint8_t>(initializeArg.waveformSampleFormat);
    m_WaveformMetadataPacket.sampleRate = initializeArg.pAudioRendererParameter->sampleRate;
    m_WaveformMetadataPacket.sampleCountPerAudioFrame = initializeArg.pAudioRendererParameter->sampleCount;

    for (int i = 0; i < m_WaveformMetadataPacket.channelCount; ++i)
    {
        auto& spec = m_WaveformMetadataPacket.channelInfos[i];
        spec.channelType = static_cast<uint8_t>(initializeArg.pWaveformChannelTypes[i]);
    }

    // nn::audio の出力デバイスに "AudioTvOutput" があるかを確認します。
    m_DeviceVolume = 1.0f;
    m_IsAudioTvOutputAvailable = false;
    {
        const int AudioDeviceMaxCount = 10; // NOTE: サンプルコードから拝借した適当な数値
        nn::audio::AudioDeviceName deviceNames[AudioDeviceMaxCount];

        int actualCount = nn::audio::ListAudioDeviceName(deviceNames, AudioDeviceMaxCount);
        for (int i = 0; i < actualCount; ++i)
        {
            if (std::strcmp(deviceNames[i].name, DeviceNameAudioTvOutput.name) == 0)
            {
                m_IsAudioTvOutputAvailable = true;
                break;
            }
        }
    }

    return true;
}

void WaveformModule::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(!IsRegistered());

    m_IsInitialized = false;
}

bool WaveformModule::PushWaveform(
    const void* buffer,
    size_t bufferSize,
    nn::os::Tick tick) NN_NOEXCEPT
{
    // Note:
    // 同時に取得したブロックのうち、最後(最新)のブロックがすでに再生中か、
    // 遅くとも tick から 1 オーディオフレーム以内に再生を開始すると考えられる。
    // 厳密には分からないので、tick から再生されると仮定する。

    NN_SDK_ASSERT(IsInitialized());

    if (m_WaveformMetadataPacket.channelCount == 0)
    {
        return false;
    }

    if (!IsRequested())
    {
        return false;
    }

    if (m_IsDataRequested.exchange(false))
    {
        m_DeviceVolume = 1.0f;

        PushWaveformMetadata();
    }

    // TV のボリュームを送信します。
    if (m_IsAudioTvOutputAvailable)
    {
        float volume = nn::audio::GetAudioDeviceOutputVolume(&DeviceNameAudioTvOutput);
        if (volume != m_DeviceVolume)
        {
            m_DeviceVolume = volume;

            WaveformPacketData::DeviceOutputVolumePacket packet = {};
            packet.packetType = WaveformPacketData::PacketType_DeviceOutputVolume;
            packet.volume = volume;
            if (!PushDataAt(packet, tick))
            {
                return false;
            }
        }
    }

    const auto channelCount = m_WaveformMetadataPacket.channelCount;
    const auto sampleCount = m_WaveformMetadataPacket.sampleCountPerAudioFrame;

    NN_SDK_ASSERT(m_WaveformMetadataPacket.sampleFormat == WaveformPacketData::SampleFormat_PcmInt16); // 他は未サポート
    const size_t blockSize = channelCount * static_cast<uint32_t>(sampleCount) * sizeof(int16_t);
    NN_SDK_ASSERT(bufferSize % blockSize == 0);

    const size_t totalSampleCount = bufferSize / channelCount / sizeof(int16_t);
    NN_SDK_ASSERT(totalSampleCount <= std::numeric_limits<uint32_t>::max());
    NN_SDK_ASSERT(totalSampleCount % sampleCount == 0);

    WaveformPacketData::WaveformPacket packet = {};
    packet.packetType = WaveformPacketData::PacketType_Waveform;
    packet.sampleCount = static_cast<uint16_t>(sampleCount);

    nn::util::ConstBytePtr bufferCurrent(buffer);

    const void* buffers[2] = { &packet, nullptr };
    size_t lengths[2] = { sizeof(packet), blockSize };

    for (size_t sampleIndex = 0;
        sampleIndex < totalSampleCount;
        sampleIndex += sampleCount)
    {
        packet.sampleIndexOnTick = static_cast<int32_t>(totalSampleCount - sampleCount - sampleIndex);

        NN_SDK_ASSERT(-bufferCurrent.Distance(buffer) + blockSize <= bufferSize);

        buffers[1] = bufferCurrent.Get();
        if (!PushDataAt(buffers, lengths, 2, tick))
        {
            return false;
        }

        bufferCurrent.Advance(blockSize);
    }

    return true;
}

void WaveformModule::OnRegistered() NN_NOEXCEPT
{
    Base::OnRegistered();

    NN_SDK_ASSERT(IsInitialized());
}

void WaveformModule::OnRequested(bool isRequested) NN_NOEXCEPT
{
    // IsRequested() が true になった時点では m_IsDataRequested はすでに設定されているようにします。
    // IsRequested() が true に変化するときだけ m_IsDataRequested を設定します。
    if (!IsRequested() && isRequested)
    {
        m_IsDataRequested = isRequested;
    }

    Base::OnRequested(isRequested);
}

void WaveformModule::PushWaveformMetadata() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized());

    auto packetSize =
        sizeof(m_WaveformMetadataPacket)
        - sizeof(m_WaveformMetadataPacket.channelInfos)
        + sizeof(m_WaveformMetadataPacket.channelInfos[0]) * m_WaveformMetadataPacket.channelCount;
    PushData(&m_WaveformMetadataPacket, packetSize);
}

} // namespace nn::spy::audio::detail
} // namespace nn::spy::audio
} // namespace nn::spy
} // namespace nn

#endif // NN_BUILD_CONFIG_SPY_ENABLED
