﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/spy/audio/detail/spy_PlotAudioPerformanceMetrics.h>

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

#include <limits>
#include <new>

#include <nn/nn_Allocator.h>
#include <nn/util/util_BytePtr.h>
#include <nn/util/util_FormatString.h>

namespace nn {
namespace spy {
namespace audio {
namespace detail {

namespace {

    static const double PlotMinValue = 0;
    static const double PlotMaxValue = 6000;

}

PlotPerformanceItem::PlotPerformanceItem() NN_NOEXCEPT
    : m_pPlotModule(nullptr)
{
}

void PlotPerformanceItem::Initialize(
    const char* name,
    uint8_t r,
    uint8_t g,
    uint8_t b) NN_NOEXCEPT
{
    m_Node.SetName(name);
    m_Node.SetColor(r, g, b);

    m_StartTime.SetName("StartTime");
    m_StartTime.SetRange(PlotMinValue, PlotMaxValue);
    m_StartTime.SetColor(r, g, b);
    m_StartTime.SetParent(&m_Node);

    m_ProcessingTime.SetName("ProcessingTime");
    m_ProcessingTime.SetRange(PlotMinValue, PlotMaxValue);
    m_ProcessingTime.SetColor(r, g, b);
    m_ProcessingTime.SetParent(&m_Node);
}

void PlotPerformanceItem::Finalize() NN_NOEXCEPT
{
    DetachFromModule();
}

void PlotPerformanceItem::AttachToModule(nn::spy::PlotModule& plotModule) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsAttachedToModule());

    plotModule.AttachItem(m_Node);
    plotModule.AttachItem(m_StartTime);
    plotModule.AttachItem(m_ProcessingTime);

    m_pPlotModule = &plotModule;
}

void PlotPerformanceItem::DetachFromModule() NN_NOEXCEPT
{
    if(!IsAttachedToModule())
    {
        return;
    }

    m_pPlotModule->DetachItem(m_StartTime);
    m_pPlotModule->DetachItem(m_ProcessingTime);
    m_pPlotModule->DetachItem(m_Node);

    m_pPlotModule = nullptr;
}

bool PlotPerformanceItem::PushValueAt(
    int startTime,
    int processingTime,
    nn::os::Tick tick) NN_NOEXCEPT
{
    m_StartTime.PushValueAt(startTime, tick);
    m_ProcessingTime.PushValueAt(processingTime, tick);

    return true;
}

//----------------------------------------

PlotPerformanceEntry::PlotPerformanceEntry() NN_NOEXCEPT
    : m_IsActive(false)
    , m_IsPushed(false)
{
}

void PlotPerformanceEntry::Initialize(
    const char* name,
    uint8_t r,
    uint8_t g,
    uint8_t b) NN_NOEXCEPT
{
    m_PlotItem.Initialize(name, r, g, b);
    m_NodeId = 0;
    m_IsActive = false;
    m_IsPushed = false;
}

void PlotPerformanceEntry::Finalize() NN_NOEXCEPT
{
    m_PlotItem.Finalize();
}

bool PlotPerformanceEntry::PushValueAt(
    const nn::audio::PerformanceEntry* pPerformanceEntry,
    nn::os::Tick tick) NN_NOEXCEPT
{
    m_IsActive = true;
    m_IsPushed = true;

    return m_PlotItem.PushValueAt(
        pPerformanceEntry->startTime,
        pPerformanceEntry->processingTime,
        tick);
}

bool PlotPerformanceEntry::PushValueAt(
    int startTime,
    int processingTime,
    nn::os::Tick tick) NN_NOEXCEPT
{
    return m_PlotItem.PushValueAt(
        startTime,
        processingTime,
        tick);
}

//----------------------------------------

PlotPerformanceSummary::PlotPerformanceSummary() NN_NOEXCEPT
    : m_EntryList(nullptr)
    , m_CountMax(0)
    , m_Count(0)
    , m_NewCount(0)
    , m_StartTime(0)
    , m_EndTime(0)
{
}

void PlotPerformanceSummary::Initialize(
    const char* name,
    uint8_t r,
    uint8_t g,
    uint8_t b,
    PlotPerformanceEntry* list,
    int count) NN_NOEXCEPT
{
    m_PlotItem.Initialize(name, r, g, b);
    m_EntryList = list;
    m_CountMax = count;
    m_Count = 0;
}

void PlotPerformanceSummary::Finalize() NN_NOEXCEPT
{
    m_PlotItem.Finalize();
}

void PlotPerformanceSummary::PrepareForFrame() NN_NOEXCEPT
{
    m_StartTime = std::numeric_limits<int>::max();
    m_EndTime = 0;
    m_NewCount = 0;

    for (int i = 0; i < m_Count; ++i)
    {
        m_EntryList[i].SetIsPushed(false);
    }
}

void PlotPerformanceSummary::PushSucceedingPerformanceEntry(
    const nn::audio::PerformanceEntry& entry,
    nn::os::Tick tick) NN_NOEXCEPT
{
    int index = GetEntryIndex(entry.id);
    if (index >= 0)
    {
        m_StartTime = std::min(m_StartTime, static_cast<int>(entry.startTime));
        m_EndTime = std::max(m_EndTime, static_cast<int>(entry.startTime + entry.processingTime));

        m_EntryList[index].PushValueAt(&entry, tick);

        m_NewCount = std::max(m_NewCount, index + 1);
    }
}

void PlotPerformanceSummary::PushNewPerformanceEntry(
    const nn::audio::PerformanceEntry& entry,
    nn::os::Tick tick) NN_NOEXCEPT
{
    int index = GetEntryIndex(entry.id);
    if (index >= 0)
    {
        return;
    }

    index = GetInactiveEntryIndex();
    if (index >= 0)
    {
        m_StartTime = std::min(m_StartTime, static_cast<int>(entry.startTime));
        m_EndTime = std::max(m_EndTime, static_cast<int>(entry.startTime + entry.processingTime));

        m_EntryList[index].SetNodeId(entry.id);
        m_EntryList[index].PushValueAt(&entry, tick);

        m_NewCount = std::max(m_NewCount, index + 1);
    }
}

int PlotPerformanceSummary::GetActiveEntryCount() const NN_NOEXCEPT
{
    int result = 0;

    for (int i = 0; i < m_Count; ++i)
    {
        if (m_EntryList[i].IsActive() == true)
        {
            result++;
        }
    }

    return result;
}

int PlotPerformanceSummary::GetEntryIndex(nn::audio::NodeId nodeId) NN_NOEXCEPT
{
    for (int i = 0; i < m_CountMax; ++i)
    {
        if (m_EntryList[i].IsActive() == true && m_EntryList[i].GetNodeId() == nodeId)
        {
            return i;
        }
    }

    return -1;
}

int PlotPerformanceSummary::GetInactiveEntryIndex() NN_NOEXCEPT
{
    for (int i = 0; i < m_CountMax; ++i)
    {
        if (m_EntryList[i].IsActive() == false)
        {
            return i;
        }
    }

    return -1;
}

void PlotPerformanceSummary::PushValueForInactiveItem(nn::os::Tick lastTick) NN_NOEXCEPT
{
    // データの無くなったエントリーを非アクティブにします。
    for (int i = 0; i < m_Count; ++i)
    {
        auto& item = m_EntryList[i];
        if (item.IsActive() == true && item.IsPushed() == false)
        {
            item.PushValueAt(0, 0, lastTick);
            item.SetIsActive(false);
        }
    }
}

bool PlotPerformanceSummary::PushSummaryAt(nn::os::Tick tick) NN_NOEXCEPT
{
    if (m_StartTime <= m_EndTime)
    {
        m_PlotItem.PushValueAt(m_StartTime, m_EndTime - m_StartTime, tick);
    }
    else
    {
        m_PlotItem.PushValueAt(0, 0, tick);
    }

    m_Count = m_NewCount;

    return true;
}

//----------------------------------------

size_t PlotAudioPerformanceMetrics::GetRequiredMemorySize(
    const InitializeArg& arg) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(NN_ALIGNOF(PlotPerformanceEntry) <= ::nn::DefaultAlignment);

    NN_SDK_REQUIRES_NOT_NULL(arg.pAudioRendererParameter);

    auto& parameter = *arg.pAudioRendererParameter;

    return sizeof(PlotPerformanceEntry) * parameter.voiceCount
        + sizeof(PlotPerformanceEntry) * parameter.subMixCount
        + sizeof(PlotPerformanceEntry) * parameter.sinkCount;
}

PlotAudioPerformanceMetrics::PlotAudioPerformanceMetrics() NN_NOEXCEPT
    : m_pSpyController(nullptr)
    , m_pVoiceInfos(nullptr)
    , m_pSubMixInfos(nullptr)
    , m_pSinkInfos(nullptr)
    , m_LastTick(0)
    , m_VoiceCountMax(0)
    , m_SubMixCountMax(0)
    , m_SinkCountMax(0)
    , m_IsInitialized(false)
{
}

bool PlotAudioPerformanceMetrics::Initialize(
    const InitializeArg& arg,
    void* buffer,
    size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized() == false);
    NN_SDK_REQUIRES_NOT_NULL(arg.pAudioRendererParameter);

    NN_UNUSED(bufferSize);

    const auto& parameter = *arg.pAudioRendererParameter;

    m_RootNode.SetName("@nn/audio/PerformanceMetrics");

    m_TotalProcessingTime.SetName("TotalProcessingTime");
    m_TotalProcessingTime.SetRange(PlotMinValue, PlotMaxValue);
    m_TotalProcessingTime.SetParent(&m_RootNode);

    m_VoiceCount.SetName("VoiceCount");
    m_VoiceCount.SetRange(0, parameter.voiceCount);
    m_VoiceCount.SetParent(&m_RootNode);

    m_VoiceDropCount.SetName("VoiceDropCount");
    m_VoiceDropCount.SetColor(0xff, 0xff, 0x00);
    m_VoiceDropCount.SetRange(0, parameter.voiceCount);
    m_VoiceDropCount.SetParent(&m_RootNode);

    m_IsRenderingTimeLimitExceeded.SetName("IsRenderingTimeLimitExceeded");
    m_IsRenderingTimeLimitExceeded.SetColor(0xff, 0x7f, 0x00);
    m_IsRenderingTimeLimitExceeded.SetRange(0, 1);
    m_IsRenderingTimeLimitExceeded.SetParent(&m_RootNode);

    nn::util::BytePtr bytePtr(buffer);

    m_pVoiceInfos = nullptr;
    m_VoiceCountMax = parameter.voiceCount;
    if (m_VoiceCountMax > 0)
    {
        m_pVoiceInfos = bytePtr.Get<PlotPerformanceEntry>();
        bytePtr.Advance(sizeof(PlotPerformanceEntry) * m_VoiceCountMax);

        for (int i = 0; i < m_VoiceCountMax; ++i)
        {
            char name[nn::spy::PlotItem::MaxNameLength];
            nn::util::SNPrintf(name, sizeof(name), "@nn/audio/PerformanceMetrics/Voice/%d", i);
            PlotPerformanceEntry* pItem = new(&m_pVoiceInfos[i]) PlotPerformanceEntry();
            pItem->Initialize(name, 0xd0, 0x40, 0x40);
        }
    }

    m_pSubMixInfos = nullptr;
    m_SubMixCountMax = parameter.subMixCount;
    if (m_SubMixCountMax > 0)
    {
        m_pSubMixInfos = bytePtr.Get<PlotPerformanceEntry>();
        bytePtr.Advance(sizeof(PlotPerformanceEntry) * m_SubMixCountMax);

        for (int i = 0; i < m_SubMixCountMax; ++i)
        {
            char name[nn::spy::PlotItem::MaxNameLength];
            nn::util::SNPrintf(name, sizeof(name), "@nn/audio/PerformanceMetrics/SubMix/%d", i);
            PlotPerformanceEntry* pItem = new(&m_pSubMixInfos[i]) PlotPerformanceEntry();
            pItem->Initialize(name, 0xe0, 0xb0, 0x40);
        }
    }

    m_pSinkInfos = nullptr;
    m_SinkCountMax = parameter.sinkCount;
    if (m_SinkCountMax > 0)
    {
        m_pSinkInfos = bytePtr.Get<PlotPerformanceEntry>();
        bytePtr.Advance(sizeof(PlotPerformanceEntry) * m_SinkCountMax);

        for (int i = 0; i < m_SinkCountMax; ++i)
        {
            char name[nn::spy::PlotItem::MaxNameLength];
            nn::util::SNPrintf(name, sizeof(name), "@nn/audio/PerformanceMetrics/Sink/%d", i);
            PlotPerformanceEntry* pItem = new(&m_pSinkInfos[i]) PlotPerformanceEntry();
            pItem->Initialize(name, 0x40, 0x60, 0xd0);
        }
    }

    NN_SDK_ASSERT_LESS_EQUAL(static_cast<size_t>(-bytePtr.Distance(buffer)), bufferSize);

    m_VoiceSummaryInfo.Initialize(
        "@nn/audio/PerformanceMetrics/VoiceSummary",
        0xd0 /* r */,
        0x40 /* g */,
        0x40 /* b */,
        m_pVoiceInfos,
        m_VoiceCountMax);

    m_SubMixSummaryInfo.Initialize(
        "@nn/audio/PerformanceMetrics/SubMixSummary",
        0xe0 /* r */,
        0xb0 /* g */,
        0x40 /* b */,
        m_pSubMixInfos,
        m_SubMixCountMax);

    m_SinkSummaryInfo.Initialize(
        "@nn/audio/PerformanceMetrics/SinkSummary",
        0x40 /* r */,
        0x60 /* g */,
        0xd0 /* b */,
        m_pSinkInfos,
        m_SinkCountMax);

    m_FinalMixInfo.Initialize("@nn/audio/PerformanceMetrics/FinalMix", 0x60, 0xd0, 0x40);

    m_LastTick = nn::os::Tick(0);

    m_IsInitialized = true;
    return true;
}

void PlotAudioPerformanceMetrics::Finalize() NN_NOEXCEPT
{
    if (IsInitialized() == false)
    {
        return;
    }

    DetachFromModules();

    m_VoiceSummaryInfo.Finalize();
    m_SubMixSummaryInfo.Finalize();
    m_SinkSummaryInfo.Finalize();
    m_FinalMixInfo.Finalize();

    for (int i = 0; i < m_VoiceCountMax; ++i)
    {
        m_pVoiceInfos[i].Finalize();
    }

    for (int i = 0; i < m_SubMixCountMax; ++i)
    {
        m_pSubMixInfos[i].Finalize();
    }

    for (int i = 0; i < m_SinkCountMax; ++i)
    {
        m_pSinkInfos[i].Finalize();
    }

    m_IsInitialized = false;
}

void PlotAudioPerformanceMetrics::AttachToModules(nn::spy::SpyController& spyController) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(!IsAttachedToModules());

    nn::spy::PlotModule& plotModule = spyController.GetPlotModule();
    plotModule.AttachItem(m_RootNode);
    plotModule.AttachItem(m_TotalProcessingTime);
    plotModule.AttachItem(m_VoiceCount);
    plotModule.AttachItem(m_VoiceDropCount);
    plotModule.AttachItem(m_IsRenderingTimeLimitExceeded);

    m_VoiceSummaryInfo.AttachToModule(plotModule);
    m_SubMixSummaryInfo.AttachToModule(plotModule);
    m_SinkSummaryInfo.AttachToModule(plotModule);
    m_FinalMixInfo.AttachToModule(plotModule);

    for (int i = 0; i < m_VoiceCountMax; ++i)
    {
        m_pVoiceInfos[i].AttachToModule(plotModule);
    }

    for (int i = 0; i < m_SubMixCountMax; ++i)
    {
        m_pSubMixInfos[i].AttachToModule(plotModule);
    }

    for (int i = 0; i < m_SinkCountMax; ++i)
    {
        m_pSinkInfos[i].AttachToModule(plotModule);
    }

    m_pSpyController = &spyController;
}

void PlotAudioPerformanceMetrics::DetachFromModules() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    if (!IsAttachedToModules())
    {
        return;
    }

    nn::spy::PlotModule& plotModule = m_pSpyController->GetPlotModule();
    plotModule.DetachItem(m_TotalProcessingTime);
    plotModule.DetachItem(m_VoiceCount);
    plotModule.DetachItem(m_VoiceDropCount);
    plotModule.DetachItem(m_IsRenderingTimeLimitExceeded);
    plotModule.DetachItem(m_RootNode);

    m_pSpyController = nullptr;
}

bool PlotAudioPerformanceMetrics::PushPerformanceMetrics(
    const void* performanceFrameBuffer,
    size_t performanceFrameBufferSize,
    nn::os::Tick tick) NN_NOEXCEPT
{
    const int64_t DefaultFrameInterval = nn::os::ConvertToTick(nn::TimeSpan::FromMilliSeconds(5)).GetInt64Value();

    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(tick <= nn::os::GetSystemTick());
    NN_SDK_REQUIRES(m_LastTick <= tick);

    if (performanceFrameBuffer == nullptr)
    {
        return false;
    }

    if (!IsAttachedToModules())
    {
        return false;
    }

    nn::audio::PerformanceInfo performanceInfo;

    // 記録されている PerformanceFrame の数を数えます。
    int frameCount = 0;
    if (performanceInfo.SetBuffer(performanceFrameBuffer, performanceFrameBufferSize))
    {
        do
        {
            ++frameCount;
        }
        while (performanceInfo.MoveToNextFrame());
    }

    if (frameCount == 0)
    {
        return false;
    }

    // フレームあたりの経過時間を類推します。
    int64_t frameInterval = std::min(
        DefaultFrameInterval,
        (tick - m_LastTick).GetInt64Value() / frameCount);

    m_LastTick = tick;

    // 最初の PerformanceFrame の開始時刻を求めます。
    nn::os::Tick frameTick = tick - nn::os::Tick(frameInterval * frameCount);

    if (performanceInfo.SetBuffer(performanceFrameBuffer, performanceFrameBufferSize))
    {
        do
        {
            PrepareForFrame();

            int entryCount;
            const nn::audio::PerformanceEntry* pEntry = performanceInfo.GetEntries(&entryCount);

            // 継続して記録されているエントリーをプロットします。
            // FinalMix をプロットします。
            // サマリーの情報を収集します。
            for (int i = 0; i < entryCount; ++i)
            {
                const nn::audio::PerformanceEntry& entry = pEntry[i];

                switch (pEntry[i].entryType)
                {
                case nn::audio::PerformanceEntryType_Unknown:
                    break;

                case nn::audio::PerformanceEntryType_Voice:
                    m_VoiceSummaryInfo.PushSucceedingPerformanceEntry(entry, frameTick);
                    break;

                case nn::audio::PerformanceEntryType_SubMix:
                    m_SubMixSummaryInfo.PushSucceedingPerformanceEntry(entry, frameTick);
                    break;

                case nn::audio::PerformanceEntryType_FinalMix:
                    m_FinalMixInfo.PushValueAt(&entry, frameTick);
                    break;

                case nn::audio::PerformanceEntryType_Sink:
                    m_SinkSummaryInfo.PushSucceedingPerformanceEntry(entry, frameTick);
                    break;

                default:
                    break;
                }
            }

            // 新しいエントリーをプロットします。
            for (int i = 0; i < entryCount; ++i)
            {
                const nn::audio::PerformanceEntry& entry = pEntry[i];

                switch (pEntry[i].entryType)
                {
                case nn::audio::PerformanceEntryType_Unknown:
                    break;

                case nn::audio::PerformanceEntryType_Voice:
                    m_VoiceSummaryInfo.PushNewPerformanceEntry(entry, frameTick);
                    break;

                case nn::audio::PerformanceEntryType_SubMix:
                    m_SubMixSummaryInfo.PushNewPerformanceEntry(entry, frameTick);
                    break;

                case nn::audio::PerformanceEntryType_Sink:
                    m_SinkSummaryInfo.PushNewPerformanceEntry(entry, frameTick);
                    break;

                default:
                    break;
                }
            }

            PushValueForInactiveItem(frameTick);

            m_VoiceSummaryInfo.PushSummaryAt(frameTick);
            m_SubMixSummaryInfo.PushSummaryAt(frameTick);
            m_SinkSummaryInfo.PushSummaryAt(frameTick);

            m_TotalProcessingTime.PushValueAt(performanceInfo.GetTotalProcessingTime(), frameTick);
            m_VoiceCount.PushValueAt(m_VoiceSummaryInfo.GetActiveEntryCount(), frameTick);
            m_VoiceDropCount.PushValueAt(performanceInfo.GetVoiceDropCount(), frameTick);
            m_IsRenderingTimeLimitExceeded.PushValueAt(performanceInfo.IsRenderingTimeLimitExceeded() ? 1 : 0, frameTick);

            frameTick += nn::os::Tick(frameInterval);

        } while (performanceInfo.MoveToNextFrame());

        return true;
    }

    return false;
}

void PlotAudioPerformanceMetrics::PrepareForFrame() NN_NOEXCEPT
{
    m_VoiceSummaryInfo.PrepareForFrame();
    m_SubMixSummaryInfo.PrepareForFrame();
    m_SinkSummaryInfo.PrepareForFrame();
    m_FinalMixInfo.SetIsPushed(false);
}

void PlotAudioPerformanceMetrics::PushValueForInactiveItem(nn::os::Tick lastTick) NN_NOEXCEPT
{
    m_VoiceSummaryInfo.PushValueForInactiveItem(lastTick);

    m_SubMixSummaryInfo.PushValueForInactiveItem(lastTick);

    m_SinkSummaryInfo.PushValueForInactiveItem(lastTick);

    {
        auto& item = m_FinalMixInfo;
        if (item.IsActive() == true && item.IsPushed() == false)
        {
            item.PushValueAt(0, 0, lastTick);
            item.SetIsActive(false);
        }
    }
}

} // namespace detail
} // namespace audio
} // namespace spy
} // namespace nn

#endif
