﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <nn/nn_Macro.h>

#include <nn/spy/atk/spy_AtkSpyModule.h>

#include <nn/atk/detail/atk_AddonSoundArchiveContainer.h>

namespace nn { namespace spy { namespace atk {

NN_DEFINE_STATIC_CONSTANT(const int AtkSpyModule::InitializeArg::AudioPerformanceMetricsCountMin);

}}}

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

#include <nn/spy/detail/fnd/basis/spyfnd_Inlines.h>

namespace nn {
namespace spy {
namespace atk {

// 追加サウンドアーカイブの名前を収めるのに十分なスペースがあるか。
NN_STATIC_ASSERT(detail::AddonSoundArchiveDataPacket::MaxNameLength >= nn::atk::detail::AddonSoundArchiveContainer::SoundArchiveNameLengthMax);

namespace {
    void CalculateMemorySize(
        size_t* pTotalMemorySize,
        size_t memorySize,
        int alignment) NN_NOEXCEPT
    {
        *pTotalMemorySize = ::nn::spy::detail::fnd::RoundUp(*pTotalMemorySize, alignment) + memorySize;
    }

    void* CalculateMemoryAddress(
        void** pBuffer,
        size_t memorySize,
        int alignment,
        void* bufferEnd) NN_NOEXCEPT
    {
        NN_UNUSED(bufferEnd);
        void* addr = ::nn::spy::detail::fnd::RoundUp(*pBuffer, alignment);
        *pBuffer = ::nn::spy::detail::fnd::AddOffsetToPtr(addr, memorySize);
        NN_SDK_ASSERT(*pBuffer <= bufferEnd);
        return addr;
    }

    ::nn::spy::audio::AudioSpyModule::InitializeArg ToAudioSpyModuleInitializeArg(
        const AtkSpyModule::InitializeArg& args,
        const ::nn::audio::AudioRendererParameter& audioRendererParameter,
        nn::spy::audio::AudioSpyModule::ChannelType* pChannelType,
        int channelTypeCount) NN_NOEXCEPT
    {
        NN_UNUSED(channelTypeCount);
        ::nn::spy::audio::AudioSpyModule::InitializeArg initializeArg;
        initializeArg.pAudioRendererParameter = &audioRendererParameter;

        if (args.isWaveformInitializationEnabled)
        {
            if ( nn::atk::SoundSystem::GetRendererChannelCountMax() == 2 )
            {
                //  2ch のときの設定
                NN_SDK_ASSERT_GREATER_EQUAL( channelTypeCount, 2 );
                NN_STATIC_ASSERT( nn::atk::ChannelIndex_FrontLeft < 2 );
                NN_STATIC_ASSERT( nn::atk::ChannelIndex_FrontRight < 2 );

                pChannelType[nn::atk::ChannelIndex_FrontLeft]  = nn::spy::audio::AudioSpyModule::ChannelType_FrontLeft;
                pChannelType[nn::atk::ChannelIndex_FrontRight] = nn::spy::audio::AudioSpyModule::ChannelType_FrontRight;
                initializeArg.pWaveformChannelTypes = pChannelType;
                initializeArg.waveformChannelCount = 2;
            }
            else
            {
                NN_SDK_ASSERT_GREATER_EQUAL( channelTypeCount, nn::atk::ChannelIndex_Count );
                NN_STATIC_ASSERT( nn::atk::ChannelIndex_FrontLeft < nn::atk::ChannelIndex_Count );
                NN_STATIC_ASSERT( nn::atk::ChannelIndex_FrontRight < nn::atk::ChannelIndex_Count );
                NN_STATIC_ASSERT( nn::atk::ChannelIndex_RearLeft < nn::atk::ChannelIndex_Count );
                NN_STATIC_ASSERT( nn::atk::ChannelIndex_RearRight < nn::atk::ChannelIndex_Count );
                NN_STATIC_ASSERT( nn::atk::ChannelIndex_FrontCenter < nn::atk::ChannelIndex_Count );
                NN_STATIC_ASSERT( nn::atk::ChannelIndex_Lfe < nn::atk::ChannelIndex_Count );

                pChannelType[nn::atk::ChannelIndex_FrontLeft]   = nn::spy::audio::AudioSpyModule::ChannelType_FrontLeft;
                pChannelType[nn::atk::ChannelIndex_FrontRight]  = nn::spy::audio::AudioSpyModule::ChannelType_FrontRight;
                pChannelType[nn::atk::ChannelIndex_RearLeft]    = nn::spy::audio::AudioSpyModule::ChannelType_RearLeft;
                pChannelType[nn::atk::ChannelIndex_RearRight]   = nn::spy::audio::AudioSpyModule::ChannelType_RearRight;
                pChannelType[nn::atk::ChannelIndex_FrontCenter] = nn::spy::audio::AudioSpyModule::ChannelType_FrontCenter;
                pChannelType[nn::atk::ChannelIndex_Lfe]         = nn::spy::audio::AudioSpyModule::ChannelType_Lfe;
                initializeArg.pWaveformChannelTypes = pChannelType;
                initializeArg.waveformChannelCount = nn::atk::ChannelIndex_Count;
            }
            initializeArg.waveformSampleFormat = nn::audio::SampleFormat_PcmInt16;
        }

        return initializeArg;
    }

} // namespace {anonymous}

AtkSpyModule* AtkSpyModule::g_pInstance;

size_t AtkSpyModule::GetRequiredMemorySize(const InitializeArg &args) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(nn::atk::SoundSystem::IsInitialized());
    NN_SDK_REQUIRES(!args.isPerformanceMetricsInitializationEnabled || args.audioPerformanceMetricsCount >= InitializeArg::AudioPerformanceMetricsCountMin);
    NN_SDK_REQUIRES(!args.isAtkProfilesInitializationEnabled || args.atkProfilesCount >= InitializeArg::AtkProfilesCountMin);

    size_t totalMemorySize = 0;

    const auto audioRendererParameter = ::nn::atk::SoundSystem::GetAudioRendererParameter();
    nn::spy::audio::AudioSpyModule::ChannelType channelType[nn::atk::ChannelIndex_Count];
    const auto audioSpyModuleInitializeArg = ToAudioSpyModuleInitializeArg(args, audioRendererParameter, channelType, sizeof(channelType) / sizeof(channelType[0]));

    CalculateMemorySize(
        &totalMemorySize,
        ::nn::spy::audio::AudioSpyModule::GetRequiredMemorySize(audioSpyModuleInitializeArg),
        ::nn::DefaultAlignment);

    if (args.isPerformanceMetricsInitializationEnabled)
    {
        CalculateMemorySize(
            &totalMemorySize,
            ::nn::atk::AudioRendererPerformanceReader::GetRequiredMemorySize(args.audioPerformanceMetricsCount),
            ::nn::DefaultAlignment);
    }

    if (args.isAtkProfilesInitializationEnabled)
    {
        CalculateMemorySize(&totalMemorySize, sizeof(::nn::atk::ProfileReader), ::nn::DefaultAlignment);
        CalculateMemorySize(&totalMemorySize, sizeof(::nn::atk::TaskProfileReader), ::nn::DefaultAlignment);

        CalculateMemorySize(
            &totalMemorySize,
            ::nn::atk::TaskProfileReader::GetRequiredMemorySize(args.atkProfilesCount),
            ::nn::DefaultAlignment);
    }

    if (args.streamSoundInfoCountMax > 0)
    {
        CalculateMemorySize(
            &totalMemorySize,
            sizeof(StreamSoundPlotNode) * args.streamSoundInfoCountMax,
            ::nn::DefaultAlignment);
    }

    if (args.addonSoundArchiveCountMax > 0)
    {
        CalculateMemorySize(
            &totalMemorySize,
            sizeof(AddonSoundArchiveInfo) * args.addonSoundArchiveCountMax,
            ::nn::DefaultAlignment);
    }

    return totalMemorySize;
}

AtkSpyModule::AtkSpyModule() NN_NOEXCEPT
    : Base("NnSpyAtk", detail::PacketTraits::Version)
    , m_pAtkProfileReader(nullptr)
    , m_pAtkTaskProfileReader(nullptr)
    , m_pAtkTaskProfileReaderBuffer(nullptr)
    , m_IsDataRequested(false)
    , m_IsInitialized(false)
    , m_IsLogEnabled(false)
    , m_IsSoundStateEnabled(false)
    , m_IsPerformanceMetricsInitialized(false)
    , m_IsPerformanceMetricsEnabled(false)
    , m_IsWaveformInitialized(false)
    , m_IsWaveformEnabled(false)
    , m_IsAtkProfilesInitialized(false)
    , m_IsAtkProfilesEnabled(false)
    , m_IsSoundArchivePlayerInfoEnabled(false)
    , m_IsSoundSystemInfoEnabled(false)
    , m_IsSoundHeapInfoEnabled(false)
    , m_IsStreamSoundInfoEnabled(false)
    , m_AddonSoundArchiveInfoArray(nullptr)
    , m_AddonSoundArchiveInfoCountMax(0)
    , m_AtkSoundThreadState("@nn/atk/atk SoundThread")
    , m_AtkTaskThreadState("@nn/atk/atk TaskThread")
    , m_DspState("@nn/atk/DSP")
    , m_AtkSoundArchivePlayerNode("@nn/atk/SoundArchivePlayer")
    , m_AtkSoundSystemNode("@nn/atk/SoundSystem")
    , m_AtkSoundHeapNode("@nn/atk/SoundHeap")
    , m_AtkWaveSoundActiveCount("WaveSoundActiveCount")
    , m_AtkStreamSoundActiveCount("StreamSoundActiveCount")
    , m_AtkSequenceSoundActiveCount("SequenceSoundActiveCount")
    , m_AtkAllocatedDriverCommandBufferSize("DriverCommandAllocatedBufferSize")
    , m_AtkDriverCommandBufferSize("DriverCommandBufferSize")
    , m_AtkAllocatedTaskCommandBufferSize("TaskCommandAllocatedBufferSize")
    , m_AtkTaskCommandBufferSize("TaskCommandBufferSize")
    , m_AtkAllocatedSoundHeapSize("AllocatedSoundHeapSize")
    , m_AtkSoundHeapSize("SoundHeapSize")
    , m_StreamSoundPlots(nullptr)
    , m_StreamSoundPlotCount(0)
    , m_PrevStreamCount(0)
{
    m_AtkWaveSoundActiveCount.SetParent(&m_AtkSoundArchivePlayerNode);
    m_AtkStreamSoundActiveCount.SetParent(&m_AtkSoundArchivePlayerNode);
    m_AtkSequenceSoundActiveCount.SetParent(&m_AtkSoundArchivePlayerNode);

    m_AtkAllocatedSoundHeapSize.SetParent(&m_AtkSoundHeapNode);
    m_AtkSoundHeapSize.SetParent(&m_AtkSoundHeapNode);

    m_AtkAllocatedDriverCommandBufferSize.SetParent(&m_AtkSoundSystemNode);
    m_AtkDriverCommandBufferSize.SetParent(&m_AtkSoundSystemNode);
    m_AtkAllocatedTaskCommandBufferSize.SetParent(&m_AtkSoundSystemNode);
    m_AtkTaskCommandBufferSize.SetParent(&m_AtkSoundSystemNode);
}

AtkSpyModule::~AtkSpyModule() NN_NOEXCEPT
{
}

bool AtkSpyModule::Initialize(
    const InitializeArg& args,
    void* buffer,
    size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    NN_SDK_REQUIRES(!IsRegistered());
    NN_SDK_REQUIRES(g_pInstance == nullptr);
    NN_SDK_REQUIRES(bufferSize == 0 || buffer != nullptr);
    NN_SDK_REQUIRES_ALIGNED(buffer, ::nn::DefaultAlignment);
    NN_SDK_REQUIRES(bufferSize >= GetRequiredMemorySize(args));
    NN_SDK_REQUIRES(nn::atk::SoundSystem::IsInitialized());
    NN_SDK_REQUIRES(!args.isPerformanceMetricsInitializationEnabled || args.audioPerformanceMetricsCount >= InitializeArg::AudioPerformanceMetricsCountMin);
    NN_SDK_REQUIRES(!args.isPerformanceMetricsEnabled || args.isPerformanceMetricsInitializationEnabled);
    NN_SDK_REQUIRES(!args.isWaveformEnabled || args.isWaveformInitializationEnabled);
    NN_SDK_REQUIRES(!args.isAtkProfilesEnabled || args.isAtkProfilesInitializationEnabled);

    g_pInstance = this;

    m_IsInitialized = true;
    m_PrevStreamCount = 0;

    void* bufferCurrent = buffer;
    void* bufferEnd = ::nn::spy::detail::fnd::AddOffsetToPtr(buffer, bufferSize);

    if(args.isLogEnabled)
    {
        InitializeLog();
    }

    SetSoundStateEnabled(args.isSoundStateEnabled);
    SetSoundArchivePlayerInfoEnabled(args.isSoundArchivePlayerInfoEnabled);
    SetSoundSystemInfoEnabled(args.isSoundSystemInfoEnabled);
    SetSoundHeapInfoEnabled(args.isSoundHeapInfoEnabled);
    SetStreamSoundInfoEnabled(args.isStreamSoundInfoEnabled);

    InitializeAudioSpyModule(args, &bufferCurrent, bufferEnd);
    InitializeAtkProfiles(args, &bufferCurrent, bufferEnd);
    InitializeAddonSoundArchive(args, &bufferCurrent, bufferEnd);
    InitializeStreamSoundInfo(args, &bufferCurrent, bufferEnd);

    return true;
}

void AtkSpyModule::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(!IsRegistered());

    FinalizeLog();

    SetSoundStateEnabled(false);

    FinalizeAudioSpyModule();
    FinalizeAtkProfiles();
    FinalizeAddonSoundArchive();
    FinalizeStreamSoundInfo();

    m_IsInitialized = false;

    g_pInstance = nullptr;
}

bool AtkSpyModule::IsInitialized() const NN_NOEXCEPT
{
    return m_IsInitialized;
}

bool AtkSpyModule::IsLogEnabled() const NN_NOEXCEPT
{
    return m_IsLogEnabled;
}

bool AtkSpyModule::IsSoundStateEnabled() const NN_NOEXCEPT
{
    return m_IsSoundStateEnabled;
}

bool AtkSpyModule::IsPerformanceMetricsInitialized() const NN_NOEXCEPT
{
    return m_IsPerformanceMetricsInitialized;
}

bool AtkSpyModule::IsPerformanceMetricsEnabled() const NN_NOEXCEPT
{
    return m_IsPerformanceMetricsEnabled;
}

bool AtkSpyModule::IsWaveformInitialized() const NN_NOEXCEPT
{
    return m_IsWaveformInitialized;
}

bool AtkSpyModule::IsWaveformEnabled() const NN_NOEXCEPT
{
    return m_IsWaveformEnabled;
}

bool AtkSpyModule::IsAtkProfilesInitialized() const NN_NOEXCEPT
{
    return m_IsAtkProfilesInitialized;
}

bool AtkSpyModule::IsAtkProfilesEnabled() const NN_NOEXCEPT
{
    return m_IsAtkProfilesEnabled;
}

bool AtkSpyModule::IsSoundArchivePlayerInfoEnabled() const NN_NOEXCEPT
{
    return m_IsSoundArchivePlayerInfoEnabled;
}

bool AtkSpyModule::IsSoundSystemInfoEnabled() const NN_NOEXCEPT
{
    return m_IsSoundSystemInfoEnabled;
}

bool AtkSpyModule::IsSoundHeapInfoEnabled() const NN_NOEXCEPT
{
    return m_IsSoundHeapInfoEnabled;
}

bool AtkSpyModule::IsStreamSoundInfoEnabled() const NN_NOEXCEPT
{
    return m_IsStreamSoundInfoEnabled;
}

void AtkSpyModule::SetStreamSoundInfoEnabled(bool value) NN_NOEXCEPT
{
    m_IsStreamSoundInfoEnabled = value;
}

void AtkSpyModule::SetLogEnabled(bool value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    if (value == m_IsLogEnabled)
    {
        return;
    }

    if (value)
    {
        InitializeLog();
    }
    else
    {
        FinalizeLog();
    }
}

void AtkSpyModule::SetSoundStateEnabled(bool value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    m_IsSoundStateEnabled = value;
}

void AtkSpyModule::SetPerformanceMetricsEnabled(bool value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(IsPerformanceMetricsInitialized());

    if (value == m_IsPerformanceMetricsEnabled)
    {
        return;
    }

    if (value)
    {
        m_IsPerformanceMetricsEnabled = true;

        ::nn::atk::SoundSystem::RegisterAudioRendererPerformanceReader(m_AudioRendererPerformanceReader);
    }
    else
    {
        ::nn::atk::SoundSystem::UnregisterAudioRendererPerformanceReader();

        m_IsPerformanceMetricsEnabled = false;
    }
}

void AtkSpyModule::SetWaveformEnabled(bool value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(IsWaveformInitialized());

    m_IsWaveformEnabled = value;
}

void AtkSpyModule::SetAtkProfilesEnabled(bool value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(IsAtkProfilesInitialized());

    if (m_IsAtkProfilesEnabled == value)
    {
        return;
    }

    if (value)
    {
        ::nn::atk::SoundSystem::RegisterProfileReader(*m_pAtkProfileReader);
        ::nn::atk::SoundSystem::RegisterTaskProfileReader(*m_pAtkTaskProfileReader);
    }
    else
    {
        ::nn::atk::SoundSystem::UnregisterProfileReader(*m_pAtkProfileReader);
        ::nn::atk::SoundSystem::UnregisterTaskProfileReader(*m_pAtkTaskProfileReader);
    }

    m_IsAtkProfilesEnabled = value;
}

void AtkSpyModule::SetSoundArchivePlayerInfoEnabled(bool value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    m_IsSoundArchivePlayerInfoEnabled = value;
}

void AtkSpyModule::SetSoundSystemInfoEnabled(bool value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    m_IsSoundSystemInfoEnabled = value;
}

void AtkSpyModule::SetSoundHeapInfoEnabled(bool value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    m_IsSoundHeapInfoEnabled = value;
}

void AtkSpyModule::OnRegistered() NN_NOEXCEPT
{
    Base::OnRegistered();

    NN_SDK_ASSERT(IsInitialized());

    if (IsInitialized())
    {
        GetController()->RegisterModule(m_AudioSpyModule);

        RegisterPlotItems();
    }
}

void AtkSpyModule::OnUnregistered() NN_NOEXCEPT
{
    if (IsInitialized())
    {
        UnregisterPlotItems();
    }

    Base::OnUnregistered();

    if (IsInitialized())
    {
        GetController()->UnregisterModule(m_AudioSpyModule);
    }
}

void AtkSpyModule::OnRequested(bool bRequested) NN_NOEXCEPT
{
    Base::OnRequested(bRequested);

    if (bRequested)
    {
        m_IsDataRequested = true;
    }
}

void AtkSpyModule::OnSessionStarted() NN_NOEXCEPT
{
    Base::OnSessionStarted();

    for(int i = 0; i < m_StreamSoundPlotCount; i++)
    {
        StreamSoundPlotNode& streamSoundPlotNode = m_StreamSoundPlots[i];

        nn::util::SNPrintf(streamSoundPlotNode.stateString, StreamStateNameLengthMax, "");
    }
    m_PrevStreamCount = 0;
}


AtkSpyModule& AtkSpyModule::GetInstance() NN_NOEXCEPT
{
    return *g_pInstance;
}

bool AtkSpyModule::IsDataRequested() NN_NOEXCEPT
{
    // 呼び出しのたびにデータ要求はリセットされます。
    return m_IsDataRequested.exchange(false);
}

void AtkSpyModule::SpyLog(const char* message) NN_NOEXCEPT
{
    SpyController* pController = GetInstance().GetController();
    if (pController == nullptr)
    {
        return;
    }

    ::nn::spy::LogModule::Write(*pController, message);
}

void AtkSpyModule::WriteSoundStateToPacket(
    detail::SoundStatePacket* packet,
    const ::nn::atk::detail::BasicSound* sound) NN_NOEXCEPT
{
    if (packet->itemCount >= detail::SoundStatePacket::ItemCountMax)
    {
        return;
    }

    detail::SoundStatePacket::Item* item = &packet->items[packet->itemCount];
    item->instanceId = sound->GetInstanceId();
    item->soundId = sound->GetId();
    uint8_t statusBitFlag = 0;
    if (sound->IsPause())
    {
        statusBitFlag |= detail::SoundStatePacket::StatusPause;
    }
    if (sound->IsStarted())
    {
        statusBitFlag |= detail::SoundStatePacket::StatusStart;
    }
    if (sound->IsPrepared())
    {
        statusBitFlag |= detail::SoundStatePacket::StatusPrepared;
    }
    item->statusBitFlag = statusBitFlag;
    item->padding1[0] = 0;
    item->padding1[1] = 0;
    item->padding1[2] = 0;
    item->archiveInstanceId.Set(sound->GetSoundArchive());

    nn::atk::SoundParamCalculationValues values;
    sound->CalculateSoundParamCalculationValues(&values);
    item->volume = values.resultParam.volume;
    item->pitch = values.resultParam.pitch;
    item->lpf = values.resultParam.lpf;
    item->bqfType = values.resultParam.bqfType;
    item->bqfValue = values.resultParam.bqfValue;
    item->playerPriority = values.resultParam.playerPriority;
    item->pan = values.resultParam.outputParamResult[nn::atk::OutputDevice::OutputDevice_Main].pan;
    item->surroundPan = values.resultParam.outputParamResult[nn::atk::OutputDevice::OutputDevice_Main].span;

    packet->itemCount++;
}

bool AtkSpyModule::PushSoundState(::nn::atk::SoundArchivePlayer& soundArchivePlayer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(soundArchivePlayer.IsAvailable());

    if (!IsRequested())
    {
        return false;
    }

    if (!IsSoundStateEnabled())
    {
        return false;
    }

    // 新規のデータ送信要請
    if (IsDataRequested())
    {
        // メインサウンドアーカイブのデータを送信します。
        this->PushSoundArchiveData(soundArchivePlayer.GetSoundArchive());

        // この時点で送信済みの追加サウンドアーカイブを未送信に戻します。
        if (IsAddonSoundArchiveEnabled())
        {
            ClearPushedAddonSoundArchiveInfos();
        }
    }

    // 未送信の追加サウンドアーカイブのデータを送信します。
    if (IsAddonSoundArchiveEnabled())
    {
        PushAddonSoundArchiveData(soundArchivePlayer);
    }

    detail::SoundStatePacket packet;
    packet.itemSize = sizeof(detail::SoundStatePacket::Item);
    packet.itemCount = 0;

    auto func = [this, &packet](const ::nn::atk::SoundHandle& item) mutable
    {
        WriteSoundStateToPacket(&packet, item.detail_GetAttachedSound());
    };

    uint32_t soundPlayerCount = soundArchivePlayer.GetSoundPlayerCount();
    for (uint32_t i = 0; i < soundPlayerCount; ++i)
    {
        auto playerId = ::nn::atk::SoundArchive::GetPlayerIdFromIndex(i);
        auto& soundPlayer = soundArchivePlayer.GetSoundPlayer(playerId);
        soundPlayer.ForEachSoundReference(func);
    }

    {
        // items は最後のメンバーでなければなりません。
        NN_STATIC_ASSERT(offsetof(detail::SoundStatePacket, items) + sizeof(packet.items) == sizeof(packet));
        size_t packetSize = sizeof(packet)
            - sizeof(packet.items)
            + sizeof(packet.items[0]) * packet.itemCount;

        bool result = PushData(&packet, packetSize);
        if (!result)
        {
            return false;
        }
    }

    return true;
}

AtkSpyModule::AddonSoundArchiveInfo& AtkSpyModule::AddonSoundArchiveInfoDic::RemoveSingle() NN_NOEXCEPT
{
    // 辞書から要素を１つ切り出します。
    NN_SDK_ASSERT(!empty());
    auto top = begin();
    AddonSoundArchiveInfo& value = *top;
    erase(top);
    return value;
}

void AtkSpyModule::AddonSoundArchiveInfoList::Push(AtkSpyModule::AddonSoundArchiveInfo& value) NN_NOEXCEPT
{
    // リストに要素を追加します。
    push_front(value);
}

AtkSpyModule::AddonSoundArchiveInfo& AtkSpyModule::AddonSoundArchiveInfoList::Pop() NN_NOEXCEPT
{
    // リストから要素を１つ切り出します。
    NN_SDK_ASSERT(!empty());
    AddonSoundArchiveInfo& value = *begin();
    pop_front();
    return value;
}

void AtkSpyModule::ClearPushedAddonSoundArchiveInfos() NN_NOEXCEPT
{
    // 登録済みの追加サウンドアーカイブをすべて未登録に戻します。
    m_AddonSoundArchiveLastAddTick = ::nn::os::Tick();
    while (!m_PushedAddonSoundArchiveInfoDic.empty())
    {
        AddonSoundArchiveInfo& info = m_PushedAddonSoundArchiveInfoDic.RemoveSingle();
        m_FreeAddonSoundArchiveInfoList.Push(info);
    }
}

void AtkSpyModule::InitializeStreamSoundInfo(const InitializeArg & args, void ** pBuffer, void * bufferEnd) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_StreamSoundPlots == nullptr);
    NN_SDK_ASSERT(m_StreamSoundPlotCount == 0);

    if (args.streamSoundInfoCountMax == 0)
    {
        return;
    }

    m_StreamSoundPlotCount = args.streamSoundInfoCountMax;

    m_StreamSoundPlots =
        static_cast<StreamSoundPlotNode*>(
            CalculateMemoryAddress(
                pBuffer,
                sizeof(StreamSoundPlotNode) * args.streamSoundInfoCountMax,
                ::nn::DefaultAlignment,
                bufferEnd));

    for (int i = 0; i < m_StreamSoundPlotCount; i++)
    {
        StreamSoundPlotNode* streamSoundPlotNode = new(&m_StreamSoundPlots[i]) StreamSoundPlotNode();

        char name[nn::spy::PlotItem::MaxNameLength];
        nn::util::SNPrintf(name, sizeof(name), "@nn/atk/atk StreamSound/%d", i);
        streamSoundPlotNode->streamSoundNode.SetName(name);
        nn::util::SNPrintf(name, sizeof(name), "FillingPercentage");
        streamSoundPlotNode->streamSoundRateFloat.SetName(name);
        nn::util::SNPrintf(name, sizeof(name), "State");
        streamSoundPlotNode->streamSoundState.SetName(name);
        nn::util::SNPrintf(name, sizeof(name), "RemainingCachePercentage");
        streamSoundPlotNode->remainingCachePercentageFloat.SetName(name);
        streamSoundPlotNode->streamSoundRateFloat.SetParent(&streamSoundPlotNode->streamSoundNode);
        streamSoundPlotNode->streamSoundRateFloat.SetRange(0.0f, 100.0f);
        streamSoundPlotNode->streamSoundState.SetParent(&streamSoundPlotNode->streamSoundNode);
        streamSoundPlotNode->remainingCachePercentageFloat.SetParent(&streamSoundPlotNode->streamSoundNode);
        streamSoundPlotNode->remainingCachePercentageFloat.SetRange(0.0f, 100.0f);
    }
}

void AtkSpyModule::FinalizeStreamSoundInfo() NN_NOEXCEPT
{
    m_StreamSoundPlots = nullptr;
    m_StreamSoundPlotCount = 0;
}

bool AtkSpyModule::PushAddonSoundArchiveData(const ::nn::atk::SoundArchivePlayer& soundArchivePlayer) NN_NOEXCEPT
{
    ::nn::os::Tick lastAddTick = soundArchivePlayer.GetAddonSoundArchiveLastAddTick();
    if (lastAddTick == m_AddonSoundArchiveLastAddTick)
    {
        return true;
    }

    m_AddonSoundArchiveLastAddTick = lastAddTick;

    int count = soundArchivePlayer.GetAddonSoundArchiveCount();

    // SoundArchivePlayer から削除された追加サウンドアーカイブの AddonSoundArchiveInfo を回収します。
    {
        for (auto& item : m_PushedAddonSoundArchiveInfoDic)
        {
            item.m_IsPushed = false;
        }

        for (int i = 0; i < count; i++)
        {
            const char* name = soundArchivePlayer.GetAddonSoundArchiveName(i);

            auto found = m_PushedAddonSoundArchiveInfoDic.find(name);
            if (found != m_PushedAddonSoundArchiveInfoDic.end())
            {
                AddonSoundArchiveInfo& info = *found;

                const ::nn::atk::AddonSoundArchive* pArchive = soundArchivePlayer.GetAddonSoundArchive(i);
                ::nn::os::Tick tick = soundArchivePlayer.GetAddonSoundArchiveAddTick(i);

                if (pArchive == info.m_pAddonSoundArchive && tick == info.m_AddTick)
                {
                    info.m_IsPushed = true;
                }
            }
        }

        auto iterator = m_PushedAddonSoundArchiveInfoDic.begin();
        while (iterator != m_PushedAddonSoundArchiveInfoDic.end())
        {
            AddonSoundArchiveInfo& info = *iterator;

            if (!info.m_IsPushed)
            {
                iterator = m_PushedAddonSoundArchiveInfoDic.erase(iterator);
                m_FreeAddonSoundArchiveInfoList.Push(info);
                continue;
            }

            ++iterator;
        }
    }

    bool result = true;
    for (int i = 0; i < count; i++)
    {
        const ::nn::atk::AddonSoundArchive* pArchive = soundArchivePlayer.GetAddonSoundArchive(i);
        if (!pArchive->IsAvailable())
        {
            continue;
        }

        const char* name = soundArchivePlayer.GetAddonSoundArchiveName(i);
        auto found = m_PushedAddonSoundArchiveInfoDic.find(name);
        if (found != m_PushedAddonSoundArchiveInfoDic.end())
        {
            // データはプッシュ済み。
            continue;
        }

        if (m_FreeAddonSoundArchiveInfoList.empty())
        {
            break;
        }

        ::nn::os::Tick tick = soundArchivePlayer.GetAddonSoundArchiveAddTick(i);

        AddonSoundArchiveInfo& info = m_FreeAddonSoundArchiveInfoList.Pop();

        info.m_IsPushed = true;
        info.m_AddTick = tick;
        info.m_pAddonSoundArchive = pArchive;

        detail::AddonSoundArchiveDataPacket& packet = info.m_addonSoundArchiveDataPacket;
        packet.instanceId.Set(pArchive);
        packet.name.Set(name);

        m_PushedAddonSoundArchiveInfoDic.insert(packet.name.text, info);

        // name は最後のメンバーでなければなりません。
        NN_STATIC_ASSERT(offsetof(detail::AddonSoundArchiveDataPacket, name) + sizeof(packet.name) == sizeof(packet));
        size_t packetSize = sizeof(packet) - sizeof(packet.name) + packet.name.GetMinimumDataSize();

        result = PushData(&packet, packetSize);
        if (!result)
        {
            break;
        }

        result = PushSoundData(*info.m_pAddonSoundArchive);
        if (!result)
        {
            break;
        }
    }

    return result;
}

bool AtkSpyModule::PushPerformanceMetrics() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(IsPerformanceMetricsInitialized());

    if (!IsPerformanceMetricsEnabled())
    {
        return false;
    }

    if (!IsRegistered())
    {
        return false;
    }

    for(;;)
    {
        const auto* info = m_AudioRendererPerformanceReader.ReadPerformanceInfo();
        if (info != nullptr)
        {
            m_AudioSpyModule.PushPerformanceMetrics(
                info->performanceBuffer,
                info->performanceBufferSize,
                info->tick);

            continue;
        }

        break;
    }

    return true;
}

bool AtkSpyModule::PushSoundArchiveData(const ::nn::atk::SoundArchive& soundArchive) NN_NOEXCEPT
{
    if (!soundArchive.IsAvailable())
    {
        return false;
    }

    return PushSoundData(soundArchive) &&
        PushPlayerData(soundArchive);
}

// NOTE: 異なるスレッドから呼ばれる
bool AtkSpyModule::PushSoundData(const ::nn::atk::SoundArchive& soundArchive) NN_NOEXCEPT
{
    typedef detail::SoundDataPacket::Item ItemType;

    detail::SoundDataPacket packet;
    packet.archiveInstanceId.Set(&soundArchive);
    packet.itemCount = 0;
    size_t itemAreaSize = 0;

    uint32_t soundCount = soundArchive.GetSoundCount();
    for (uint32_t index = 0; index < soundCount; ++index)
    {
        if (itemAreaSize + sizeof(ItemType) > packet.ItemAreaSize)
        {
            bool result = PushSoundDataPacket(packet, itemAreaSize);
            if (!result)
            {
                return false;
            }

            packet.itemCount = 0;
            itemAreaSize = 0;
        }

        ::nn::atk::SoundArchive::ItemId soundId = soundArchive.GetSoundIdFromIndex(index);
        ::nn::atk::SoundArchive::SoundInfo soundInfo;
        if (!soundArchive.ReadSoundInfo(&soundInfo, soundId))
        {
            continue;
        }

        auto* pItem = reinterpret_cast<ItemType*>(&packet.itemArea[itemAreaSize / sizeof(uint32_t)]);
        pItem->soundId = soundId;
        pItem->playerId = soundInfo.playerId;
        switch (soundArchive.GetSoundType(soundId))
        {
        case nn::atk::SoundArchive::SoundType_Sequence:
            pItem->soundType = detail::SoundDataPacket::SoundType_Sequence;
            break;
        case nn::atk::SoundArchive::SoundType_Stream:
            pItem->soundType = detail::SoundDataPacket::SoundType_Stream;
            break;
        case nn::atk::SoundArchive::SoundType_Wave:
            pItem->soundType = detail::SoundDataPacket::SoundType_Wave;
            break;
        default:
            break;
        }
        pItem->volume = soundInfo.volume;
        pItem->playerPriority = soundInfo.playerPriority;
        pItem->padding1[0] = 0;

        const char* soundLabel = soundArchive.GetItemLabel(soundId);
        pItem->label.Set(soundLabel);

        // label は最後のメンバーでなければなりません。
        NN_STATIC_ASSERT(offsetof(ItemType, label) + sizeof(pItem->label) == sizeof(*pItem));
        size_t itemSize = sizeof(*pItem)
            - sizeof(pItem->label)
            + pItem->label.GetMinimumDataSize();

        itemAreaSize += ::nn::spy::detail::fnd::RoundUp(itemSize, sizeof(uint32_t));
        packet.itemCount += 1;
    }

    if (packet.itemCount > 0)
    {
        bool result = PushSoundDataPacket(packet, itemAreaSize);
        if (!result)
        {
            return false;
        }
    }

    return true;
}

bool AtkSpyModule::PushPlayerData(const nn::atk::SoundArchive& soundArchive) NN_NOEXCEPT
{
    typedef detail::PlayerDataPacket::Item ItemType;

    detail::PlayerDataPacket packet;
    packet.itemCount = 0;
    size_t itemAreaSize = 0;

    uint32_t playerCount = soundArchive.GetPlayerCount();
    for (uint32_t index = 0; index < playerCount; ++index)
    {
        if (itemAreaSize + sizeof(ItemType) > packet.ItemAreaSize)
        {
            bool result = PushPlayerDataPacket(packet, itemAreaSize);
            if (!result)
            {
                return false;
            }

            packet.itemCount = 0;
            itemAreaSize = 0;
        }

        nn::atk::SoundArchive::ItemId playerId = soundArchive.GetPlayerIdFromIndex(index);
        nn::atk::SoundArchive::PlayerInfo playerInfo;
        if (!soundArchive.ReadPlayerInfo(&playerInfo, playerId))
        {
            continue;
        }

        auto* pItem = reinterpret_cast<ItemType*>(&packet.itemArea[itemAreaSize / sizeof(uint32_t)]);
        pItem->playerId = playerId;

        const char* label = soundArchive.GetItemLabel(playerId);
        pItem->label.Set(label);

        // label は最後のメンバーでなければなりません。
        NN_STATIC_ASSERT(offsetof(ItemType, label) + sizeof(pItem->label) == sizeof(*pItem));
        size_t itemSize = sizeof(*pItem)
            - sizeof(pItem->label)
            + pItem->label.GetMinimumDataSize();

        itemAreaSize += ::nn::spy::detail::fnd::RoundUp(itemSize, sizeof(uint32_t));
        packet.itemCount += 1;
    }

    if (packet.itemCount > 0)
    {
        bool result = PushPlayerDataPacket(packet, itemAreaSize);
        if (!result)
        {
            return false;
        }
    }

    return true;
}

size_t AtkSpyModule::WriteSequenceVariableToPacket(
    detail::SequenceVariablePacket::Item* item,
    const ::nn::atk::detail::SequenceSound* sequenceSound) NN_NOEXCEPT
{
    item->soundInstanceId = sequenceSound->GetInstanceId();
    item->soundId = sequenceSound->GetId();
    item->archiveInstanceId.Set(sequenceSound->GetSoundArchive());
    item->availableVariable = 1;
    item->trackCount = ::nn::atk::SequenceSoundHandle::TrackIndexMax + 1;
    item->localVariableCount = ::nn::atk::SequenceSoundHandle::VariableIndexMax + 1;
    item->trackVariableCount = ::nn::atk::SequenceSoundHandle::VariableIndexMax + 1;

    int writeIndex = 0;
    int16_t variable = 0;

    if (sequenceSound->ReadVariable(0, &variable) == true)
    {
        for (int i = 0; i <= ::nn::atk::SequenceSoundHandle::VariableIndexMax; ++i)
        {
            sequenceSound->ReadVariable(i, &variable);
            item->variables[writeIndex + 1 + i] = variable;
        }

        {
            int count = GetValidVariableLastIndex(item->variables + writeIndex + 1, ::nn::atk::SequenceSoundHandle::VariableIndexMax + 1) + 1;
            item->variables[writeIndex++] = static_cast<int16_t>(count);
            writeIndex += count;
        }

        for (int t = 0; t <= ::nn::atk::SequenceSoundHandle::TrackIndexMax; ++t)
        {
            for (int i = 0; i <= ::nn::atk::SequenceSoundHandle::VariableIndexMax; ++i)
            {
                sequenceSound->ReadTrackVariable(t, i, &variable);
                item->variables[writeIndex + 1 + i] = variable;
            }

            int count = GetValidVariableLastIndex(item->variables + writeIndex + 1, ::nn::atk::SequenceSoundHandle::VariableIndexMax + 1) + 1;
            item->variables[writeIndex++] = static_cast<int16_t>(count);
            writeIndex += count;
        }
    }
    else
    {
        item->availableVariable = 0;
    }

    return offsetof(detail::SequenceVariablePacket::Item, variables) + sizeof(int16_t) * writeIndex;
}

int AtkSpyModule::GetValidVariableLastIndex(int16_t* variables, int variableCount)
{
    int index = variableCount - 1;
    while (index >= 0)
    {
        if (variables[index] != -1)
        {
            break;
        }
        index--;
    }
    return index;
}

bool AtkSpyModule::PushSequenceVariable(::nn::atk::SoundArchivePlayer& soundArchivePlayer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(soundArchivePlayer.IsAvailable());

    if (!IsRequested())
    {
        return false;
    }

    {
        detail::GlobalSequenceVariablePacket packet;
        packet.variableCount = static_cast<uint8_t>(nn::atk::SequenceSoundHandle::VariableIndexMax + 1);
        for (int i = 0; i <= nn::atk::SequenceSoundHandle::VariableIndexMax; i++)
        {
            int16_t variable = 0;
            nn::atk::SequenceSoundHandle::ReadGlobalVariable(&variable, i);
            packet.variables[i] = variable;
        }

        int count = GetValidVariableLastIndex(packet.variables, nn::atk::SequenceSoundHandle::VariableIndexMax + 1) + 1;
        packet.actualVariableCount = static_cast<uint8_t>(count);

        bool result = PushData(&packet, sizeof(detail::GlobalSequenceVariablePacket) - sizeof(packet.variables) + sizeof(int16_t) * count);
        if (!result)
        {
            return false;
        }
    }

    detail::SequenceVariablePacket packet;
    packet.itemCount = 0;
    size_t itemAreaSize = 0;
    bool pushFailure = false;

    auto func = [this, &packet, &itemAreaSize, &pushFailure](::nn::atk::SoundHandle& item)
    {
        ::nn::atk::SequenceSoundHandle sequenceSoundHandle(&item);
        if (sequenceSoundHandle.IsAttachedSound() == true)
        {
            if (itemAreaSize + sizeof(detail::SequenceVariablePacket::Item) > packet.ItemAreaSize)
            {
                bool result = PushSequenceVariablePacket(packet, itemAreaSize);
                if (!result)
                {
                    pushFailure = true;
                    return;
                }

                packet.itemCount = 0;
                itemAreaSize = 0;
            }

            auto* pItem = reinterpret_cast<detail::SequenceVariablePacket::Item*>(&packet.itemArea[itemAreaSize / sizeof(uint32_t)]);
            size_t itemSize = WriteSequenceVariableToPacket(pItem, sequenceSoundHandle.detail_GetAttachedSound());
            itemAreaSize += ::nn::spy::detail::fnd::RoundUp(itemSize, sizeof(uint32_t));
            packet.itemCount++;
        }
    };

    uint32_t soundPlayerCount = soundArchivePlayer.GetSoundPlayerCount();
    for (uint32_t i = 0; i < soundPlayerCount; ++i)
    {
        auto playerId = ::nn::atk::SoundArchive::GetPlayerIdFromIndex(i);
        auto& soundPlayer = soundArchivePlayer.GetSoundPlayer(playerId);
        soundPlayer.ForEachSoundReference(func);

        if (pushFailure)
        {
            return false;
        }
    }

    if (packet.itemCount > 0)
    {
        bool result = PushSequenceVariablePacket(packet, itemAreaSize);
        if (!result)
        {
            return false;
        }
    }

    return true;
}

bool AtkSpyModule::PushSequenceVariablePacket(
    const detail::SequenceVariablePacket& packet,
    size_t itemAreaSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(itemAreaSize % sizeof(packet.itemArea[0]) == 0);
    NN_SDK_ASSERT(itemAreaSize <= packet.ItemAreaSize);

    // itemArea は最後のメンバーでなければなりません。
    NN_STATIC_ASSERT(offsetof(detail::SequenceVariablePacket, itemArea) + sizeof(packet.itemArea) == sizeof(packet));
    size_t packetSize = sizeof(packet) - sizeof(packet.itemArea) + itemAreaSize;

    return PushData(&packet, packetSize);
}

bool AtkSpyModule::PushStreamSoundData(const::nn::atk::StreamSoundHandle &soundHandle, const char* const label, int streamSoundCount) NN_NOEXCEPT
{
    if(streamSoundCount >= m_StreamSoundPlotCount)
    {
        return false;
    }
    char state[StreamStateNameLengthMax];
    char partialLabel[LabelNameLengthMax + 1]; // 終端文字の分も加えておく

    nn::util::SNPrintf(partialLabel, sizeof(partialLabel), "%s", label);
    if (soundHandle.IsLoadingDelayState())
    {
        nn::util::SNPrintf(state, StreamStateNameLengthMax, "%s Delayed", partialLabel);
    }
    else
    {
        nn::util::SNPrintf(state, StreamStateNameLengthMax, "%s InTime", partialLabel);
    }

    StreamSoundPlotNode& streamSoundPlotNode = m_StreamSoundPlots[streamSoundCount];

    streamSoundPlotNode.streamSoundRateFloat.PushValue(soundHandle.GetFilledBufferPercentage());

    if( strncmp(state, streamSoundPlotNode.stateString, StreamStateNameLengthMax) != 0)
    {
        if (soundHandle.IsLoadingDelayState())
        {
            streamSoundPlotNode.streamSoundState.PushValue(state, 255, 0, 0);
        }
        else
        {
            streamSoundPlotNode.streamSoundState.PushValue(state, 0, 255, 0);
        }

        strncpy(streamSoundPlotNode.stateString, state, StreamStateNameLengthMax);
    }
    return false;
}

bool AtkSpyModule::PushSoundDataPacket(
    const detail::SoundDataPacket& packet,
    size_t itemAreaSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(itemAreaSize % sizeof(packet.itemArea[0]) == 0);
    NN_SDK_ASSERT(itemAreaSize <= packet.ItemAreaSize);

    // itemArea は最後のメンバーでなければなりません。
    NN_STATIC_ASSERT(offsetof(detail::SoundDataPacket, itemArea) + sizeof(packet.itemArea) == sizeof(packet));
    size_t packetSize = sizeof(packet) - sizeof(packet.itemArea) + itemAreaSize;

    return PushData(&packet, packetSize);
}

bool AtkSpyModule::PushPlayerDataPacket(
    const detail::PlayerDataPacket& packet,
    size_t itemAreaSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(itemAreaSize % sizeof(packet.itemArea[0]) == 0);
    NN_SDK_ASSERT(itemAreaSize <= packet.ItemAreaSize);

    // itemArea は最後のメンバーでなければなりません。
    NN_STATIC_ASSERT(offsetof(detail::PlayerDataPacket, itemArea) + sizeof(packet.itemArea) == sizeof(packet));
    size_t packetSize = sizeof(packet) - sizeof(packet.itemArea) + itemAreaSize;

    return PushData(&packet, packetSize);
}

void AtkSpyModule::PushAtkSoundThreadAndDspProfiles() NN_NOEXCEPT
{
    ::nn::atk::SoundProfile soundProfile;
    while (m_pAtkProfileReader->Read(&soundProfile, 1) > 0)
    {
        // atk の仕様でサウンドスレッドのパフォーマンス情報は全く同じ情報が連続で取得されることがあるため、
        // 最後に出力した nwFrameProcess.end より新しい情報のみ送信します。
        if (m_LastNwFrameProcessEnd < soundProfile.nwFrameProcess.begin)
        {
            m_AtkSoundThreadState.PushValueAt("NwFrame", 255, 112, 67, soundProfile.nwFrameProcess.begin);
            m_AtkSoundThreadState.PushValueAt("", 0, 0, 0, soundProfile.nwFrameProcess.end);
            m_LastNwFrameProcessEnd = soundProfile.nwFrameProcess.end;
        }

        ::nn::os::Tick dspBegin = std::max(soundProfile.nwFrameProcess.end, m_LastDspFrameProcessEnd);

        if (soundProfile.voiceProcess.begin < soundProfile.voiceProcess.end)
        {
            m_LastDspFrameProcessEnd = dspBegin + soundProfile.voiceProcess.end;
            m_DspState.PushValueAt("Voice", 208, 64, 64, dspBegin + soundProfile.voiceProcess.begin);
            m_DspState.PushValueAt("", 0, 0, 0, m_LastDspFrameProcessEnd);
        }
        if (soundProfile.mainMixProcess.begin < soundProfile.mainMixProcess.end)
        {
            m_LastDspFrameProcessEnd = dspBegin + soundProfile.mainMixProcess.end;
            m_DspState.PushValueAt("MainMix", 224, 176, 64, dspBegin + soundProfile.mainMixProcess.begin);
            m_DspState.PushValueAt("", 0, 0, 0, m_LastDspFrameProcessEnd);
        }
        if (soundProfile.finalMixProcess.begin < soundProfile.finalMixProcess.end)
        {
            m_LastDspFrameProcessEnd = dspBegin + soundProfile.finalMixProcess.end;
            m_DspState.PushValueAt("FinalMix", 96, 208, 64, dspBegin + soundProfile.finalMixProcess.begin);
            m_DspState.PushValueAt("", 0, 0, 0, m_LastDspFrameProcessEnd);
        }
        if (soundProfile.sinkProcess.begin < soundProfile.sinkProcess.end)
        {
            m_LastDspFrameProcessEnd = dspBegin + soundProfile.sinkProcess.end;
            m_DspState.PushValueAt("Sink", 64, 96, 208, dspBegin + soundProfile.sinkProcess.begin);
            m_DspState.PushValueAt("", 0, 0, 0, m_LastDspFrameProcessEnd);
        }
    }
}

void AtkSpyModule::PushRemainingCachePercentage(nn::atk::SoundArchivePlayer* soundArchivePlayer, nn::atk::detail::driver::StreamSoundPlayer* streamSoundPlayer, float remainingCachePercentage) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(soundArchivePlayer);

    int streamSoundCount = 0;
    auto func = [this, &streamSoundCount, streamSoundPlayer, &remainingCachePercentage](::nn::atk::SoundHandle& item) mutable
    {
        ::nn::atk::StreamSoundHandle streamSoundHandle(&item);
        if (streamSoundHandle.IsAttachedSound())
        {
            if (streamSoundHandle.HasStreamSoundPlayer(streamSoundPlayer))
            {
                m_StreamSoundPlots[streamSoundCount].remainingCachePercentageFloat.PushValue(remainingCachePercentage);
            }
            streamSoundCount++;
        }
    };

    uint32_t soundPlayerCount = soundArchivePlayer->GetSoundPlayerCount();
    for (uint32_t i = 0; i < soundPlayerCount; ++i)
    {
        auto playerId = ::nn::atk::SoundArchive::GetPlayerIdFromIndex(i);
        auto& soundPlayer = soundArchivePlayer->GetSoundPlayer(playerId);
        soundPlayer.ForEachSoundReference(func);
    }
}

void AtkSpyModule::PushAtkTaskThreadProfiles(::nn::atk::SoundArchivePlayer* soundArchivePlayer) NN_NOEXCEPT
{
    ::nn::atk::TaskProfile taskProfile;
    while (m_pAtkTaskProfileReader->Read(&taskProfile, 1) > 0)
    {
        switch (taskProfile.type)
        {
        case nn::atk::TaskProfile::TaskProfileType_LoadStreamBlock:
        {
            const auto& data = taskProfile.loadStreamBlock;
            m_AtkTaskThreadState.PushValueAt("bfstm load", 255, 128, 128, data.GetBeginTick());
            m_AtkTaskThreadState.PushValueAt("", 0, 0, 0, data.GetEndTick());
            if (soundArchivePlayer != nullptr)
            {
                PushRemainingCachePercentage(soundArchivePlayer, data.GetStreamSoundPlayer(), data.GetRemainingCachePercentage());
            }

            break;
        }
        case nn::atk::TaskProfile::TaskProfileType_LoadOpusStreamBlock:
        {
            const auto& data = taskProfile.loadOpusStreamBlock;

            const int stringBufferLength = 32;
            char stringBuffer[stringBufferLength];
            nn::util::SNPrintf(stringBuffer, stringBufferLength, "opus load (size:%d byte)", data.GetFsReadSize());
            m_AtkTaskThreadState.PushValueAt(stringBuffer, 0x4c, 0xaf, 0x50, data.GetBeginTick());

            m_AtkTaskThreadState.PushValueAt("opus decode", 0x4c, 0x50, 0xaf, data.GetBeginTick() + nn::os::Tick(data.GetFsAccessTime()));
            m_AtkTaskThreadState.PushValueAt("", 0, 0, 0, data.GetEndTick());
            if (soundArchivePlayer != nullptr)
            {
                PushRemainingCachePercentage(soundArchivePlayer, data.GetStreamSoundPlayer(), data.GetRemainingCachePercentage());
            }

            break;
        }
        default:
        {
            NN_UNEXPECTED_DEFAULT;
            break;
        }
        }
    }
}

bool AtkSpyModule::PushWaveform(
    const void* buffer,
    size_t bufferSize,
    nn::os::Tick tick) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(IsWaveformInitialized());

    if (!IsRegistered())
    {
        return false;
    }

    if (!IsWaveformEnabled())
    {
        return false;
    }

    if (bufferSize == 0)
    {
        return true;
    }

    m_AudioSpyModule.PushWaveform(buffer, bufferSize, tick);

    return true;
}

bool AtkSpyModule::PushAtkProfiles() NN_NOEXCEPT
{
    return PushAtkProfiles(nullptr);
}

bool AtkSpyModule::PushAtkProfiles(::nn::atk::SoundArchivePlayer& soundArchivePlayer) NN_NOEXCEPT
{
    return PushAtkProfiles(&soundArchivePlayer);
}

bool AtkSpyModule::PushAtkProfiles(::nn::atk::SoundArchivePlayer* soundArchivePlayer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(IsAtkProfilesInitialized());

    if (!IsRegistered())
    {
        return false;
    }

    if (!IsAtkProfilesEnabled())
    {
        return false;
    }

    PushAtkSoundThreadAndDspProfiles();
    PushAtkTaskThreadProfiles(soundArchivePlayer);

    return true;
}

bool AtkSpyModule::PushSoundArchivePlayerInfo(::nn::atk::SoundArchivePlayer& soundArchivePlayer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(soundArchivePlayer.IsAvailable());

    if (!IsRegistered())
    {
        return false;
    }

    if (!IsSoundArchivePlayerInfoEnabled())
    {
        return false;
    }

    m_AtkWaveSoundActiveCount.PushValue(static_cast<double>(soundArchivePlayer.GetActiveWaveSoundCount()));
    m_AtkStreamSoundActiveCount.PushValue(static_cast<double>(soundArchivePlayer.GetActiveStreamSoundCount()));
    m_AtkSequenceSoundActiveCount.PushValue(static_cast<double>(soundArchivePlayer.GetActiveSequenceSoundCount()));

    return true;
}

bool AtkSpyModule::PushSoundSystemInfo() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(nn::atk::SoundSystem::IsInitialized());

    if (!IsRegistered())
    {
        return false;
    }

    if (!IsSoundSystemInfoEnabled())
    {
        return false;
    }

    m_AtkAllocatedDriverCommandBufferSize.PushValue(static_cast<double>(nn::atk::SoundSystem::GetAllocatedDriverCommandBufferSize()));
    m_AtkDriverCommandBufferSize.PushValue(static_cast<double>(nn::atk::SoundSystem::GetDriverCommandBufferSize()));
    m_AtkAllocatedTaskCommandBufferSize.PushValue(static_cast<double>(nn::atk::detail::DriverCommand::GetInstanceForTaskThread().GetAllocatedCommandBufferSize()));
    m_AtkTaskCommandBufferSize.PushValue(static_cast<double>(nn::atk::detail::DriverCommand::GetInstanceForTaskThread().GetCommandBufferSize()));

    return true;
}

bool AtkSpyModule::PushSoundHeapInfo(::nn::atk::SoundHeap& soundHeap) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    if (!IsRegistered())
    {
        return false;
    }

    if (!IsSoundHeapInfoEnabled())
    {
        return false;
    }

    m_AtkAllocatedSoundHeapSize.PushValue(static_cast<double>(soundHeap.GetSize() - soundHeap.GetFreeSize()));
    m_AtkSoundHeapSize.PushValue(static_cast<double>(soundHeap.GetSize()));

    return true;
}

bool AtkSpyModule::PushStreamSoundInfo(::nn::atk::SoundArchivePlayer& soundArchivePlayer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(soundArchivePlayer.IsAvailable());

    if (!IsRegistered())
    {
        return false;
    }

    SpyController* pController = GetController();
    NN_SDK_ASSERT_NOT_NULL(pController);
    if (!pController->GetPlotModule().IsRequested())
    {
        return false;
    }

    if (!IsStreamSoundInfoEnabled())
    {
        return false;
    }

    int streamSoundCount = 0;

    // TODO: どうやってインスタンスの利用順番の整合性を保つか考える
    auto func = [this, &streamSoundCount](::nn::atk::SoundHandle& item) mutable
    {
        ::nn::atk::StreamSoundHandle streamSoundHandle(&item);
        if (streamSoundHandle.IsAttachedSound())
        {
            ::nn::atk::detail::StreamSound* sound = streamSoundHandle.detail_GetAttachedSound();
            const ::nn::atk::SoundArchive* soundArchive = sound->GetSoundArchive();

            char labelName[LabelNameLengthMax + 1]; // 終端文字の分も加えておく
            const char* itemLabel = soundArchive->GetItemLabel(item.GetId());
            if(itemLabel != nullptr)
            {
                nn::util::SNPrintf(labelName, sizeof(labelName), "%s", itemLabel);
            }
            else
            {
                nn::util::SNPrintf(labelName, sizeof(labelName), "%d", static_cast<int>(item.GetId()));
            }

            PushStreamSoundData(streamSoundHandle, labelName, streamSoundCount);
            streamSoundCount++;
        }
    };

    uint32_t soundPlayerCount = soundArchivePlayer.GetSoundPlayerCount();
    for (uint32_t i = 0; i < soundPlayerCount; ++i)
    {
        auto playerId = ::nn::atk::SoundArchive::GetPlayerIdFromIndex(i);
        auto& soundPlayer = soundArchivePlayer.GetSoundPlayer(playerId);
        soundPlayer.ForEachSoundReference(func);
    }

    // 前回から要素数が減少している時、それをプロットに反映させるために -1.0f を送る
    const float InvalidFillingPercentage = -1.0f;
    for (int i = streamSoundCount; i < m_PrevStreamCount; i++)
    {
        StreamSoundPlotNode& streamSoundPlotNode = m_StreamSoundPlots[i];

        streamSoundPlotNode.streamSoundRateFloat.PushValue(InvalidFillingPercentage);
        streamSoundPlotNode.streamSoundState.PushValue("", 0, 0, 0);
        nn::util::SNPrintf(streamSoundPlotNode.stateString, StreamStateNameLengthMax, "");
    }
    m_PrevStreamCount = streamSoundCount;

    return true;
}

void AtkSpyModule::InitializeLog() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsLogEnabled);

    m_IsLogEnabled = true;

    nn::atk::detail::Debug_HookLogFunc(SpyLog);
}

void AtkSpyModule::FinalizeLog() NN_NOEXCEPT
{
    if (!m_IsLogEnabled)
    {
        return;
    }

    nn::atk::detail::Debug_HookLogFunc(nullptr);

    m_IsLogEnabled = false;
}

void AtkSpyModule::InitializeAudioSpyModule(
    const InitializeArg& args,
    void** pBuffer,
    void* bufferEnd) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsPerformanceMetricsInitialized);
    NN_SDK_ASSERT(!m_IsWaveformInitialized);

    const auto audioRendererParameter = ::nn::atk::SoundSystem::GetAudioRendererParameter();
    nn::spy::audio::AudioSpyModule::ChannelType channelType[nn::atk::ChannelIndex_Count];
    const auto audioSpyModuleInitializeArg = ToAudioSpyModuleInitializeArg(args, audioRendererParameter, channelType, sizeof(channelType) / sizeof(channelType[0]));

    NN_SDK_ASSERT(!args.isWaveformInitializationEnabled || nn::atk::SoundSystem::GetRendererChannelCountMax() == audioSpyModuleInitializeArg.waveformChannelCount);
    NN_SDK_ASSERT(!args.isWaveformInitializationEnabled || static_cast<int>(nn::atk::SoundSystem::GetRendererSampleCount()) == audioRendererParameter.sampleCount);

    size_t bufferSizeForAudioSpyModule =
        ::nn::spy::audio::AudioSpyModule::GetRequiredMemorySize(audioSpyModuleInitializeArg);

    void* bufferForAudioSpyModule =
        CalculateMemoryAddress(
            pBuffer,
            bufferSizeForAudioSpyModule,
            ::nn::DefaultAlignment,
            bufferEnd);

    m_AudioSpyModule.Initialize(
        audioSpyModuleInitializeArg,
        bufferForAudioSpyModule,
        bufferSizeForAudioSpyModule);

    if (args.isPerformanceMetricsInitializationEnabled)
    {
        m_IsPerformanceMetricsInitialized = true;

        size_t bufferSizeForAudioRendererPerformanceReader =
            ::nn::atk::AudioRendererPerformanceReader::GetRequiredMemorySize(args.audioPerformanceMetricsCount);

        void* bufferForAudioRendererPerformanceReader =
            CalculateMemoryAddress(
                pBuffer,
                bufferSizeForAudioRendererPerformanceReader,
                ::nn::DefaultAlignment,
                bufferEnd);

        m_AudioRendererPerformanceReader.Initialize(
            args.audioPerformanceMetricsCount,
            bufferForAudioRendererPerformanceReader,
            bufferSizeForAudioRendererPerformanceReader);

        if (args.isPerformanceMetricsEnabled)
        {
            SetPerformanceMetricsEnabled(true);
        }
    }

    if (args.isWaveformInitializationEnabled)
    {
        m_IsWaveformInitialized = true;

        if (args.isWaveformEnabled)
        {
            SetWaveformEnabled(true);
        }
    }
}

void AtkSpyModule::FinalizeAudioSpyModule() NN_NOEXCEPT
{
    if (m_IsPerformanceMetricsInitialized)
    {
        SetPerformanceMetricsEnabled(false);
        m_IsPerformanceMetricsInitialized = false;
    }

    if (m_IsWaveformInitialized)
    {
        SetWaveformEnabled(false);
        m_IsWaveformInitialized = false;
    }

    m_AudioSpyModule.Finalize();
}

void AtkSpyModule::InitializeAtkProfiles(
    const InitializeArg& args,
    void** pBuffer,
    void* bufferEnd) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsAtkProfilesInitialized);

    if (!args.isAtkProfilesInitializationEnabled)
    {
        return;
    }

    m_IsAtkProfilesInitialized = true;

    void* instanceBufferForAtkReader =
        CalculateMemoryAddress(
            pBuffer,
            sizeof(::nn::atk::ProfileReader),
            ::nn::DefaultAlignment,
            bufferEnd);
    m_pAtkProfileReader = new(instanceBufferForAtkReader) ::nn::atk::ProfileReader();

    void* instanceBufferForAtkTaskReader =
        CalculateMemoryAddress(
            pBuffer,
            sizeof(::nn::atk::TaskProfileReader),
            ::nn::DefaultAlignment,
            bufferEnd);
    m_pAtkTaskProfileReader = new(instanceBufferForAtkTaskReader) ::nn::atk::TaskProfileReader();

    size_t atkTaskProfileReaderBufferSize = nn::atk::TaskProfileReader::GetRequiredMemorySize(args.atkProfilesCount);
    m_pAtkTaskProfileReaderBuffer =
        CalculateMemoryAddress(
            pBuffer,
            atkTaskProfileReaderBufferSize,
            ::nn::DefaultAlignment,
            bufferEnd);
    m_pAtkTaskProfileReader->Initialize(
        m_pAtkTaskProfileReaderBuffer,
        atkTaskProfileReaderBufferSize,
        args.atkProfilesCount);

    if (args.isAtkProfilesEnabled)
    {
        SetAtkProfilesEnabled(true);
    }
}

void AtkSpyModule::FinalizeAtkProfiles() NN_NOEXCEPT
{
    if (m_IsAtkProfilesInitialized)
    {
        SetAtkProfilesEnabled(false);

        m_pAtkProfileReader->~ProfileReader();
        m_pAtkProfileReader = nullptr;

        m_pAtkTaskProfileReader->Finalize();
        m_pAtkTaskProfileReader->~AtkProfileReader();
        m_pAtkTaskProfileReader = nullptr;

        m_pAtkTaskProfileReaderBuffer = nullptr;

        m_IsAtkProfilesInitialized = false;
    }
}

void AtkSpyModule::InitializeAddonSoundArchive(
    const InitializeArg& args,
    void** pBuffer,
    void* bufferEnd) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_AddonSoundArchiveInfoArray == nullptr);
    NN_SDK_ASSERT(m_AddonSoundArchiveInfoCountMax == 0);

    if (args.addonSoundArchiveCountMax == 0)
    {
        return;
    }

    m_AddonSoundArchiveInfoCountMax = args.addonSoundArchiveCountMax;
    m_AddonSoundArchiveLastAddTick = ::nn::os::Tick();

    m_AddonSoundArchiveInfoArray =
        static_cast<AddonSoundArchiveInfo*>(
            CalculateMemoryAddress(
                pBuffer,
                sizeof(AddonSoundArchiveInfo) * args.addonSoundArchiveCountMax,
                ::nn::DefaultAlignment,
                bufferEnd));

    for (int i = 0; i < m_AddonSoundArchiveInfoCountMax; i++)
    {
        AddonSoundArchiveInfo* pInfo = new(&m_AddonSoundArchiveInfoArray[i]) AddonSoundArchiveInfo();
        m_FreeAddonSoundArchiveInfoList.Push(*pInfo);
    }
}

void AtkSpyModule::FinalizeAddonSoundArchive() NN_NOEXCEPT
{
    m_AddonSoundArchiveInfoArray = nullptr;
    m_AddonSoundArchiveInfoCountMax = 0;
    // 要素毎に erase しないように、強制的に破棄します。
    new(&m_PushedAddonSoundArchiveInfoDic) AddonSoundArchiveInfoDic();
    new(&m_FreeAddonSoundArchiveInfoList) AddonSoundArchiveInfoList();
}

void AtkSpyModule::RegisterPlotItems() NN_NOEXCEPT
{
    GetController()->GetPlotModule().AttachItem(m_AtkSoundThreadState);
    GetController()->GetPlotModule().AttachItem(m_AtkTaskThreadState);
    GetController()->GetPlotModule().AttachItem(m_DspState);
    GetController()->GetPlotModule().AttachItem(m_AtkWaveSoundActiveCount);
    GetController()->GetPlotModule().AttachItem(m_AtkStreamSoundActiveCount);
    GetController()->GetPlotModule().AttachItem(m_AtkSequenceSoundActiveCount);
    GetController()->GetPlotModule().AttachItem(m_AtkAllocatedSoundHeapSize);
    GetController()->GetPlotModule().AttachItem(m_AtkSoundHeapSize);
    GetController()->GetPlotModule().AttachItem(m_AtkAllocatedDriverCommandBufferSize);
    GetController()->GetPlotModule().AttachItem(m_AtkDriverCommandBufferSize);
    GetController()->GetPlotModule().AttachItem(m_AtkAllocatedTaskCommandBufferSize);
    GetController()->GetPlotModule().AttachItem(m_AtkTaskCommandBufferSize);
    GetController()->GetPlotModule().AttachItem(m_AtkSoundArchivePlayerNode);
    GetController()->GetPlotModule().AttachItem(m_AtkSoundHeapNode);
    GetController()->GetPlotModule().AttachItem(m_AtkSoundSystemNode);

    for (int i = 0; i < m_StreamSoundPlotCount; i++)
    {
        StreamSoundPlotNode& streamSoundPlotNode = m_StreamSoundPlots[i];
        GetController()->GetPlotModule().AttachItem(streamSoundPlotNode.streamSoundRateFloat);
        GetController()->GetPlotModule().AttachItem(streamSoundPlotNode.streamSoundState);
        GetController()->GetPlotModule().AttachItem(streamSoundPlotNode.streamSoundNode);
        GetController()->GetPlotModule().AttachItem(streamSoundPlotNode.remainingCachePercentageFloat);
    }
}

void AtkSpyModule::UnregisterPlotItems() NN_NOEXCEPT
{
    GetController()->GetPlotModule().DetachItem(m_AtkSoundThreadState);
    GetController()->GetPlotModule().DetachItem(m_AtkTaskThreadState);
    GetController()->GetPlotModule().DetachItem(m_DspState);
    GetController()->GetPlotModule().DetachItem(m_AtkWaveSoundActiveCount);
    GetController()->GetPlotModule().DetachItem(m_AtkStreamSoundActiveCount);
    GetController()->GetPlotModule().DetachItem(m_AtkSequenceSoundActiveCount);
    GetController()->GetPlotModule().DetachItem(m_AtkAllocatedSoundHeapSize);
    GetController()->GetPlotModule().DetachItem(m_AtkSoundHeapSize);
    GetController()->GetPlotModule().DetachItem(m_AtkAllocatedDriverCommandBufferSize);
    GetController()->GetPlotModule().DetachItem(m_AtkDriverCommandBufferSize);
    GetController()->GetPlotModule().DetachItem(m_AtkAllocatedTaskCommandBufferSize);
    GetController()->GetPlotModule().DetachItem(m_AtkTaskCommandBufferSize);
    GetController()->GetPlotModule().DetachItem(m_AtkSoundArchivePlayerNode);
    GetController()->GetPlotModule().DetachItem(m_AtkSoundHeapNode);
    GetController()->GetPlotModule().DetachItem(m_AtkSoundSystemNode);

    for (int i = 0; i < m_StreamSoundPlotCount; i++)
    {
        StreamSoundPlotNode& streamSoundPlotNode = m_StreamSoundPlots[i];
        GetController()->GetPlotModule().DetachItem(streamSoundPlotNode.streamSoundRateFloat);
        GetController()->GetPlotModule().DetachItem(streamSoundPlotNode.streamSoundState);
        GetController()->GetPlotModule().DetachItem(streamSoundPlotNode.streamSoundNode);
        GetController()->GetPlotModule().DetachItem(streamSoundPlotNode.remainingCachePercentageFloat);
    }
}

} // namespace nn::spy::atk
} // namespace nn::spy
} // namespace nn

#endif // NN_BUILD_CONFIG_SPY_ENABLED
