﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/util/util_FormatString.h>
#include <nn/ige/ige_Panel.h>

namespace nn { namespace ige {

typedef util::BitPack32::Field<0, 1, bool> OrientationField;
typedef util::BitPack32::Field<1, 1, bool> PanelTypeField; // 今後伸びるかも

Panel::Panel(detail::IgeAllocator* pAllocator, PanelOption option) NN_NOEXCEPT
    : Control(&m_Node, option)
    , ControlFactory()
    , m_pBackground(option.background)
    , m_Pack()
    , m_Width(static_cast<int16_t>(option.width))
    , m_Height(static_cast<int16_t>(option.height))
{
    NodeOption nodeOption;
    nodeOption.pUserData = this;
    nodeOption.pMetadataFunction = GenerateMetadata;
    nodeOption.enableAutoNotify = true;

    SetPanelType(option.panelType);
    SetOrientation(option.orientation);

    m_Node.Initialize(nodeOption);
    InitializeFactory(pAllocator, &m_Node);
}

Panel::~Panel() NN_NOEXCEPT
{
    DeleteAll();
}

void Panel::GenerateMetadata(BufferWriter* pWriter, const Node& node) NN_NOEXCEPT
{
    auto& self = *node.GetUserData<Panel>();
    WriteAttribute(pWriter, "type", "Panel");
    self.WriteCommonMetadata(pWriter);

    WriteAttribute(pWriter, "orientation", self.m_Pack.Get<OrientationField>() ? "true" : "false");
    WriteAttribute(pWriter, "panelType", self.m_Pack.Get<PanelTypeField>() ? "true" : "false");
    WriteAttribute(pWriter, "background", self.m_pBackground);

    {
        char buffer[16] = {};
        ::nn::util::SNPrintf(buffer, NN_ARRAY_SIZE(buffer), "%d", self.m_Width);
        WriteAttribute(pWriter, "width", buffer);
    }

    {
        char buffer[16] = {};
        ::nn::util::SNPrintf(buffer, NN_ARRAY_SIZE(buffer), "%d", self.m_Height);
        WriteAttribute(pWriter, "height", buffer);
    }
}

Orientation Panel::GetOrientation() const NN_NOEXCEPT
{
    return m_Pack.Get<OrientationField>() ? Orientation_Vertical : Orientation_Horizontal;
}

void Panel::SetOrientation(Orientation orientation) NN_NOEXCEPT
{
    if (orientation == Orientation_Horizontal)
    {
        m_Pack.Set<OrientationField>(false);
    }
    else if (orientation == Orientation_Vertical)
    {
        m_Pack.Set<OrientationField>(true);
    }
    else
    {
        NN_SDK_ASSERT(false, "invalid orientation");
    }
}

PanelType Panel::GetPanelType() const NN_NOEXCEPT
{
    return m_Pack.Get<PanelTypeField>() ? PanelType_Wrap : PanelType_Stack;
}

void Panel::SetPanelType(PanelType panelType) NN_NOEXCEPT
{
    if (panelType == PanelType_Stack)
    {
        m_Pack.Set<PanelTypeField>(false);
    }
    else if (panelType == PanelType_Wrap)
    {
        m_Pack.Set<PanelTypeField>(true);
    }
    else
    {
        NN_SDK_ASSERT(false, "invalid panelType");
    }
}

}} // namespace nn::ige::detail
