﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ige/ige_EditService.h>
#include <nn/ige/ige_Node.h>
#include "detail/ige_Allocator.h"

#include "detail/messages/ige_NodeMessage.h"
#include "detail/messages/ige_MetadataMessage.h"
#include "detail/messages/ige_MetadataRequestMessage.h"
#include "detail/messages/ige_ChildNodeRequestMessage.h"

namespace nn { namespace ige {

Node::Node() NN_NOEXCEPT
    : m_pNext(NULL)
    , m_pParent(NULL)
    , m_pUserData(NULL)
    , m_pMetadataFunc(NULL)
{
}

Node::~Node() NN_NOEXCEPT
{
    FinalizeNode();
}

void Node::Initialize(NodeOption option) NN_NOEXCEPT
{
    Finalize();
    SetEnableAutoNotify(option.enableAutoNotify);
    m_pMetadataFunc = option.pMetadataFunction;
    m_pUserData = option.pUserData;
}

void Node::FinalizeNode() NN_NOEXCEPT
{
    if (m_pParent)
    {
        m_pParent->Unregister(this);
    }

    m_pNext = NULL;
    m_pParent = NULL;
    m_pUserData = NULL;
    m_pMetadataFunc = NULL;
}

void Node::PushInfo() NN_NOEXCEPT
{
    BufferWriter writer(GetService()->GetAllocator().GetResource());
    WriteMetadata(&writer);

    detail::NodeMessage nodeInfo(
        GetId(),
        GetParentId(),
        GetNodeType(),
        util::string_view(),
        writer.GetBuffer());

    GetService()->SendMessage(nodeInfo);

    if (IsEnabledAutoNotify())
    {
        PushChildInfo();
    }
}

void Node::PushMetadata() NN_NOEXCEPT
{
    BufferWriter writer(GetService()->GetAllocator().GetResource());
    WriteMetadata(&writer);

    if (writer.GetSize() > 0)
    {
        GetService()->SendMessage(detail::MetadataMessage(GetId(), writer.GetBuffer()));
    }
}

void Node::WriteMetadata(BufferWriter* pWriter) NN_NOEXCEPT
{
    if (m_pMetadataFunc)
    {
        m_pMetadataFunc(pWriter, *this);
    }
}

NodeIdType Node::GetParentId() const NN_NOEXCEPT
{
    return m_pParent ? m_pParent->m_Id : detail::s_InvalidId;
}

void Node::OnMessageReceived(const detail::NodeTargetMessage& message) NN_NOEXCEPT
{
    auto type = message.GetType();
    if (detail::MetadataRequestMessage::IsTarget(type))
    {
        PushMetadata();
    }
    else if (detail::ChildNodeRequestMessage::IsTarget(type))
    {
        PushChildInfo();
    }
}

}} // namespace nn::ige
