﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ige/ige_HtcsHelper.h>

#ifdef NN_BUILD_CONFIG_SPEC_CAFE

#include <cafe/hio.h>

namespace nn { namespace ige {
namespace
{
    const int HtcsOK = 0;
    const int HtcsError = -1;

    // igeではアロケータを取得できるタイミングがないので、固定長バッファで代用しています。
    // サイズは経験則による、データが壊れずに読める上限値です。
    const int HioTemporaryBufferLength = 512;
    const int HioChannelMaxCount = 64;

    const htcs::HtcsPeerName PeerNameAny = { "" };
    const htcs::HtcsPeerName DefaultHostName = { "" };

    struct HioChannel
    {
        char        hioBuffer[HioTemporaryBufferLength];
        OSEvent     event;
        char        portName[nn::htcs::PortNameBufferLength];
        HIOHandle   handle;
        void *      recvBuffer;
        size_t      recvSize;
        bool        isNonBlocking;
        bool        isConnected;
        bool        isRecieving;
    };

    HioChannel g_Channels[HioChannelMaxCount];
    int g_UniqueId = 0;
    int g_LastError = HIO_STATUS_OK;

    void SetLastError(HIOStatus status)
    {
        switch(status)
        {
        case HIO_STATUS_NEW_CONNECTION:
        case HIO_STATUS_OK:
            g_LastError = HtcsOK;
            break;

        case HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE:
            g_LastError = htcs::HTCS_EAGAIN;
            break;

        case HIO_STATUS_NO_CONNECTIONS:
            g_LastError = htcs::HTCS_ENOTCONN;
            break;

        case HIO_STATUS_NOT_INITIALIZED:
            g_LastError = htcs::HTCS_ENETRESET;
            break;

        case HIO_STATUS_OUT_OF_MEMORY:
            g_LastError = htcs::HTCS_ENOMEM;
            break;

        case HIO_STATUS_INVALID_CHANNEL_NAME:
            g_LastError = htcs::HTCS_EMFILE;
            break;

        case HIO_STATUS_INVALID_HANDLE:
            g_LastError = htcs::HTCS_EBADF;
            break;

        case HIO_STATUS_NO_CHANNELS:
            g_LastError = htcs::HTCS_EINPROGRESS;
            break;

        case HIO_STATUS_INVALID_PARAM:
        case HIO_STATUS_INVALID_OPERATION:
            g_LastError = htcs::HTCS_EINVAL;
            break;

        default:
            g_LastError = HtcsError;
            break;
        }
    }

    void ProcStatusCallback(HIOStatus status, void * context)
    {
        HioChannel *channel = static_cast<HioChannel *>(context);
        if (status == HIO_STATUS_NEW_CONNECTION)
        {
            channel->isConnected = true;
        }
        else if (status == HIO_STATUS_NO_CONNECTIONS)
        {
            channel->isConnected = false;
        }

        OSSignalEvent(&(channel->event));
        SetLastError(status);
    }

    void ProcReadCallback(HIOStatus status, void * context)
    {
        HioChannel *channel = static_cast<HioChannel *>(context);
        channel->isRecieving = false;
        SetLastError(status);
    }

    bool IsOpenedSocket(int sock)
    {
        return (0 <= sock && sock < g_UniqueId);
    }

    bool IsValidSocket(int sock)
    {
        if (!IsOpenedSocket(sock))
        {
            return false;
        }

        return g_Channels[sock].handle >= 0;
    }

    void CopyBytes(char *dst, const char *src, size_t length)
    {
        for (int i = 0; i < length; ++i)
        {
            dst[i] = src[i];
        }
    }
}

int HtcsHelperSocket() NN_NOEXCEPT
{
    if (g_UniqueId >= HioChannelMaxCount)
    {
        g_LastError = htcs::HTCS_EMFILE;
        return HtcsError;
    }

    OSInitEvent(&g_Channels[g_UniqueId].event, FALSE, OS_EVENT_MANUAL);
    g_Channels[g_UniqueId].handle = HtcsError;
    g_Channels[g_UniqueId].isNonBlocking = false;
    g_Channels[g_UniqueId].isConnected = false;
    g_Channels[g_UniqueId].isRecieving = false;
    g_Channels[g_UniqueId].portName[0] = '\0';
    g_Channels[g_UniqueId].recvBuffer = NULL;
    g_Channels[g_UniqueId].recvSize = 0;

    return g_UniqueId++;
}

int HtcsHelperClose(int descriptor) NN_NOEXCEPT
{
    // 開いたソケット数の中に収まっているかチェック
    if (!IsOpenedSocket(descriptor))
    {
        g_LastError = htcs::HTCS_EBADF;
        return HtcsError;
    }

    HIOStatus closeResult = HIOClose(g_Channels[descriptor].handle);
    SetLastError(closeResult);

    if (closeResult == HIO_STATUS_OK)
    {
        g_Channels[descriptor].handle = HtcsError;
        g_Channels[descriptor].isConnected = false;
        return HtcsOK;
    }

    return HtcsError;
}

int HtcsHelperConnect(int descriptor, const htcs::SockAddrHtcs* address) NN_NOEXCEPT
{
    NN_UNUSED(descriptor);
    NN_UNUSED(address);

    // Cafe実装ではサポートしないので常時失敗
    g_LastError = HtcsError;
    return HtcsError;
}

int HtcsHelperBind(int descriptor, const htcs::SockAddrHtcs* address) NN_NOEXCEPT
{
    // 開いたソケット数の中に収まっているかチェック
    if (!IsOpenedSocket(descriptor))
    {
        g_LastError = htcs::HTCS_EBADF;
        return HtcsError;
    }

    CopyBytes(g_Channels[descriptor].portName, address->portName.name, nn::htcs::PortNameBufferLength);

    g_LastError = HtcsOK;
    return HtcsOK;
}

int HtcsHelperListen(int descriptor, int backlogCount) NN_NOEXCEPT
{
    NN_UNUSED(backlogCount);

    // 開いたソケットで、ポート名がバインド済みかチェック
    if (!IsOpenedSocket(descriptor) || g_Channels[descriptor].portName[0] == '\0')
    {
        g_LastError = htcs::HTCS_EBADF;
        return HtcsError;
    }

    OSResetEvent(&g_Channels[descriptor].event);
    g_Channels[descriptor].handle = HIOOpen(
        g_Channels[descriptor].portName,
        ProcStatusCallback,
        &g_Channels[descriptor],
        HIO_CHANNEL_OPTION_READ_WRITE,
        0);

    if (g_Channels[descriptor].handle >= 0)
    {
        g_LastError = HtcsOK;
        return HtcsOK;
    }

    g_LastError = HtcsError;
    return HtcsError;
}

int HtcsHelperAccept(int descriptor, htcs::SockAddrHtcs* address) NN_NOEXCEPT
{
    // HIOは接続先の情報を取得できないので第2引数は無意味
    NN_UNUSED(address);

    // Listen開始済みのソケットかチェック
    if (!IsValidSocket(descriptor))
    {
        g_LastError = htcs::HTCS_EINVAL;
        return HtcsError;
    }

    if (!g_Channels[descriptor].isNonBlocking)
    {
        OSWaitEvent(&g_Channels[descriptor].event);
    }

    return g_Channels[descriptor].isConnected ? descriptor : HtcsError;
}

htcs::ssize_t HtcsHelperRecv(int descriptor, void* buffer, size_t bufferByteSize, int flags) NN_NOEXCEPT
{
    NN_UNUSED(flags);

    if (!IsValidSocket(descriptor))
    {
        g_LastError = htcs::HTCS_EINVAL;
        return HtcsError;
    }

    if (g_Channels[descriptor].isNonBlocking)
    {
        if (g_Channels[descriptor].isRecieving)
        {
            // 予約した送受信が未解決の場合はリトライを要求してエラー終了
            g_LastError = htcs::HTCS_EAGAIN;
            return HtcsError;
        }

        if (g_Channels[descriptor].recvBuffer != NULL)
        {
            // 解決済みの読み込みがある場合は一時バッファからデータをコピーして状態を初期化して正常終了
            int size = g_Channels[descriptor].recvSize;
            CopyBytes(static_cast<char *>(g_Channels[descriptor].recvBuffer), g_Channels[descriptor].hioBuffer, size);

            g_Channels[descriptor].recvBuffer = NULL;
            g_Channels[descriptor].recvSize = 0;

            return size;
        }

        // ユーザーから指示されたバッファとサイズを記憶して非同期読み込み開始
        g_Channels[descriptor].isRecieving = true;
        g_Channels[descriptor].recvBuffer = buffer;
        g_Channels[descriptor].recvSize = bufferByteSize;
        int result = HIOReadAsync(
            g_Channels[descriptor].handle,
            bufferByteSize,
            g_Channels[descriptor].hioBuffer,
            ProcReadCallback,
            &g_Channels[descriptor]);

        if (result == HIO_STATUS_OK)
        {
            // 読み込み開始後すぐに完了した場合は即座に値を書き戻して正常終了
            if (!g_Channels[descriptor].isRecieving)
            {
                int size = g_Channels[descriptor].recvSize;
                CopyBytes(static_cast<char *>(g_Channels[descriptor].recvBuffer), g_Channels[descriptor].hioBuffer, size);
                g_Channels[descriptor].recvBuffer = NULL;
                g_Channels[descriptor].recvSize = 0;

                return size;
            }

            // 読み込みを開始してまだデータが届いてない状態なので0バイトとして返す
            return 0;
        }

        return HtcsError;
    }

    int result = HIORead(g_Channels[descriptor].handle, bufferByteSize, buffer);
    return result;
}

htcs::ssize_t HtcsHelperSend(int descriptor, const void* buffer, size_t bufferByteSize, int flags) NN_NOEXCEPT
{
    NN_UNUSED(flags);

    if (!IsValidSocket(descriptor))
    {
        g_LastError = htcs::HTCS_EINVAL;
        return HtcsError;
    }

    if (g_Channels[descriptor].isNonBlocking)
    {
        if (bufferByteSize > HioTemporaryBufferLength)
        {
            g_LastError = htcs::HTCS_ENOMEM;
            return HtcsError;
        }

        OSResetEvent(&g_Channels[descriptor].event);
        int result = HIOWriteAsync(
            g_Channels[descriptor].handle,
            bufferByteSize,
            buffer,
            ProcStatusCallback,
            &g_Channels[descriptor]);

        return result == HIO_STATUS_OK ?
            static_cast<htcs::ssize_t>(bufferByteSize) : HtcsError;
    }

    int result = HIOWrite(g_Channels[descriptor].handle, bufferByteSize, buffer);
    return result;
}

int HtcsHelperShutdown(int descriptor, int how) NN_NOEXCEPT
{
    NN_UNUSED(descriptor);
    NN_UNUSED(how);

    // 開いたソケット数の中に収まっているかチェック
    if (!IsValidSocket(descriptor))
    {
        g_LastError = htcs::HTCS_EINVAL;
        return HtcsError;
    }

    // シャットダウンの代わりにイベントをシグナルしてブロックを解除
    OSSignalEvent(&g_Channels[descriptor].event);

    return HtcsOK;
}

int HtcsHelperFcntl(int descriptor, int command, int value) NN_NOEXCEPT
{
    if (!IsOpenedSocket(descriptor))
    {
        g_LastError = htcs::HTCS_EBADF;
        return HtcsError;
    }

    switch (command)
    {
    case htcs::HTCS_F_GETFL:
        g_LastError = HtcsOK;
        return g_Channels[descriptor].isNonBlocking ? htcs::HTCS_O_NONBLOCK : 0;

    case htcs::HTCS_F_SETFL:
        if (value == htcs::HTCS_O_NONBLOCK)
        {
            g_Channels[descriptor].isNonBlocking = true;
            g_LastError = HtcsOK;
            return HtcsOK;
        }
        else if (value == 0)
        {
            g_Channels[descriptor].isNonBlocking = false;
            g_LastError = HtcsOK;
            return HtcsOK;
        }

    default:
        break;
    }

    g_LastError = htcs::HTCS_EINVAL;
    return HtcsError;
}

int HtcsHelperGetLastError() NN_NOEXCEPT
{
    return g_LastError;
}

const htcs::HtcsPeerName HtcsHelperGetPeerNameAny() NN_NOEXCEPT
{
    return PeerNameAny;
}

const htcs::HtcsPeerName HtcsHelperGetDefaultHostName() NN_NOEXCEPT
{
    return DefaultHostName;
}

}}

#endif
