﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_Endian.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_BinTypes.h>
#include <nn/ige/ige_EditService.h>
#include <nn/ige/ige_Node.h>

#include "detail/ige_Allocator.h"
#include "detail/ige_NodeIdManager.h"
#include "detail/ige_Channel.h"

#include "detail/messages/ige_VersionMessage.h"
#include "detail/messages/ige_NodeMessage.h"
#include "detail/messages/ige_RemoveNodeMessage.h"

#include "detail/messages/ige_ValueMessage.h"
#include "detail/messages/ige_ValueRequestMessage.h"
#include "detail/messages/ige_MetadataMessage.h"
#include "detail/messages/ige_MetadataRequestMessage.h"
#include "detail/messages/ige_UserCommandMessage.h"
#include "detail/messages/ige_ChildNodeRequestMessage.h"

namespace nn { namespace ige {

namespace {
    const int s_Signature = NN_UTIL_CREATE_SIGNATURE_4('I', 'G', 'E', '_');
} // anonymous namespace

EditService::EditService() NN_NOEXCEPT
    : NodeHolder(this)
    , m_PortName()
    , m_pAllocator(NULL)
    , m_pNodeIdManager(NULL)
    , m_pChannel(NULL)
{
    SetEnableAutoNotify(true);
}

EditService::~EditService() NN_NOEXCEPT
{
    Finalize();
}

void EditService::Initialize(InitializeArg arg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(arg.allocateFunc);
    NN_SDK_REQUIRES_NOT_NULL(arg.freeFunc);

    Finalize();

    m_PortName = arg.portName;

    void* pResourceBuffer = arg.allocateFunc(sizeof(detail::IgeMemoryResource), NN_ALIGNOF(detail::IgeMemoryResource), arg.pAllocateUserData);
    detail::IgeMemoryResource* pResource = new (pResourceBuffer) detail::IgeMemoryResource(arg.allocateFunc, arg.freeFunc, arg.pAllocateUserData, arg.pFreeUserData);
    void* pAllocator = pResource->allocate(sizeof(detail::IgeAllocator), NN_ALIGNOF(detail::IgeAllocator));
    m_pAllocator = new (pAllocator) detail::IgeAllocator(pResource);

    void* pManager = m_pAllocator->Allocate<detail::NodeIdManager>();
    m_pNodeIdManager = new (pManager) detail::NodeIdManager(m_pAllocator);
}

bool EditService::IsInitialized() const NN_NOEXCEPT
{
    return m_pAllocator != NULL;
}

void EditService::Finalize() NN_NOEXCEPT
{
    m_PortName.clear();

    Stop();

    UnregisterAll();

    if (m_pNodeIdManager)
    {
        m_pNodeIdManager->~NodeIdManager();
        m_pAllocator->Free(m_pNodeIdManager);
        m_pNodeIdManager = NULL;
    }

    if (m_pAllocator)
    {
        detail::IgeMemoryResource* pResource = m_pAllocator->GetResource();
        pResource->Free(m_pAllocator);
        m_pAllocator = NULL;
        pResource->Free(pResource);
    }
}

bool EditService::Start() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    if (m_pChannel == NULL)
    {
        void* pChannel = m_pAllocator->Allocate<detail::Channel>();
        m_pChannel = new (pChannel) detail::Channel(m_PortName, m_pAllocator, s_Signature);
    }

    return m_pChannel->IsOpened();
}

bool EditService::IsStarted() const NN_NOEXCEPT
{
    return m_pChannel != NULL;
}

void EditService::Stop() NN_NOEXCEPT
{
    if (m_pChannel)
    {
        m_pChannel->~Channel();
        m_pAllocator->Free(m_pChannel);
        m_pChannel = NULL;
    }
}

bool EditService::IsConnected() const NN_NOEXCEPT
{
    return IsStarted() && m_pChannel->IsConnected();
}

void EditService::SendMessage(const detail::Message& message) NN_NOEXCEPT
{
    if (IsConnected())
    {
        BufferWriter writer(m_pAllocator->GetResource());
        message.Write(&writer);
        m_pChannel->SendPacket(writer.GetBuffer());
    }
}

const util::string_view& EditService::GetPortName() const NN_NOEXCEPT
{
    return m_PortName;
}

void EditService::Poll() NN_NOEXCEPT
{
    if (!IsStarted())
    {
        return;
    }

    bool oldConnected = m_pChannel->IsConnected();

    m_pChannel->Poll();

    bool newConnected = m_pChannel->IsConnected();

    if (oldConnected && !newConnected)
    {
        NN_IGE_LOG("Disconnected\n");
        m_pChannel->ResetConnection();
    }
    else if (!oldConnected && newConnected)
    {
        NN_IGE_LOG("Connected\n");
        SendMessage(detail::VersionMessage(detail::Version::CurrentVersion()));

        if (IsEnabledAutoNotify())
        {
            PushChildInfo();
        }
    }
}

void EditService::Update() NN_NOEXCEPT
{
    if (!IsConnected())
    {
        return;
    }

    auto receiveFunc = [this](const BufferView& buffer) { this->ReadMessage(buffer); };
    m_pChannel->ReceivePacket(receiveFunc);
}

detail::IgeAllocator& EditService::GetAllocator() NN_NOEXCEPT
{
    return *m_pAllocator;
}

void EditService::RegisterToService(Node* pNode) NN_NOEXCEPT
{
    if (IsInitialized() && pNode->GetId() == detail::s_InvalidId)
    {
        pNode->m_Id = m_pNodeIdManager->Register(pNode);
    }
}

void EditService::UnregisterFromService(Node* pNode) NN_NOEXCEPT
{
    NodeIdType id = pNode->GetId();
    if (IsInitialized() && m_pNodeIdManager->Unregister(id))
    {
        pNode->m_Id = detail::s_InvalidId;
        SendMessage(detail::RemoveNodeMessage(id));
    }
}

void EditService::ReadMessage(const BufferView& buffer) NN_NOEXCEPT
{
    auto type = detail::Message::GetMessageType(buffer);

    if (detail::SystemMessage::IsTarget(type))
    {
        if (detail::VersionMessage::IsTarget(type))
        {
            ReceiveMessage(detail::VersionMessage::Read(buffer));
            return;
        }
    }
    else if (detail::NodeTargetMessage::IsTarget(type))
    {
        if (detail::ValueMessage::IsTarget(type))
        {
            ReceiveMessage(detail::ValueMessage::Read(buffer));
            return;
        }
        else if (detail::ValueRequestMessage::IsTarget(type))
        {
            ReceiveMessage(detail::ValueRequestMessage::Read(buffer));
            return;
        }
        else if (detail::MetadataRequestMessage::IsTarget(type))
        {
            ReceiveMessage(detail::MetadataRequestMessage::Read(buffer));
            return;
        }
        else if (detail::ChildNodeRequestMessage::IsTarget(type))
        {
            ReceiveMessage(detail::ChildNodeRequestMessage::Read(buffer));
            return;
        }
        else if (detail::UserCommandMessage::IsTarget(type))
        {
            ReceiveMessage(detail::UserCommandMessage::Read(buffer));
            return;
        }
    }

    NN_IGE_NOT_SUPPORTED();
}

void EditService::ReceiveMessage(const detail::SystemMessage& message) NN_NOEXCEPT
{
    // ノード情報要求：その場で送り返す
    if (detail::VersionMessage::IsTarget(message.GetType()))
    {
        auto version = static_cast<const detail::VersionMessage&>(message).GetVersion();
        if (!detail::Version::CurrentVersion().IsValid(version))
        {
            NN_IGE_WARNING_LOG("invalid packet version '%d.%d.%d' received.", version.major, version.minor, version.micro);
            m_pChannel->ResetConnection();
        }
    }
}

void EditService::ReceiveMessage(const detail::NodeTargetMessage& message) NN_NOEXCEPT
{
    if (Node* pNode = m_pNodeIdManager->Find(message.GetId()))
    {
        pNode->OnMessageReceived(message);
    }
}

}} // namespace nn::ige
