﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/ige/ige_BufferWriter.h>
#include <nn/ige/ige_Buffer.h>
#include <nn/ige/ige_NodeIdType.h>

#include "../util/ige_BufferReader.h"
#include "../ige_DetailDefine.h"

namespace nn { namespace ige { namespace detail {

// メッセージを受信側の誰にハンドリングしてもらうか
enum MessageTarget : int32_t
{
    MessageTarget_Invalid = 0,
    MessageTarget_System  = 0x0100,
    MessageTarget_Node    = 0x0200,
};

enum MessageType : int32_t
{
    MessageType_Invalid = 0,

    MessageType_Version         = MessageTarget_System | 0x01,  // ランタイム→ツール or 双方向
    MessageType_Node            = MessageTarget_System | 0x02,  // ランタイム→ツール
    MessageType_RemoveNode      = MessageTarget_System | 0x03,  // ランタイム→ツール

    MessageType_Metadata        = MessageTarget_Node | 0x01,    // ランタイム→ツール
    MessageType_MetadataRequest = MessageTarget_Node | 0x02,    // ツール→ランタイム
    MessageType_Value           = MessageTarget_Node | 0x03,    // 双方向
    MessageType_ValueRequest    = MessageTarget_Node | 0x04,    // ツール→ランタイム
    MessageType_UserCommand     = MessageTarget_Node | 0x05,    // 双方向
    MessageType_ChildNodeRequest= MessageTarget_Node | 0x06,    // ツール→ランタイム
};

class Message
{
public:
    virtual ~Message() NN_NOEXCEPT
    {
    }

    virtual MessageType GetType() const NN_NOEXCEPT = 0;

    virtual void Write(BufferWriter* pWriter) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pWriter);
        pWriter->Write(static_cast<int32_t>(GetType()));
        WritePacket(pWriter);
    }

    static MessageType GetMessageType(const BufferView& buffer) NN_NOEXCEPT
    {
        const size_t size = sizeof(MessageType);

        if (buffer.GetSize() < size)
        {
            return MessageType_Invalid;
        }

        int32_t value;
        std::memcpy(&value, buffer.GetPtr(), size);
        return static_cast<MessageType>(util::LoadLittleEndian(&value));
    }

protected:
    virtual void WritePacket(BufferWriter*) const NN_NOEXCEPT
    {
        NN_IGE_NOT_SUPPORTED();
    }
};

class SystemMessage : public Message
{
public:
    static bool IsTarget(MessageType type) NN_NOEXCEPT
    {
        return (type & MessageTarget_System) != 0;
    }

    static bool IsTarget(const BufferView& buffer) NN_NOEXCEPT
    {
        return (GetMessageType(buffer) & MessageTarget_System) != 0;
    }
};

class NodeTargetMessage : public Message
{
public:
    explicit NodeTargetMessage(NodeIdType id) NN_NOEXCEPT
        : m_Id(id)
    {
    }

    NodeIdType GetId() const NN_NOEXCEPT
    {
        return m_Id;
    }

    static bool IsTarget(MessageType type) NN_NOEXCEPT
    {
        return (type & MessageTarget_Node) != 0;
    }

    static bool IsTarget(const BufferView& buffer) NN_NOEXCEPT
    {
        return (GetMessageType(buffer) & MessageTarget_Node) != 0;
    }

    virtual void Write(BufferWriter* pWriter) const NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(pWriter);
        pWriter->Write(static_cast<int32_t>(GetType()));
        pWriter->Write(static_cast<uint32_t>(m_Id));
        WritePacket(pWriter);
    }

private:
    NodeIdType m_Id;
};

}}} // namespace nn::ige::detail
