﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Mutex.h>
#include <nn/util/util_StringView.h>
#include <nn/ige/ige_Buffer.h>
#include "ige_Allocator.h"
#include "util/ige_IgeDeque.h"
#include "util/ige_ScopedLock.h"

namespace nn { namespace ige { namespace detail {

class ServerSocket;

class Channel
{
    NN_DISALLOW_COPY(Channel);
public:
    typedef void(*ReceivePacketCallbackType)(const BufferView&);

    Channel(const util::string_view& channelName, IgeAllocator* pAllocator) NN_NOEXCEPT;
    Channel(const util::string_view& channelName, IgeAllocator* pAllocator, int32_t signature) NN_NOEXCEPT;

    ~Channel() NN_NOEXCEPT;

    void Poll() NN_NOEXCEPT;

    bool IsOpened() const NN_NOEXCEPT;

    bool IsConnected() const NN_NOEXCEPT;

    void ResetConnection() NN_NOEXCEPT;

    void SendPacket(const BufferView& packet) NN_NOEXCEPT;

    template<typename Functor>
    void ReceivePacket(Functor packetReceiver) NN_NOEXCEPT
    {
        if (!IsConnected())
        {
            return;
        }

        ScopedLock lock(&m_ReadMutex);

        for (BufferQueue::iterator iter = m_ReadPacketQueue.begin(); iter != m_ReadPacketQueue.end(); ++iter)
        {
            packetReceiver(*iter);
            m_pAllocator->FreeBuffer(&*iter);
        }

        m_ReadPacketQueue.clear();
    }

private:
    enum ReadState
    {
        ReadState_Invalid,
        ReadState_Ready,
        ReadState_Signature,
        ReadState_SignatureReading,
        ReadState_Length,
        ReadState_LengthReading,
        ReadState_PayloadReading,
    };

    void Initialize(const util::string_view& channelName) NN_NOEXCEPT;

    bool ReadPacket() NN_NOEXCEPT;
    bool WritePacket(const BufferView& buffer) NN_NOEXCEPT;
    bool WriteTestPacket() NN_NOEXCEPT;

    void Close() NN_NOEXCEPT;
    void ResetReadState() NN_NOEXCEPT;

private:
    typedef IgeDeque<BufferReference>::Type BufferQueue;

    IgeAllocator* m_pAllocator;
    ServerSocket* m_pSocket;

    os::Mutex m_WriteMutex;
    os::Mutex m_ReadMutex;
    BufferQueue m_ReadPacketQueue;

    ReadState m_ReadState;

    void* m_pReadingPayload;
    int32_t m_ReadingSignature;
    int32_t m_ReadingLength;

    int32_t m_Signature;
    bool m_HasSignature;
};

}}} // namespace nn::ige::detail
