﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../util/g3d_ViewerUtility.h"
#include "../g3d_Allocator.h"
#include "../g3d_ViewerKeyManager.h"

#include <nn/g3d/g3d_ResFile.h>
#include <nn/diag/text/diag_SdkTextG3dviewer.h>
#include <nn/g3d/g3d_ModelObj.h>
#include <nn/g3d/g3d_SkeletonObj.h>
#include <nn/g3d/g3d_ResAnimCurve.h>
#include <nn/g3d/g3d_MaterialAnimObj.h>
#include <nn/gfx/gfx_ResTexture.h>
#include <nn/os.h>

using namespace nn::g3d;

#define CASE_RETURN_STRING(cmd) case cmd: return #cmd;

namespace {
    struct BindTextureCallbackUserData
    {
        nn::gfx::ResTextureFile* pResTextureFile;
        nn::g3d::viewer::detail::CallbackCaller* pCallbackCaller;
    };

    nn::g3d::TextureRef BindTextureCallback(const char* name, void* pUserData) NN_NOEXCEPT
    {
        BindTextureCallbackUserData* pTextureBindUserData = reinterpret_cast<BindTextureCallbackUserData*>(pUserData);
        nn::g3d::viewer::detail::CallbackCaller* pCallbackCaller = pTextureBindUserData->pCallbackCaller;
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pCallbackCaller);
        return pCallbackCaller->CallTextureBindCallback(name, pTextureBindUserData->pResTextureFile);
    }

    template <typename TRes>
    void ForceUnbindTextureImpl(TRes* pRes,
        const nn::gfx::ResTextureFile* pResTextureFile,
        nn::g3d::viewer::detail::CallbackCaller& callback) NN_NOEXCEPT
    {
        for (int idxTex = 0, texCount = pResTextureFile->GetTextureDic()->GetCount(); idxTex < texCount; ++idxTex)
        {
            const nn::gfx::ResTexture* pResTexture = pResTextureFile->GetResTexture(idxTex);

            // ユーザがダミーテクスチャを挿せるようにNULLでバインドする
            nn::g3d::TextureRef texRef = callback.CallTextureBindCallback(pResTexture->GetName(), nullptr);
            bool isBound = pRes->ForceBindTexture(texRef, pResTexture->GetName());
            if (!isBound)
            {
                NN_G3D_VIEWER_DEBUG_PRINT(
                    "Failed to unbind texture \"%s\" for \"%s\"\n",
                    pResTexture->GetName(), pRes->GetName());
            }
        }
    }

    void ForceBindTextureImpl(nn::g3d::ModelObj* pModelObj, const char* pTextureName, TextureRef& texRef) NN_NOEXCEPT
    {
        for (int idxMat = 0, matCount = pModelObj->GetMaterialCount(); idxMat < matCount; ++idxMat)
        {
            nn::g3d::MaterialObj* pMatObj = pModelObj->GetMaterial(idxMat);
            for (int idxMatTex = 0, matTexCount = pMatObj->GetTextureCount(); idxMatTex < matTexCount; ++idxMatTex)
            {
                if (strcmp(pMatObj->GetTextureName(idxMatTex), pTextureName) == 0)
                {
                    pMatObj->SetTexture(idxMatTex, texRef);
                }
            }
        }
    }

    void ForceBindTextureImpl(nn::g3d::MaterialAnimObj* pMaterialAnimObj, const char* pTextureName, TextureRef& texRef) NN_NOEXCEPT
    {
        for (int idxMat = 0, matCount = pMaterialAnimObj->GetResource()->GetPerMaterialAnimCount(); idxMat < matCount; ++idxMat)
        {
            for (int idxMatTex = 0, matTexCount = pMaterialAnimObj->GetTextureCount(); idxMatTex < matTexCount; ++idxMatTex)
            {
                if (strcmp(pMaterialAnimObj->GetResource()->GetTextureName(idxMatTex), pTextureName) == 0)
                {
                    pMaterialAnimObj->SetTexture(idxMatTex, texRef);
                }
            }
        }
    }

    void ForceBindTextureImpl(
        nn::g3d::MaterialAnimObj* pMaterialAnimObj,
        const nn::gfx::ResTextureFile* pTargetResTextureFile,
        const nn::gfx::ResTextureFile* pBindSourceResTextureFile,
        nn::g3d::viewer::detail::CallbackCaller& callback) NN_NOEXCEPT
    {
        ResMaterialAnim* pResMaterialAnim = const_cast<ResMaterialAnim*>(pMaterialAnimObj->GetResource());
        if (pResMaterialAnim->GetTexturePatternAnimCount() == 0)
        {
            return;
        }

        for (int idxTex = 0, texCount = pTargetResTextureFile->GetTextureDic()->GetCount(); idxTex < texCount; ++idxTex)
        {
            const nn::gfx::ResTexture* pResTexture = pTargetResTextureFile->GetResTexture(idxTex);
            nn::g3d::TextureRef texRef = callback.CallTextureBindCallback(pResTexture->GetName(), pBindSourceResTextureFile);

            // Obj にバインドされているテクスチャを強制上書き
            ForceBindTextureImpl(pMaterialAnimObj, pResTexture->GetName(), texRef);
        }
    }

    void ForceBindTextureImpl(
        nn::g3d::ModelObj* pModelObj,
        const nn::gfx::ResTextureFile* pResTextureFile,
        const nn::gfx::ResTextureFile* pBindSourceResTextureFile,
        nn::g3d::viewer::detail::CallbackCaller& callback) NN_NOEXCEPT
    {
        for (int idxTex = 0, texCount = pResTextureFile->GetTextureDic()->GetCount(); idxTex < texCount; ++idxTex)
        {
            const nn::gfx::ResTexture* pResTexture = pResTextureFile->GetResTexture(idxTex);
            nn::g3d::TextureRef texRef = callback.CallTextureBindCallback(pResTexture->GetName(), pBindSourceResTextureFile);

            // ModelObj にバインドされているテクスチャを強制上書き
            ForceBindTextureImpl(pModelObj, pResTexture->GetName(), texRef);
        }
    }
}

const char* nn::g3d::viewer::detail::GetTextureName(nn::gfx::ResTexture* pResTexture) NN_NOEXCEPT
{
    nn::gfx::ResTextureData* pResTextureData = &pResTexture->ToData();
    nn::util::BinPtrToString* bintstr = &pResTextureData->pName;
    nn::util::BinString* nameString = bintstr->Get();
    return nameString->Get().data();
}

void nn::g3d::viewer::detail::PrintTextureNames(nn::gfx::ResTextureFile* pResTextureFile) NN_NOEXCEPT
{
    NN_G3D_VIEWER_DEBUG_PRINT("Texture Names for Texture File %s:\n", pResTextureFile->ToData().fileHeader.GetFileName());
    for (int index = 0, count = pResTextureFile->ToData().textureContainerData.textureCount; index < count; ++index)
    {
        nn::gfx::ResTexture* pResTexture = pResTextureFile->GetResTexture(index);
        NN_UNUSED(pResTexture);
        NN_G3D_VIEWER_DEBUG_PRINT("[%d]: %s\n", index, GetTextureName(pResTexture));
    }
}

bool nn::g3d::viewer::detail::IsTextureFileInitialized(nn::gfx::ResTextureFile* pResTextureFile) NN_NOEXCEPT
{
    nn::gfx::MemoryPool* pMemoryPool = static_cast<nn::gfx::MemoryPool*>(pResTextureFile->ToData().textureContainerData.pTextureMemoryPool.Get());
    return nn::gfx::IsInitialized(*pMemoryPool);
}

void nn::g3d::viewer::detail::BindTexture(
    nn::g3d::ResFile* pBindDestinationResFile,
    nn::gfx::ResTextureFile* pResTextureFile,
    nn::g3d::viewer::detail::CallbackCaller& callback) NN_NOEXCEPT
{
    BindTextureCallbackUserData userData;
    userData.pCallbackCaller = &callback;

    // モデルのテクスチャバインド
    for (int idxModel = 0, numModel = pBindDestinationResFile->GetModelCount(); idxModel < numModel; ++idxModel)
    {
        ResModel* pResModel = pBindDestinationResFile->GetModel(idxModel);
        userData.pResTextureFile = pResTextureFile;
        BindResult isBound = pResModel->BindTexture(BindTextureCallback, &userData);
        if (isBound.IsMissed())
        {
            NN_G3D_VIEWER_DEBUG_PRINT("Failed to bind textures for model \"%s\"\n", pResModel->GetName());
        }
    }

    // テクスチャパターンアニメーションのテクスチャバインド
    for (int idxMatAnim = 0, numMatAnim = pBindDestinationResFile->GetMaterialAnimCount();
        idxMatAnim < numMatAnim; ++idxMatAnim)
    {
        ResMaterialAnim* pResMaterialAnim = pBindDestinationResFile->GetMaterialAnim(idxMatAnim);
        if (pResMaterialAnim->GetTexturePatternAnimCount() > 0)
        {
            userData.pResTextureFile = pResTextureFile;
            BindResult isBound = pResMaterialAnim->BindTexture(BindTextureCallback, &userData);
            if (isBound.IsMissed())
            {
                NN_G3D_VIEWER_DEBUG_PRINT("Failed to bind textures for texture pattern animation \"%s\"\n", pResMaterialAnim->GetName());
            }
        }
    }
}

void nn::g3d::viewer::detail::ForceBindTexture(
    nn::g3d::ResModel* pBindDestinationResModel,
    const nn::gfx::ResTextureFile* pResTextureFile,
    nn::g3d::viewer::detail::CallbackCaller& callback) NN_NOEXCEPT
{
    NN_G3D_VIEWER_REQUIRES_NOT_NULL(pBindDestinationResModel);
    NN_G3D_VIEWER_REQUIRES_NOT_NULL(pResTextureFile);
    for (int idxTex = 0, texCount = pResTextureFile->GetTextureDic()->GetCount(); idxTex < texCount; ++idxTex)
    {
        const nn::gfx::ResTexture* pResTexture = pResTextureFile->GetResTexture(idxTex);
        nn::gfx::DescriptorSlot descriptorSlot;
        pResTexture->GetUserDescriptorSlot(&descriptorSlot);
        nn::g3d::TextureRef texRef = callback.CallTextureBindCallback(pResTexture->GetName(), pResTextureFile);
        bool isBound = pBindDestinationResModel->ForceBindTexture(texRef, pResTexture->GetName());
        if (!isBound)
        {
            NN_G3D_VIEWER_DEBUG_PRINT("Failed to force bind texture \"%s\" for model \"%s\"\n",
                pResTexture->GetName(), pBindDestinationResModel->GetName());
        }
    }
}

void nn::g3d::viewer::detail::ForceBindTexture(
    nn::g3d::ModelObj* pModelObj,
    const nn::gfx::ResTextureFile* pResTextureFile,
    nn::g3d::viewer::detail::CallbackCaller& callback) NN_NOEXCEPT
{
    ForceBindTextureImpl(pModelObj, pResTextureFile, pResTextureFile, callback);
}

void nn::g3d::viewer::detail::ForceUnbindTexture(
    nn::g3d::ModelObj* pModelObj,
    const nn::gfx::ResTextureFile* pResTextureFile,
    nn::g3d::viewer::detail::CallbackCaller& callback) NN_NOEXCEPT
{
    ForceBindTextureImpl(pModelObj, pResTextureFile, nullptr, callback);
}

void nn::g3d::viewer::detail::ForceBindTexture(
    nn::g3d::ResMaterialAnim* pBindDestinationResAnim,
    const nn::gfx::ResTextureFile* pResTextureFile,
    nn::g3d::viewer::detail::CallbackCaller& callback) NN_NOEXCEPT
{
    for (int idxTex = 0, texCount = pResTextureFile->GetTextureDic()->GetCount(); idxTex < texCount; ++idxTex)
    {
        const nn::gfx::ResTexture* pResTexture = pResTextureFile->GetResTexture(idxTex);
        nn::gfx::DescriptorSlot descriptorSlot;
        pResTexture->GetUserDescriptorSlot(&descriptorSlot);
        nn::g3d::TextureRef texRef = callback.CallTextureBindCallback(pResTexture->GetName(), pResTextureFile);
        if (pBindDestinationResAnim->GetTexturePatternAnimCount() > 0)
        {
            bool isBound = pBindDestinationResAnim->ForceBindTexture(texRef, pResTexture->GetName());
            if (!isBound)
            {
                NN_G3D_VIEWER_DEBUG_PRINT("Failed to force bind texture \"%s\" for texture pattern animation \"%s\"\n",
                    pResTexture->GetName(), pBindDestinationResAnim->GetName());
            }
        }
    }
}

void nn::g3d::viewer::detail::ForceBindTexture(
    nn::g3d::MaterialAnimObj* pMaterialAnimObj,
    const nn::gfx::ResTextureFile* pResTextureFile,
    nn::g3d::viewer::detail::CallbackCaller& callback) NN_NOEXCEPT
{
    ForceBindTextureImpl(pMaterialAnimObj, pResTextureFile, pResTextureFile, callback);
}

void nn::g3d::viewer::detail::ForceUnbindTexture(
    nn::g3d::MaterialAnimObj* pMaterialAnimObj,
    const nn::gfx::ResTextureFile* pResTextureFile,
    nn::g3d::viewer::detail::CallbackCaller& callback) NN_NOEXCEPT
{
    ForceBindTextureImpl(pMaterialAnimObj, pResTextureFile, nullptr, callback);
}

void nn::g3d::viewer::detail::ForceUnbindTexture(
    nn::g3d::ResModel* pResModel,
    const nn::gfx::ResTextureFile* pResTextureFile,
    nn::g3d::viewer::detail::CallbackCaller& callback) NN_NOEXCEPT
{
    ForceUnbindTextureImpl<nn::g3d::ResModel>(
        pResModel, pResTextureFile, callback);
}

void nn::g3d::viewer::detail::ForceUnbindTexture(
    nn::g3d::ResMaterialAnim* pResAnim,
    const nn::gfx::ResTextureFile* pResTextureFile,
    nn::g3d::viewer::detail::CallbackCaller& callback) NN_NOEXCEPT
{
    ForceUnbindTextureImpl<nn::g3d::ResMaterialAnim>(
        pResAnim, pResTextureFile, callback);
}

void nn::g3d::viewer::detail::ForceBindTexture(
    nn::g3d::ResFile* pBindDestinationResFile,
    const nn::gfx::ResTextureFile* pResTextureFile,
    nn::g3d::viewer::detail::CallbackCaller& callback) NN_NOEXCEPT
{
    // モデルのテクスチャバインド
    for (int idxModel = 0, numModel = pBindDestinationResFile->GetModelCount(); idxModel < numModel; ++idxModel)
    {
        ResModel* pResModel = pBindDestinationResFile->GetModel(idxModel);
        ForceBindTexture(pResModel, pResTextureFile, callback);
    }

    // テクスチャパターンアニメーションのテクスチャバインド
    for (int idxMatAnim = 0, numMatAnim = pBindDestinationResFile->GetMaterialAnimCount();
        idxMatAnim < numMatAnim; ++idxMatAnim)
    {
        ResMaterialAnim* pResMaterialAnim = pBindDestinationResFile->GetMaterialAnim(idxMatAnim);
        ForceBindTexture(pResMaterialAnim, pResTextureFile, callback);
    }
}


bool nn::g3d::viewer::detail::ExaminesTextureFileUsed(const ResMaterialAnim* pResMatAnim, const nn::gfx::ResTextureFile* pResTextureFile) NN_NOEXCEPT
{
    for (int idxTex = 0, texCount = pResTextureFile->GetTextureDic()->GetCount(); idxTex < texCount; ++idxTex)
    {
        const nn::gfx::ResTexture* pResTexture = pResTextureFile->GetResTexture(idxTex);
        for (int idxMatAnimTex = 0, count = pResMatAnim->GetTextureCount(); idxMatAnimTex < count; ++idxMatAnimTex)
        {
            const nn::gfx::TextureView* pTexView = pResMatAnim->GetTexture(idxMatAnimTex);
            if (pTexView == pResTexture->GetTextureView())
            {
                return true;
            }
        }
    }

    return false;
}

namespace nn { namespace g3d { namespace viewer { namespace detail {

    const char* GetEditCommandString(nn::g3d::viewer::detail::CommandFlag command) NN_NOEXCEPT
    {
        switch (command)
        {
            CASE_RETURN_STRING(FILEDATA_LOAD_FILE_COMMAND_FLAG);
            CASE_RETURN_STRING(FILEDATA_UNLOAD_FILE_COMMAND_FLAG);
            CASE_RETURN_STRING(FILEDATA_RELOAD_FILE_COMMAND_FLAG);
            CASE_RETURN_STRING(FILEDATA_UNLOAD_ALL_COMMAND_FLAG);

            CASE_RETURN_STRING(SYSTEM_PING_RECV_COMMAND_FLAG);
            CASE_RETURN_STRING(SYSTEM_PING_SEND_COMMAND_FLAG);
            CASE_RETURN_STRING(SYSTEM_BEGIN_FREEZE_COMMAND_FLAG);
            CASE_RETURN_STRING(SYSTEM_BEGIN_FREEZE_NO_SYNC_COMMAND_FLAG);
            CASE_RETURN_STRING(SYSTEM_END_FREEZE_COMMAND_FLAG);
            CASE_RETURN_STRING(SYSTEM_PACKET_VERSION_ERROR_COMMAND_FLAG);
            CASE_RETURN_STRING(SYSTEM_RUNTIME_LOG_COMMAND_FLAG);
            CASE_RETURN_STRING(SYSTEM_RUNTIME_STATE_NORMAL_COMMAND_FLAG);
            CASE_RETURN_STRING(SYSTEM_RUNTIME_ERROR_COMMAND_FLAG);

            CASE_RETURN_STRING(EDIT_MATERIAL_COMMAND_FLAG);

            CASE_RETURN_STRING(EDIT_SEND_ATTACH_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_RECV_ATTACH_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_SEND_DETACH_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_FILE_LOADED_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_FILE_RELOADED_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_SEND_RENDER_INFO_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_RECV_RENDER_INFO_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_SELECT_EDIT_RENDER_INFO_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_RENDER_INFO_ARRAY_SIZE_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_UPDATE_RENDER_INFO_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_SEND_MODIFIED_SHADER_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_RECV_MODIFIED_SHADER_COMMAND_FLAG);

            CASE_RETURN_STRING(EDIT_LOAD_SHADER_ARCHIVE_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_RESET_SHADER_ARCHIVE_COMMAND_FLAG);

            CASE_RETURN_STRING(ANIMATION_PLAY_FRAME_CTRL_COMMAND_FLAG);
            CASE_RETURN_STRING(ANIMATION_STOP_FRAME_CTRL_COMMAND_FLAG);
            CASE_RETURN_STRING(ANIMATION_PLAY_POLICY_COMMAND_FLAG);
            CASE_RETURN_STRING(ANIMATION_FRAME_STEP_COMMAND_FLAG);
            CASE_RETURN_STRING(ANIMATION_FRAME_COUNT_COMMAND_FLAG);
            CASE_RETURN_STRING(ANIMATION_START_FRAME_COMMAND_FLAG);

            CASE_RETURN_STRING(EDIT_SHADER_COMMAND_FLAG);

            CASE_RETURN_STRING(EDIT_BONE_COMMAND_FLAG);

            CASE_RETURN_STRING(EDIT_MODEL_BONE_BIND_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_MODEL_LAYOUT_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_RECV_MODEL_LAYOUT_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_SEND_MODEL_LAYOUT_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_SET_SHAPE_LOD_LEVEL_COMMAND_FLAG);
            CASE_RETURN_STRING(EDIT_RESET_SHAPE_LOD_LEVEL_COMMAND_FLAG);

            CASE_RETURN_STRING(MODEL_ANIMATION_BIND_COMMAND_FLAG);
            CASE_RETURN_STRING(MODEL_ANIMATION_UNBIND_COMMAND_FLAG);
            CASE_RETURN_STRING(MODEL_ANIMATION_EDIT_CURVE_COMMAND_FLAG);
            CASE_RETURN_STRING(MODEL_ANIMATION_PLAY_COMMAND_FLAG);
            CASE_RETURN_STRING(MODEL_ANIMATION_STOP_COMMAND_FLAG);
            CASE_RETURN_STRING(MODEL_ANIMATION_EDIT_RETARGET_HOST_MODEL_COMMAND_FLAG);
            CASE_RETURN_STRING(MODEL_ANIMATION_EDIT_MIRRORING_ENABLED_COMMAND_FLAG);

            CASE_RETURN_STRING(SCENE_ANIMATION_BIND_COMMAND_FLAG);
            CASE_RETURN_STRING(SCENE_ANIMATION_UNBIND_COMMAND_FLAG);
            CASE_RETURN_STRING(SCENE_ANIMATION_EDIT_CURVE_COMMAND_FLAG);

            CASE_RETURN_STRING(PICK_RUNTIME_MODEL_COMMAND_FLAG);
            CASE_RETURN_STRING(PICK_RUNTIME_MATERIAL_COMMAND_FLAG);
            CASE_RETURN_STRING(PICK_RUNTIME_BONE_COMMAND_FLAG);
            CASE_RETURN_STRING(PICK_RUNTIME_SHAPE_COMMAND_FLAG);

            CASE_RETURN_STRING(OTHER_EXECUTE_USER_SCRIPT_FLAG);

            CASE_RETURN_STRING(TEXTURE_BIND_COMMAND_FLAG);
        default:
            return "Unkown Edit Command";
        }
    }

    const char* GetFileDataKindString(FileDataKind kind) NN_NOEXCEPT
    {
        switch(kind)
        {
            CASE_RETURN_STRING(FILEDATA_MODEL);
            CASE_RETURN_STRING(FILEDATA_TEXTURE);
            CASE_RETURN_STRING(FILEDATA_SHADER_PARAM_ANIM);
            CASE_RETURN_STRING(FILEDATA_MAT_COLOR_ANIM);
            CASE_RETURN_STRING(FILEDATA_TEXTURE_SRT_ANIM);
            CASE_RETURN_STRING(FILEDATA_TEXTURE_PATTERN_ANIM);
            CASE_RETURN_STRING(FILEDATA_SKELETAL_ANIM);
            CASE_RETURN_STRING(FILEDATA_BONE_VISIBILITY_ANIM);
            CASE_RETURN_STRING(FILEDATA_MAT_VISIBILITY_ANIM);
            CASE_RETURN_STRING(FILEDATA_MATERIAL_ANIM);
            CASE_RETURN_STRING(FILEDATA_SHAPE_ANIM);
            CASE_RETURN_STRING(FILEDATA_SCENE_ANIM);
            CASE_RETURN_STRING(FILEDATA_SHADER_DEFINE);
            CASE_RETURN_STRING(FILEDATA_SHADER_ARCHIVE);
            CASE_RETURN_STRING(FILEDATA_SHADER_PROGRAM);
            CASE_RETURN_STRING(FILEDATA_EDIT_MODEL_OBJ);
        default:
            return "Unknown FileDataKind";
        }
    }

    const char* GetAnimKindString(ViewerAnimKind kind) NN_NOEXCEPT
    {
        switch(kind)
        {
            CASE_RETURN_STRING(ViewerAnimKind_ShaderParamAnim);
            CASE_RETURN_STRING(ViewerAnimKind_ColorAnim);
            CASE_RETURN_STRING(ViewerAnimKind_TextureSrtAnim);
            CASE_RETURN_STRING(ViewerAnimKind_TexturePatternAnim);
            CASE_RETURN_STRING(ViewerAnimKind_SkeletalAnim);
            CASE_RETURN_STRING(ViewerAnimKind_BoneVisibilityAnim);
            CASE_RETURN_STRING(ViewerAnimKind_MaterialVisibilityAnim);
            CASE_RETURN_STRING(ViewerAnimKind_MaterialAnim);
            CASE_RETURN_STRING(ViewerAnimKind_ShapeAnim);
        default:
            return "Unknown ViewerAnimKind";
        }
    }

    const char* GetResName(ResFile* pResFile, FileDataKind kind) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pResFile);
        switch(kind)
        {
        case FILEDATA_MODEL:
            return pResFile->GetModelName(0);
        case FILEDATA_TEXTURE:
            {
                nn::g3d::ResExternalFile* pExternalFile = pResFile->GetExternalFile(0);
                nn::gfx::ResTextureFile* pResTextureFile = nn::gfx::ResTextureFile::ResCast(pExternalFile);
                return pResTextureFile->GetTextureDic()->GetKey(0).data();
            }
        case FILEDATA_SHADER_PARAM_ANIM:
            return pResFile->GetMaterialAnimName(0);
        case FILEDATA_MAT_COLOR_ANIM:
            return pResFile->GetMaterialAnimName(0);
        case FILEDATA_TEXTURE_SRT_ANIM:
            return pResFile->GetMaterialAnimName(0);
        case FILEDATA_TEXTURE_PATTERN_ANIM:
            return pResFile->GetMaterialAnimName(0);
        case FILEDATA_SKELETAL_ANIM:
            return pResFile->GetSkeletalAnimName(0);
        case FILEDATA_BONE_VISIBILITY_ANIM:
            return pResFile->GetBoneVisibilityAnimName(0);
        case FILEDATA_MAT_VISIBILITY_ANIM:
            return pResFile->GetMaterialAnimName(0);
        case FILEDATA_MATERIAL_ANIM:
            return pResFile->GetMaterialAnimName(0);
        case FILEDATA_SHAPE_ANIM:
            return pResFile->GetShapeAnimName(0);
        case FILEDATA_SCENE_ANIM:
            return pResFile->GetSceneAnimName(0);
        case FILEDATA_SHADER_ARCHIVE:
        case FILEDATA_SHADER_DEFINE:
        case FILEDATA_SHADER_PROGRAM:
            return pResFile->GetName();
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    FileDataKind ConvertToFileDataKind(nn::g3d::viewer::detail::ViewerAnimKind kind) NN_NOEXCEPT
    {
        switch (kind)
        {
        case nn::g3d::viewer::detail::ViewerAnimKind_ShaderParamAnim: return FILEDATA_SHADER_PARAM_ANIM;
        case nn::g3d::viewer::detail::ViewerAnimKind_ColorAnim       : return FILEDATA_MAT_COLOR_ANIM;
        case nn::g3d::viewer::detail::ViewerAnimKind_TextureSrtAnim     : return FILEDATA_TEXTURE_SRT_ANIM;
        case nn::g3d::viewer::detail::ViewerAnimKind_TexturePatternAnim : return FILEDATA_TEXTURE_PATTERN_ANIM;
        case nn::g3d::viewer::detail::ViewerAnimKind_SkeletalAnim    : return FILEDATA_SKELETAL_ANIM;
        case nn::g3d::viewer::detail::ViewerAnimKind_BoneVisibilityAnim    : return FILEDATA_BONE_VISIBILITY_ANIM;
        case nn::g3d::viewer::detail::ViewerAnimKind_MaterialVisibilityAnim     : return FILEDATA_MAT_VISIBILITY_ANIM;
        case nn::g3d::viewer::detail::ViewerAnimKind_ShapeAnim       : return FILEDATA_SHAPE_ANIM;
        case nn::g3d::viewer::detail::ViewerAnimKind_MaterialAnim : return FILEDATA_MATERIAL_ANIM;
        default:
            return FILEDATA_KIND_ERROR;
        }
    }

    bool HasShaderParamAnim(nn::g3d::ResFile* pResFile) NN_NOEXCEPT
    {
        for (int i = 0, count = pResFile->GetMaterialAnimCount(); i < count; ++i)
        {
            ResMaterialAnim* pResAnim = pResFile->GetMaterialAnim(i);
            if (pResAnim->GetShaderParamAnimCount() > 0)
            {
                return true;
            }
        }

        return false;
    }

    bool HasTexturePatternAnim(nn::g3d::ResFile* pResFile) NN_NOEXCEPT
    {
        for (int i = 0, count = pResFile->GetMaterialAnimCount(); i < count; ++i)
        {
            ResMaterialAnim* pResAnim = pResFile->GetMaterialAnim(i);
            if (pResAnim->GetTexturePatternAnimCount() > 0)
            {
                return true;
            }
        }

        return false;
    }

    bool HasMaterialVisibilityAnim(nn::g3d::ResFile* pResFile) NN_NOEXCEPT
    {
        for (int i = 0, count = pResFile->GetMaterialAnimCount(); i < count; ++i)
        {
            ResMaterialAnim* pResAnim = pResFile->GetMaterialAnim(i);
            if (pResAnim->GetVisibilityAnimCount() > 0)
            {
                return true;
            }
        }

        return false;
    }

    int GetKeyValueCount(const nn::g3d::ResAnimCurveData* targetAnimCurveData) NN_NOEXCEPT
    {
        nn::g3d::ResAnimCurve::CurveType curveType = static_cast<nn::g3d::ResAnimCurve::CurveType>(targetAnimCurveData->flag & nn::g3d::ResAnimCurve::Mask_Curve);
        switch (curveType)
        {
        case nn::g3d::ResAnimCurve::CurveType_Cubic: return targetAnimCurveData->keyCount * 4;
        case nn::g3d::ResAnimCurve::CurveType_Linear: return targetAnimCurveData->keyCount * 2;
        case nn::g3d::ResAnimCurve::CurveType_StepInt:
        case nn::g3d::ResAnimCurve::CurveType_StepBool:
        default:
            return targetAnimCurveData->keyCount;
        }
    }

    size_t GetFrameArraySize(const nn::g3d::ResAnimCurveData* pResAnimCurveData) NN_NOEXCEPT
    {
        nn::g3d::ResAnimCurve::FrameType frameType = static_cast<nn::g3d::ResAnimCurve::FrameType>(pResAnimCurveData->flag & nn::g3d::ResAnimCurve::Mask_Frame);
        switch (frameType)
        {
        case nn::g3d::ResAnimCurve::FrameType_Quant32: return sizeof(float) * pResAnimCurveData->keyCount;
        case nn::g3d::ResAnimCurve::FrameType_Quant16: return sizeof(uint16_t) * pResAnimCurveData->keyCount;
        case nn::g3d::ResAnimCurve::FrameType_Quant8: return sizeof(uint8_t) * pResAnimCurveData->keyCount;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    size_t GetKeyArraySize(const nn::g3d::ResAnimCurveData* pResAnimCurveData) NN_NOEXCEPT
    {
        nn::g3d::ResAnimCurve::KeyType keyType = static_cast<nn::g3d::ResAnimCurve::KeyType>(pResAnimCurveData->flag & nn::g3d::ResAnimCurve::Mask_Key);
        int keyValueCount = GetKeyValueCount(pResAnimCurveData);
        switch (keyType)
        {
        case nn::g3d::ResAnimCurve::KeyType_Quant32: return sizeof(float) * keyValueCount;
        case nn::g3d::ResAnimCurve::KeyType_Quant16: return sizeof(uint16_t) * keyValueCount;
        case nn::g3d::ResAnimCurve::KeyType_Quant8: return sizeof(uint8_t) * keyValueCount;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    // TODO: デバッグ用なので安定したら消す
    void PrintResAnimCurveData(const nn::g3d::ResAnimCurveData* targetAnimCurveData) NN_NOEXCEPT
    {
        {
            nn::g3d::ResAnimCurve::KeyType keyType = static_cast<nn::g3d::ResAnimCurve::KeyType>(targetAnimCurveData->flag & nn::g3d::ResAnimCurve::Mask_Key);
            int keyValueCount = GetKeyValueCount(targetAnimCurveData);
            switch (keyType)
            {
            case nn::g3d::ResAnimCurve::KeyType_Quant32:
                {
                    const float* ptr = static_cast<const float*>(targetAnimCurveData->pKeyArray.Get());
                    for (int i = 0; i < keyValueCount; ++i)
                    {
                        const float* pKey = &ptr[i];
                        NN_UNUSED(pKey);
                        NN_G3D_VIEWER_DEBUG_PRINT("key(32)[%d]: %f\n", i, *pKey);
                    }
                }
                break;
            case nn::g3d::ResAnimCurve::KeyType_Quant16:
                {
                    const int16_t* ptr = static_cast<const int16_t*>(targetAnimCurveData->pKeyArray.Get());
                    for (int i = 0; i < keyValueCount; ++i)
                    {
                        const int16_t* pKey = &ptr[i];
                        NN_UNUSED(pKey);
                        NN_G3D_VIEWER_DEBUG_PRINT("key(16)[%d]: %d\n", i, *pKey);
                    }
                }
                break;
            case nn::g3d::ResAnimCurve::KeyType_Quant8:
                {
                    const uint8_t* ptr = static_cast<const uint8_t*>(targetAnimCurveData->pKeyArray.Get());
                    for (int i = 0; i < keyValueCount; ++i)
                    {
                        const uint8_t* pKey = &ptr[i];
                        NN_UNUSED(pKey);
                        NN_G3D_VIEWER_DEBUG_PRINT("key(8)[%d]: %d\n", i, *pKey);
                    }
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }

        {
            nn::g3d::ResAnimCurve::FrameType frameType = static_cast<nn::g3d::ResAnimCurve::FrameType>(targetAnimCurveData->flag & nn::g3d::ResAnimCurve::Mask_Frame);
            switch (frameType)
            {
            case nn::g3d::ResAnimCurve::FrameType_Quant32:
                {
                    const nn::g3d::Frame<float>* ptr = static_cast<const nn::g3d::Frame<float>*>(targetAnimCurveData->pFrameArray.Get());
                    for (int i = 0; i < targetAnimCurveData->keyCount; ++i)
                    {
                        const nn::g3d::Frame<float>* pFrame = &ptr[i];
                        NN_UNUSED(pFrame);
                        NN_G3D_VIEWER_DEBUG_PRINT("frame(32)[%d]: %f\n", i, pFrame->frame);
                    }
                }
                break;
            case nn::g3d::ResAnimCurve::FrameType_Quant16:
                {
                    const nn::g3d::Frame<int16_t>* ptr = static_cast<const nn::g3d::Frame<int16_t>*>(targetAnimCurveData->pFrameArray.Get());
                    for (int i = 0; i < targetAnimCurveData->keyCount; ++i)
                    {
                        const nn::g3d::Frame<int16_t>* pFrame = &ptr[i];
                        NN_UNUSED(pFrame);
                        NN_G3D_VIEWER_DEBUG_PRINT("frame(16)[%d]: %d\n", i, pFrame->frame);
                    }
                }
                break;
            case nn::g3d::ResAnimCurve::FrameType_Quant8:
                {
                    const nn::g3d::Frame<uint8_t>* ptr = static_cast<const nn::g3d::Frame<uint8_t>*>(targetAnimCurveData->pFrameArray.Get());
                    for (int i = 0; i < targetAnimCurveData->keyCount; ++i)
                    {
                        const nn::g3d::Frame<uint8_t>* pFrame = &ptr[i];
                        NN_UNUSED(pFrame);
                        NN_G3D_VIEWER_DEBUG_PRINT("frame(8)[%d]: %d\n", i, pFrame->frame);
                    }
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }

    void CopyAndFlush(void* pDestination, const void* pSource, size_t size) NN_NOEXCEPT
    {
        memcpy(pDestination, pSource, size);
        nn::os::FlushDataCache(pDestination, size);
    }

    void UnregisterKey(void* data) NN_NOEXCEPT
    {
        ViewerKeyType key = ViewerKeyManager::GetInstance().FindKey(data);
        if (key == InvalidKey)
        {
            return;
        }

        ViewerKeyManager::GetInstance().Unregister(key);
    }

}}}} // namespace nn::g3d::viewer::detail




