﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "g3d_DynamicRingBuffer.h"
#include "g3d_SynchronizedObject.h"
#include "g3d_ScopedLock.h"

namespace nn { namespace g3d { namespace viewer { namespace detail {

template<typename T>
class SynchronizedDynamicRingBuffer : public SynchronizedObject
{
    NN_DISALLOW_COPY(SynchronizedDynamicRingBuffer<T>);

public:
    explicit SynchronizedDynamicRingBuffer(nn::g3d::viewer::detail::IAllocator* pAllocator)
        : m_DynamicRingBuffer(pAllocator)
    {
    }

    int GetCount() const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicRingBuffer.GetCount();
    }

    int GetCountMax() const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicRingBuffer.GetCountMax();
    }

    void Clear() NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        m_DynamicRingBuffer.Clear();
    }

    T* UnsafeGet(int index) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_DynamicRingBuffer.UnsafeGet(index);
    }

    const T* UnsafeGet(int index) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicRingBuffer.UnsafeGet(index);
    }

    T& operator[](int index) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_DynamicRingBuffer[index];
    }

    const T& operator[](int index) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicRingBuffer[index];
    }

    bool PushBack(const T& value) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_DynamicRingBuffer.PushBack(value);
    }

    bool PushFront(const T& value) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_DynamicRingBuffer.PushFront(value);
    }

    T PopFront() NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_DynamicRingBuffer.PopFront();
    }

    T PopBack() NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_DynamicRingBuffer.PopBack();
    }

    void EraseAt(int index) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_DynamicRingBuffer.EraseAt(index);
    }

    bool Resize(int count) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_DynamicRingBuffer.Resize(count);
    }

    const T* Find(const T* ptr, int(*compareCallback)(const T*, const T*)) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicRingBuffer.Find(ptr, compareCallback);
    }

    T* Find(const T* ptr, int(*compareCallback)(const T*, const T*)) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_DynamicRingBuffer.Find(ptr, compareCallback);
    }

    template<typename TKey>
    const T* Find(const TKey* key, int(*compareCallback)(const T* a, const TKey* b)) const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_DynamicRingBuffer.Find(key, compareCallback);
    }

    template<typename TKey>
    T* Find(const TKey* key, int(*compareCallback)(const T* a, const TKey* b)) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_DynamicRingBuffer.Find(key, compareCallback);
    }

private:
    DynamicRingBuffer<T> m_DynamicRingBuffer;
};

} } } } // namespace nn::g3d::viewer::detail
