﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "g3d_DynamicArray.h"
#include <nn/g3d/detail/g3d_Inlines.h>

namespace nn { namespace g3d { namespace viewer { namespace detail {

template<typename TKey, typename TValue>
class Map
{
public:
    static const int InvalidIndex = DynamicArray<TKey>::InvalidIndex;

    explicit Map(
        nn::g3d::viewer::detail::Allocator* pAllocator,
        TKey errorKey,
        TValue errorValue) NN_NOEXCEPT
        : m_KeyArray(pAllocator, nn::g3d::detail::Alignment_Default, errorKey)
        , m_ValueArray(pAllocator, nn::g3d::detail::Alignment_Default, errorValue)
        , m_ErrorKey(errorKey)
        , m_ErrorValue(errorValue)
    {
    }

    bool Register(TKey key, TValue value) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsKeyRegistered(key), "key %d has been already registered", key);

        bool success = m_KeyArray.PushBack(key);
        if (!success)
        {
            return false;
        }

        success = m_ValueArray.PushBack(value);
        if (!success)
        {
            return false;
        }

        return true;
    }

    // @brief 既存のキーに割り当てられている値を別の値で上書きします。
    bool UpdateValue(TKey key, TValue value) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsKeyRegistered(key), "key %d has not been registered yet", key);
        int index = FindIndexFromKey(key);
        NN_G3D_VIEWER_ASSERT(index != InvalidIndex);
        m_ValueArray[index] = value;
        return true;
    }

    bool Unregister(TKey key) NN_NOEXCEPT
    {
        int index = FindIndexFromKey(key);
        if (index == InvalidIndex)
        {
            return false;
        }

        m_KeyArray.EraseByIndex(index);
        m_ValueArray.EraseByIndex(index);
        return true;
    }

    TKey FindKey(TValue value) const NN_NOEXCEPT
    {
        int index = FindIndexFromValue(value);
        return index != InvalidIndex ? m_KeyArray[index] : m_ErrorKey;
    }

    TValue FindValue(TKey key) const NN_NOEXCEPT
    {
        int index = FindIndexFromKey(key);
        return index != InvalidIndex ? m_ValueArray[index] : m_ErrorValue;
    }

    int GetCount() const NN_NOEXCEPT
    {
        return m_KeyArray.GetCount();
    }

    TKey GetKey(int index) const NN_NOEXCEPT
    {
        return m_KeyArray[index];
    }

    TValue GetValue(int index) const NN_NOEXCEPT
    {
        return m_ValueArray[index];
    }

    bool IsKeyRegistered(TKey key) const NN_NOEXCEPT
    {
        return FindIndexFromKey(key) != InvalidIndex;
    }

    bool IsValueRegistered(TValue value) const NN_NOEXCEPT
    {
        return FindIndexFromValue(value) != InvalidIndex;
    }

private:

    int FindIndexFromKey(TKey key) const NN_NOEXCEPT
    {
        for (int index = 0, indexCount = m_KeyArray.GetCount(); index < indexCount; ++index)
        {
            if (IsEqual(key, m_KeyArray[index]))
            {
                return index;
            }
        }
        return InvalidIndex;
    }

    int FindIndexFromValue(TValue value) const NN_NOEXCEPT
    {
        for (int index = 0, indexCount = m_ValueArray.GetCount(); index < indexCount; ++index)
        {
            if (IsEqual(value, m_ValueArray[index]))
            {
                return index;
            }
        }
        return InvalidIndex;
    }

    template<typename Type>
    bool IsEqual(Type t1, Type t2) const NN_NOEXCEPT
    {
        return t1 == t2;
    }

    // 文字列比較
    bool IsEqual(const char* str1, const char* str2) const NN_NOEXCEPT
    {
        return strcmp(str1, str2) == 0;
    }

private:
    DynamicArray<TKey> m_KeyArray;
    DynamicArray<TValue> m_ValueArray;
    TKey m_ErrorKey;
    TValue m_ErrorValue;
};

}}}} // namespace nn::g3d::viewer::detail


