﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../g3d_ViewerDetailDefine.h"
#include "../g3d_Allocator.h"

namespace nn { namespace g3d { namespace viewer {

namespace detail {

template<typename T>
class Iter
{
public:
    Iter(T* pDataArray, int count, int currentIndex, T errorValue) NN_NOEXCEPT
        : m_pData(pDataArray)
        , m_CurrentIndex(currentIndex)
        , m_Count(count)
        , m_ErrorValue(errorValue)
    {
    }

    explicit Iter(T errorValue) NN_NOEXCEPT
        : m_pData(nullptr)
        , m_CurrentIndex(-1)
        , m_Count(0)
        , m_ErrorValue(errorValue)
    {
    }

    Iter(const Iter& rhs) NN_NOEXCEPT
        : m_pData(rhs.m_pData)
        , m_CurrentIndex(rhs.m_CurrentIndex)
        , m_Count(rhs.m_Count)
        , m_ErrorValue(rhs.m_ErrorValue)
    {
    }

    bool operator!=(const Iter& iter) const NN_NOEXCEPT
    {
        return m_CurrentIndex != iter.m_CurrentIndex;
    }

    bool operator==(const Iter& iter) const NN_NOEXCEPT
    {
        return !(*this != iter);
    }

    Iter& operator++() NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT(m_CurrentIndex < m_Count);
        ++m_CurrentIndex;
        if (!IsCurrentIndexValid())
        {
            SetToEnd();
        }
        return *this;
    }

    Iter operator++(int) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT(m_CurrentIndex < m_Count);
        Iter result = *this;
        ++m_CurrentIndex;
        if (!IsCurrentIndexValid())
        {
            SetToEnd();
        }
        return result;
    }

    Iter operator+(int index) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT(m_CurrentIndex + index < m_Count);
        Iter result = *this;
        result.m_CurrentIndex += index;
        return result;
    }

    Iter& operator=(const Iter& rhs) NN_NOEXCEPT
    {
        m_pData = rhs.m_pData;
        m_CurrentIndex = rhs.m_CurrentIndex;
        m_Count = rhs.m_Count;
        m_ErrorValue = rhs.m_ErrorValue;
        return *this;
    }

    T& operator*() NN_NOEXCEPT
    {
        if (IsEnd())
        {
            return m_ErrorValue;
        }

        return m_pData[m_CurrentIndex];
    }

    const T& operator*() const NN_NOEXCEPT
    {
        if (IsEnd())
        {
            return m_ErrorValue;
        }

        return m_pData[m_CurrentIndex];
    }

    int GetIndex() const NN_NOEXCEPT
    {
        return m_CurrentIndex;
    }

    int GetCount() const NN_NOEXCEPT
    {
        return m_Count;
    }

private:
    bool IsEnd() const NN_NOEXCEPT
    {
        return m_CurrentIndex == -1;
    }

    void SetToEnd() NN_NOEXCEPT
    {
        m_CurrentIndex = -1;
    }

    bool IsCurrentIndexValid() NN_NOEXCEPT
    {
        return (0 <= m_CurrentIndex) && (m_CurrentIndex < m_Count);
    }

    T* m_pData;
    int m_CurrentIndex;
    int m_Count;
    T m_ErrorValue;
};

template<typename T>
class DynamicArray
{
public:
    typedef int(*CompareCallback)(const void* pValue, const void* pKey);

    static const int InvalidIndex = -1;

    explicit DynamicArray(
        nn::g3d::viewer::detail::IAllocator* pAllocator,
        size_t alignment,
        T errorValue) NN_NOEXCEPT
        : m_pAllocator(pAllocator)
        , m_Alignment(alignment)
        , m_ArrayLength(0)
        , m_ArrayLengthMax(DEFAULT_ARRAY_SIZE)
        , m_Array(nullptr)
        , m_ErrorValue(errorValue)
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pAllocator);
        NN_G3D_VIEWER_ASSERT(IsPowerOfTwo(alignment));
        size_t elemSize = sizeof(T);
        m_Array = static_cast<T*>(m_pAllocator->Allocate(elemSize * m_ArrayLengthMax, m_Alignment, AllocateType_DynamicBuffer));
        NN_G3D_VIEWER_ASSERT_NOT_NULL(m_Array);
    }

    ~DynamicArray() NN_NOEXCEPT
    {
        Destroy();
    }

    void Destroy() NN_NOEXCEPT
    {
        if (m_pAllocator == nullptr)
        {
            return;
        }

        m_pAllocator->Free(m_Array);
        m_Array = nullptr;
        m_ArrayLength = 0;
        m_pAllocator = nullptr;
    }

    int GetCount() const NN_NOEXCEPT
    {
        return m_ArrayLength;
    }

    void Clear() NN_NOEXCEPT
    {
        m_ArrayLength = 0;
    }

    bool PushBack(T elem) NN_NOEXCEPT
    {
        if (m_ArrayLength < m_ArrayLengthMax)
        {
            m_Array[m_ArrayLength] = elem;
            ++m_ArrayLength;
            return true;
        }

        if (Resize(m_ArrayLengthMax + 1))
        {
            m_Array[m_ArrayLength] = elem;
            ++m_ArrayLength;
            return true;
        }

        return false;
    }

    T PopBack() NN_NOEXCEPT
    {
        if (0 < m_ArrayLength)
        {
            --m_ArrayLength;
            return m_Array[m_ArrayLength];
        }
        else
        {
            return m_ErrorValue;
        }
    }

    int IndexOf(T target) const NN_NOEXCEPT
    {
        int index = 0;
        for (const T* iter = m_Array, *end = m_Array + m_ArrayLength; iter != end; ++iter, ++index)
        {
            if (*iter == target)
            {
                return index;
            }
        }

        return InvalidIndex;
    }

    template<typename TKey>
    T Find(const TKey* key, int(*compareCallback)(const T a, const TKey* b)) const NN_NOEXCEPT
    {
        for (int i = 0; i < m_ArrayLength; ++i)
        {
            if (compareCallback(m_Array[i], key) == 0)
            {
                return m_Array[i];
            }
        }

        return m_ErrorValue;
    }

    void EraseByIndex(int pos) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT(pos >= 0);
        NN_G3D_VIEWER_ASSERT((pos + 1) <= m_ArrayLength);

        int movePos = pos + 1;
        if (movePos < m_ArrayLength)
        {
            CopyOverlap(&m_Array[pos], &m_Array[movePos], (m_ArrayLength - movePos) * sizeof(T));
        }
        m_ArrayLength = m_ArrayLength - 1;
    }

    bool EraseByTarget(T target) NN_NOEXCEPT
    {
        int index = IndexOf(target);
        if (index == InvalidIndex)
        {
            return false;
        }

        EraseByIndex(index);
        return true;
    }

    T* GetData() NN_NOEXCEPT
    {
        return m_Array;
    }

    const T* GetData() const NN_NOEXCEPT
    {
        return m_Array;
    }

    T& operator[] (int index) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(index, m_ArrayLength);
        return m_Array[index];
    }

    const T& operator[] (int index) const NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(index, m_ArrayLength);
        return m_Array[index];
    }

    bool Contains(T target) const NN_NOEXCEPT
    {
        return IndexOf(target) != InvalidIndex;
    }

    Iter<const T> Begin() const NN_NOEXCEPT
    {
        if (GetCount() == 0)
        {
            return End();
        }
        else
        {
            Iter<const T> iter(reinterpret_cast<const T*>(m_Array), GetCount(), 0, m_ErrorValue);
            return iter;
        }
    }

    Iter<const T> End() const NN_NOEXCEPT
    {
        Iter<const T> iter(m_ErrorValue);
        return iter;
    }

    Iter<T> Begin() NN_NOEXCEPT
    {
        if (GetCount() == 0)
        {
            return End();
        }
        else
        {
            Iter<T> iter(reinterpret_cast<T*>(m_Array), GetCount(), 0, m_ErrorValue);
            return iter;
        }
    }

    Iter<T> End() NN_NOEXCEPT
    {
        Iter<T> iter(m_ErrorValue);
        return iter;
    }

    Iter<T> Erase(const Iter<T>& iter) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT(iter.GetCount() == m_ArrayLength);
        EraseByIndex(iter.GetIndex());
        int prevIndex = iter.GetIndex() - 1;
        Iter<T> result(reinterpret_cast<T*>(m_Array), GetCount(), prevIndex, m_ErrorValue);
        return result;
    }

    Allocator* GetAllocator()
    {
        return m_pAllocator;
    }

    T Front() const NN_NOEXCEPT
    {
        if (GetCount() == 0)
        {
            return m_ErrorValue;
        }

        return m_Array[0];
    }

    T Back() const NN_NOEXCEPT
    {
        if (GetCount() == 0)
        {
            return m_ErrorValue;
        }

        return m_Array[GetCount() - 1];
    }

    bool Resize(size_t size) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(m_pAllocator);

        size_t currentSize = static_cast<size_t>(m_ArrayLengthMax);
        if (currentSize >= size)
        {
            return true;
        }

        int bufferCount = m_ArrayLengthMax << 1;

        T* newPtrArray = static_cast<T*>(m_pAllocator->Allocate(sizeof(T) * bufferCount, m_Alignment, AllocateType_DynamicBuffer));
        if (newPtrArray == nullptr)
        {
            return false;
        }

        m_ArrayLengthMax = bufferCount;

        for (int i = 0; i < m_ArrayLength; ++i)
        {
            newPtrArray[i] = m_Array[i];
        }

        m_pAllocator->Free(m_Array);
        m_Array = newPtrArray;
        return true;
    }

private:
    void* CopyOverlap(void* dst, const void* src, size_t size) NN_NOEXCEPT
    {
        return std::memmove(dst, src, size);
    }

    inline bool IsPowerOfTwo(size_t size) NN_NOEXCEPT
    {
        return ((size - 1) & size) == 0 && size != 0;
    }

private:
    static const int DEFAULT_ARRAY_SIZE = 8;

    nn::g3d::viewer::detail::IAllocator* m_pAllocator;
    size_t m_Alignment;
    int m_ArrayLength;
    int m_ArrayLengthMax;
    T* m_Array;
    T m_ErrorValue;
};

} // namespace detail

}}} // namespace nn::g3d::edit::ut


