﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/viewer/g3d_ViewerDefine.h>



#include "../util/g3d_DynamicPtrArray.h"
#include "../util/g3d_DynamicLengthString.h"
#include "../util/g3d_SynchronizedEditWorkBuffer.h"
#include "../g3d_DeviceDependentObj.h"

namespace nn { namespace g3d {

struct ResShaderProgramData;
class ResShaderProgram;
class ResShadingModel;
class ResShaderArchive;

namespace viewer {
namespace detail {

class Allocator;
class EditSocketBase;
class EditShadingModel;

/**
    @briefprivate 編集対象シェーダプログラムクラスです。
 */
class EditShaderProgram : public DeviceDependentObj
{
    friend class EditShadingModel;
    friend void UpdateProgram(nn::g3d::ResShadingModel*, int);

public:
    explicit EditShaderProgram(
        nn::gfx::Device* pDevice,
        Allocator* allocator,
        ViewerKeyType shaderArchiveKey,
        int shadingModelIndex,
        int shaderProgramIndex,
        ResShaderProgram* resShaderProgram) NN_NOEXCEPT;

    void ResetToOriginal() NN_NOEXCEPT;
    void Destroy() NN_NOEXCEPT;

    // 変更されたシェーダプログラムを3DEditor に送信します。
    bool SendModifiedShaderProgram(EditSocketBase* socket) NN_NOEXCEPT;
    bool MakeModifiedShaderProgramPacket() NN_NOEXCEPT;

    bool PushOptionInfo(const char* option, const char* choice) NN_NOEXCEPT;

    bool IsPacketEnabled() const NN_NOEXCEPT
    {
        return m_PacketEnableFlag;
    }

    void UpdateShaderProgram(ResShaderArchive* resShaderArchive) NN_NOEXCEPT;

    void ClearOptionInfo() NN_NOEXCEPT;

    bool GetSendInfoCallbackFlag() const NN_NOEXCEPT
    {
        return m_SendInfoCallbackFlag;
    }
    void SetSendInfoCallbackFlag(bool value) NN_NOEXCEPT
    {
        m_SendInfoCallbackFlag = value;
    }

    Allocator* GetAllocator() NN_NOEXCEPT
    {
        return m_pAllocator;
    }

    const Allocator* GetAllocator() const NN_NOEXCEPT
    {
        return m_pAllocator;
    }

    ViewerKeyType GetShaderArchiveKey() const NN_NOEXCEPT
    {
        return m_ShaderArchiveKey;
    }

    int GetShadingModelIndex() const NN_NOEXCEPT
    {
        return m_ShadingModelIndex;
    }

    int GetShaderProgramIndex() const NN_NOEXCEPT
    {
        return m_ShaderProgramIndex;
    }

    ResShaderArchive* GetUpdateResShaderArchive() NN_NOEXCEPT
    {
        return m_pUpdateResShaderArchive;
    }

    const ResShaderArchive* GetUpdateResShaderArchive() const NN_NOEXCEPT
    {
        return m_pUpdateResShaderArchive;
    }

private:
    size_t CalculateWorkBufferSize() NN_NOEXCEPT;
    void MakeModifiedShaderProgramBuffer() NN_NOEXCEPT;

    class OptionInfo
    {
        friend class EditShaderProgram;
    public:
        explicit OptionInfo(Allocator* allocator) NN_NOEXCEPT
            : m_pAllocator(allocator)
            , m_Option(allocator)
            , m_Choice(allocator)
        {
            NN_G3D_VIEWER_ASSERT_NOT_NULL(allocator);
        }

        ~OptionInfo() NN_NOEXCEPT
        {
            Clear();
            m_pAllocator = nullptr;
        }

        bool Setup(const char* option, const char* choice) NN_NOEXCEPT;
        void Clear() NN_NOEXCEPT
        {
            m_Option.Clear();
            m_Choice.Clear();
        }
    private:
        Allocator* m_pAllocator;
        DynamicLengthString m_Option;
        DynamicLengthString m_Choice;
        size_t m_OptionOffset;
        size_t m_ChoiceOffset;
        size_t m_OptionBinarySize;
        size_t m_ChoiceBinarySize;
    };

private:
    enum
    {
        NUM_STAGE = 4
    };

    ResShaderProgram*       m_pTargetResShaderProgram;
    ResShaderArchive*       m_pUpdateResShaderArchive;

    uint16_t     m_OriginalFlag;
    uint16_t      m_OriginalSamplerCount;
    uint16_t      m_OriginalUniformBlockCount;
    uint16_t m_OriginalShaderStorageBlockCount;
    uint32_t     m_OriginalAttribuActiveFlag;
    void*   m_OriginalOffsetSamplerTable;
    void*   m_OriginalOffsetUniformBlockTable;
    int32_t* m_pOriginalShaderStorageBlockTable;

    void*   m_OriginalOffsetShader;

    SynchronizedEditWorkBuffer m_WorkBuffer;
    DynamicPtrArray<OptionInfo> m_OptionInfos;
    size_t m_ModifiedShaderProgramBufferSize;
    ViewerKeyType m_ShaderArchiveKey;
    int m_ShadingModelIndex;
    int m_ShaderProgramIndex;
    volatile bool m_PacketEnableFlag;
    bool m_SendInfoCallbackFlag;

    NN_DISALLOW_COPY(EditShaderProgram);
};

}} // nn::g3d::viewer::detail
}} // namespace nw::g3d


