﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditSkeletonObj.h"


#include "../util/g3d_ViewerUtility.h"
#include "../util/g3d_ScopedAllocator.h"
#include "../g3d_Allocator.h"
#include <nn/g3d/g3d_ResFile.h>
#include <nn/g3d/g3d_ModelObj.h>
#include <nn/diag/text/diag_SdkTextG3dviewer.h>

using namespace nn::g3d;
using namespace nn::g3d::viewer;

namespace nn { namespace g3d { namespace viewer { namespace detail {

EditSkeletonObj::EditSkeletonObj(
    nn::g3d::ModelObj* ownerModelObj, nn::g3d::SkeletonObj* pSkeletonObj,
    nn::gfx::Device* pDevice, Allocator* pAllocator) NN_NOEXCEPT
    : DeviceDependentObj(pDevice, pAllocator)
    , m_pOwnerModelObj(nullptr)
    , m_pSkeletonObj(nullptr)
    , m_pOriginalBuffer(nullptr)
    , m_OriginalBufferSize(0)
    , m_pOriginalBlockMemoryPool(nullptr)
    , m_OriginalBlockMemoryPoolOffset(0)
    , m_OriginalBlockBufferSize(0)
    , m_OriginalBufferingCount(0)
    , m_WorkBuffer(pAllocator, SkeletonObj::Alignment_Buffer, AllocateType_DynamicBuffer)
    , m_BlockBufferObjectManager(UniformBlockBufferingCount, pAllocator)
{
    NN_SDK_REQUIRES_NOT_NULL(pAllocator);
    NN_SDK_REQUIRES_NOT_NULL(pDevice);

    NN_SDK_REQUIRES_NOT_NULL(ownerModelObj);
    NN_G3D_VIEWER_REQUIRES_NOT_NULL_DETAIL(pSkeletonObj, ownerModelObj->GetResource()->GetName());
    nn::g3d::ResSkeleton* resSkeleton = const_cast<nn::g3d::ResSkeleton*>(pSkeletonObj->GetResource());
    NN_G3D_VIEWER_REQUIRES_NOT_NULL_DETAIL(resSkeleton, ownerModelObj->GetResource()->GetName());
    NN_SDK_REQUIRES(
        pSkeletonObj->CalculateBlockBufferSize(m_pDevice) == 0 || pSkeletonObj->IsBlockBufferValid(),
        NN_TEXT_G3DVIEWER("\"%s\" のユニフォームブロックが構築されていません"), ownerModelObj->GetResource()->GetName());

    m_pOwnerModelObj = ownerModelObj;
    m_pSkeletonObj = pSkeletonObj;
    m_pOriginalResSkeleton = resSkeleton;
    m_OriginalBufferingCount = pSkeletonObj->GetBufferingCount();

    m_pOriginalBuffer = m_pSkeletonObj->GetBufferPtr();

    SkeletonObj::Builder builder(resSkeleton);
    builder.BufferingCount(m_OriginalBufferingCount);
    builder.CalculateMemorySize();
    m_OriginalBufferSize = builder.GetWorkMemorySize();

    m_OriginalBlockBufferSize = m_pSkeletonObj->CalculateBlockBufferSize(m_pDevice);
    m_pOriginalBlockMemoryPool = m_pSkeletonObj->GetMemoryPoolPtr();
    m_OriginalBlockMemoryPoolOffset = m_pSkeletonObj->GetMemoryPoolOffset();
    m_pOriginalBlockBufferArray = m_pSkeletonObj->GetMtxBlockArray();
}

EditSkeletonObj::~EditSkeletonObj() NN_NOEXCEPT
{
    for (int bufferIndex = 0, bufferCount = m_BlockBufferObjectManager.GetBufferCount(); bufferIndex < bufferCount; ++bufferIndex)
    {
        if (m_BlockBufferObjectManager.GetBufferSize(bufferIndex) > 0)
        {
            nn::gfx::Buffer* pMaterialBlockArray = m_BlockBufferObjectManager.GetBuffer<nn::gfx::Buffer>(bufferIndex);
            FinalizeBuffers(m_pDevice, pMaterialBlockArray, m_OriginalBufferingCount);
        }
    }
}

void
EditSkeletonObj::ClearBuffers() NN_NOEXCEPT
{
    m_WorkBuffer.Clear();
}

void
EditSkeletonObj::ResetToOriginal() NN_NOEXCEPT
{
    if (IsOriginal())
    {
        return;
    }

    SkeletonObj::Builder builder(m_pOriginalResSkeleton);
    builder.BufferingCount(m_OriginalBufferingCount);

    void* userPtr = m_pSkeletonObj->GetUserPtr();// UserPtr を保持するために一時退避

    if (m_OriginalBlockBufferSize > 0)
    {
        m_pSkeletonObj->SetMtxBlockArray(m_pOriginalBlockBufferArray);
        m_pSkeletonObj->CleanupBlockBuffer(m_pDevice);
    }

    builder.CalculateMemorySize();
    NN_G3D_VIEWER_ASSERT(m_OriginalBufferSize == builder.GetWorkMemorySize());
    bool success = builder.Build(m_pSkeletonObj, m_pOriginalBuffer, m_OriginalBufferSize);
    NN_G3D_VIEWER_ASSERT(success);
    m_pSkeletonObj->SetUserPtr(userPtr);

    if (m_OriginalBlockBufferSize > 0)
    {
        m_pSkeletonObj->SetupBlockBuffer(m_pDevice, m_pOriginalBlockMemoryPool, m_OriginalBlockMemoryPoolOffset, m_OriginalBlockBufferSize);
    }
}

size_t
EditSkeletonObj::CalculateBlockBufferSize(const nn::g3d::ResSkeleton* pResSkeleton) const NN_NOEXCEPT
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize(sizeof(nn::util::FloatColumnMajor4x3) * pResSkeleton->GetMtxCount());
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
    size_t alignment = nn::gfx::Buffer::GetBufferAlignment(m_pDevice, info);

    return nn::util::align_up(sizeof(nn::util::FloatColumnMajor4x3) * pResSkeleton->GetMtxCount(), alignment) * m_OriginalBufferingCount;
}

size_t
EditSkeletonObj::GetBlockBufferAlignment(const nn::g3d::ResSkeleton* pResSkeleton) const NN_NOEXCEPT
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize(sizeof(nn::util::FloatColumnMajor4x3) * pResSkeleton->GetMtxCount());
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);

    return nn::gfx::Buffer::GetBufferAlignment(m_pDevice, info);
}

void
EditSkeletonObj::SetupSkeletonInstance(
    nn::g3d::ResSkeleton* pResSkeleton,
    nn::gfx::MemoryPool* pMemoryPool,
    ptrdiff_t memoryPoolOffset,
    size_t memoryPoolSize) NN_NOEXCEPT
{
    SkeletonObj::Builder builder(pResSkeleton);
    builder.BufferingCount(m_pSkeletonObj->GetBufferingCount());
    bool success;

    builder.CalculateMemorySize();
    size_t size = builder.GetWorkMemorySize();
    success = m_WorkBuffer.Resize(size);

    NN_G3D_VIEWER_ASSERT_DETAIL(
        success,
        "Memory allocation failed, %s\n",
        NN_G3D_VIEWER_RES_NAME(m_pOwnerModelObj->GetResource(), GetName()));

    {
        // UserPtr を保持するために一時退避
        void* userPtr = m_pSkeletonObj->GetUserPtr();

        // ワールド行列が既に計算済みである可能性があるので、再初期化後に引き継ぐために一時退避
        size_t worldMtxArraySize = sizeof(nn::util::Matrix4x3fType) * m_pSkeletonObj->GetBoneCount();
        nn::g3d::viewer::detail::ScopedAllocator scopedAllocator(
            *m_pAllocator,
            worldMtxArraySize, nn::DefaultAlignment, AllocateType_Other);
        nn::util::Matrix4x3fType* pWorldMtxArray = scopedAllocator.GetAllocatedBuffer<nn::util::Matrix4x3fType>();
        memcpy(pWorldMtxArray, m_pSkeletonObj->GetWorldMtxArray(), worldMtxArraySize);

        success = builder.Build(m_pSkeletonObj, m_WorkBuffer.GetWorkBufferPtr(), m_WorkBuffer.GetSize());
        NN_G3D_VIEWER_ASSERT(success);

        memcpy(m_pSkeletonObj->GetWorldMtxArray(), pWorldMtxArray, worldMtxArraySize);

        m_pSkeletonObj->SetUserPtr(userPtr);
    }

    {
        NN_G3D_VIEWER_ASSERT(memoryPoolSize == m_pSkeletonObj->CalculateBlockBufferSize(m_pDevice));

        // スキニングされない場合は 0 になる
        if (memoryPoolSize > 0)
        {
            m_BlockBufferObjectManager.ResizeCurrentBuffer(sizeof(nn::gfx::Buffer) * m_OriginalBufferingCount);
            m_pSkeletonObj->SetMtxBlockArray(new (m_BlockBufferObjectManager.GetCurrentBuffer()) nn::gfx::Buffer());

            m_pSkeletonObj->SetupBlockBuffer(m_pDevice, pMemoryPool, memoryPoolOffset, memoryPoolSize);
        }
    }
}

void EditSkeletonObj::SwitchBlockBuffer() NN_NOEXCEPT
{
    m_BlockBufferObjectManager.SwitchCurrentBuffer();
    if (m_BlockBufferObjectManager.GetCurrentBufferSize() > 0)
    {
        // 過去に使っていたバッファーを使い回すので一度終了処理する
        nn::gfx::Buffer* pOldBlockArray = m_BlockBufferObjectManager.GetCurrentBuffer<nn::gfx::Buffer>();
        FinalizeBuffers(m_pDevice, pOldBlockArray, m_OriginalBufferingCount);
    }
}

}}}}

