﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/viewer/g3d_ViewerDefine.h>



#include "../util/g3d_EditWorkBufferSwitcher.h"
#include "g3d_EditBlinkingObj.h"
#include <nn/gfx/gfx_MemoryPool.h>

#include <nn/gfx/util/gfx_MemoryPoolAllocator.h>
#include <nn/g3d/g3d_ShapeObj.h>
#include "../util/g3d_Map.h"

namespace nn { namespace g3d {

class ModelObj;
class ShapeObj;
class ResShape;

namespace viewer {

namespace detail {

class Allocator;

// TODO: 単独のヘッダーに分離する
/**
   @briefprivate WorkBlock の管理クラスです。
 */
class WorkBlockManager : nn::g3d::viewer::detail::DeviceDependentObj
{
public:
    WorkBlockManager(nn::gfx::Device* pDevice, nn::g3d::viewer::detail::Allocator* pAllocator) NN_NOEXCEPT
        : DeviceDependentObj(pDevice, pAllocator)
        , m_WorkBlockBuffer(pAllocator, Alignment_Default, AllocateType_MemoryPool)
    {
    }

    virtual ~WorkBlockManager() NN_NOEXCEPT
    {
        Clear();
    }

    void Clear() NN_NOEXCEPT
    {
        if (m_WorkBlockAllocator.IsInitialized()) {
            m_WorkBlockAllocator.Finalize();
        }
        if (nn::gfx::IsInitialized(m_WorkBlockMemoryPool)) {
            m_WorkBlockMemoryPool.Finalize(m_pDevice);
        }
        m_WorkBlockBuffer.Clear();
    }

    size_t GetBufferSize() const NN_NOEXCEPT
    {
        if (!m_WorkBlockAllocator.IsInitialized())
        {
            return 0;
        }

        return m_WorkBlockAllocator.GetSize();
    }

    bool ResizeBlockBuffer(size_t blockBufferSize) NN_NOEXCEPT
    {
        if (blockBufferSize == 0)
        {
            return false;
        }
        Clear();

        nn::gfx::Buffer::InfoType bufferInfo;
        bufferInfo.SetDefault();
        bufferInfo.SetSize(blockBufferSize);
        bufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
        size_t blockBufferAlignment = nn::gfx::Buffer::GetBufferAlignment(m_pDevice, bufferInfo);

        nn::gfx::MemoryPool::InfoType memoryPoolInfo;
        memoryPoolInfo.SetDefault();
        size_t memoryPoolAlignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(m_pDevice, memoryPoolInfo);
        size_t memoryPoolSizeAlignment = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(m_pDevice, memoryPoolInfo);

        size_t memoryPoolBufferSize = nn::util::align_up<size_t>(blockBufferSize, memoryPoolSizeAlignment);
        size_t memoryPoolBufferAlignment = nn::util::align_up<size_t>(blockBufferAlignment, memoryPoolAlignment);
        bool result = m_WorkBlockBuffer.Resize(memoryPoolBufferSize, memoryPoolBufferAlignment);
        if (!result)
        {
            return false;
        }

        memoryPoolInfo.SetPoolMemory(m_WorkBlockBuffer.GetWorkBufferPtr(), m_WorkBlockBuffer.GetSize());
        m_WorkBlockMemoryPool.Initialize(m_pDevice, memoryPoolInfo);

        m_WorkBlockAllocator.Initialize(
            AllocateFunctionForWorkBlockAllocator,
            this,
            FreeFunctionForWorkBlockAllocator,
            this,
            &m_WorkBlockMemoryPool,
            0,
            memoryPoolBufferSize,
            nn::gfx::util::MemoryPoolAllocator::AlignmentMax,
            true);

        return true;
    }

    ptrdiff_t AllocateMemoryPool(size_t size, size_t alignment) NN_NOEXCEPT
    {
        return m_WorkBlockAllocator.Allocate(size, alignment);
    }

    void FreeMemoryPool(ptrdiff_t memoryPoolOffset) NN_NOEXCEPT
    {
        if (memoryPoolOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset) {
            m_WorkBlockAllocator.Free(memoryPoolOffset);
        }
    }

    nn::gfx::MemoryPool* GetMemoryPool() NN_NOEXCEPT
    {
        return &m_WorkBlockMemoryPool;
    }

    nn::gfx::util::MemoryPoolAllocator* GetMemoryPoolAllocator() NN_NOEXCEPT
    {
        return &m_WorkBlockAllocator;
    }

private:
    static void* AllocateFunctionForWorkBlockAllocator(size_t size, void* pUserData)
    {
        WorkBlockManager* pWorkBlockManager = reinterpret_cast<WorkBlockManager*>(pUserData);
        return pWorkBlockManager->m_pAllocator->Allocate(size, nn::DefaultAlignment, AllocateType_MemoryPool);
    }

    static void FreeFunctionForWorkBlockAllocator(void* ptr, void* pUserData)
    {
        WorkBlockManager* pWorkBlockManager = reinterpret_cast<WorkBlockManager*>(pUserData);
        pWorkBlockManager->m_pAllocator->Free(ptr);
    }

private:
    nn::g3d::viewer::detail::EditWorkBuffer m_WorkBlockBuffer;
    nn::gfx::MemoryPool m_WorkBlockMemoryPool;
    nn::gfx::util::MemoryPoolAllocator m_WorkBlockAllocator;
};

class MemoryPoolManager
{
public:
    MemoryPoolManager(int bufferingCount, nn::gfx::Device* pDevice, Allocator* pAllocator) NN_NOEXCEPT
        : m_BlockManagerSwitcher(pAllocator)
        , m_pAllocator(pAllocator)
    {
        for (int poolManagerIndex = 0; poolManagerIndex < bufferingCount; ++poolManagerIndex)
        {
            void* buffer = m_pAllocator->Allocate(sizeof(WorkBlockManager), nn::DefaultAlignment, AllocateType_Other);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(buffer);
            m_BlockManagerSwitcher.AppendBuffer(new (buffer) WorkBlockManager(pDevice, pAllocator));
        }
    }

    virtual ~MemoryPoolManager() NN_NOEXCEPT
    {
        for (int bufferIndex = 0, bufferCount = m_BlockManagerSwitcher.GetBufferCount(); bufferIndex < bufferCount; ++bufferIndex)
        {
            WorkBlockManager* buffer = m_BlockManagerSwitcher.GetBuffer(bufferIndex);
            buffer->~WorkBlockManager();
            m_pAllocator->Free(buffer);
        }
    }

    bool ResizeMemoryPoolBuffer(size_t size) NN_NOEXCEPT
    {
        return m_BlockManagerSwitcher.GetCurrentBuffer()->ResizeBlockBuffer(size);
    }

    ptrdiff_t AllocateMemoryPool(size_t size, size_t alignment) NN_NOEXCEPT
    {
        return m_BlockManagerSwitcher.GetCurrentBuffer()->AllocateMemoryPool(size, alignment);
    }

    nn::gfx::MemoryPool* GetMemoryPool() NN_NOEXCEPT
    {
        return m_BlockManagerSwitcher.GetCurrentBuffer()->GetMemoryPool();
    }

    void SwitchCurrentBuffer() NN_NOEXCEPT
    {
        m_BlockManagerSwitcher.SwitchCurrentBuffer();
    }

    size_t GetAllocatedMemoryPoolBufferSize() const NN_NOEXCEPT
    {
        return m_BlockManagerSwitcher.GetCurrentBuffer()->GetBufferSize();
    }

private:
    BufferSwitcher<WorkBlockManager> m_BlockManagerSwitcher;
    Allocator* m_pAllocator;
};

/**
    @briefprivate 編集対象シェイプクラスです。
*/
class EditShapeObj : public BlinkingObj
{
public:
    explicit EditShapeObj(
        nn::gfx::Device* pDevice,
        Allocator* pAllocator,
        int index,
        nn::g3d::ModelObj* pOwnerModelObj,
        nn::g3d::ShapeObj* pShapeObj) NN_NOEXCEPT;
    ~EditShapeObj() NN_NOEXCEPT;

    void ResetToOriginal() NN_NOEXCEPT;

    void SetLodLevel(int level) NN_NOEXCEPT;

    void ResetLodLevel() NN_NOEXCEPT;

    void SetViewDependent(bool depends) NN_NOEXCEPT
    {
        m_ViewDependent = depends;
    }

    bool ReinitBuffer(nn::g3d::ResShape* pResShape) NN_NOEXCEPT;
    size_t CalculateBlockBufferSize(bool isViewDependent, const ResShape* pResShape) const NN_NOEXCEPT;
    size_t GetBlockBufferAlignment() const NN_NOEXCEPT;
    void SetupShapeInstance(
        nn::g3d::ResShape* pResShape,
        nn::gfx::MemoryPool* pMemoryPool,
        ptrdiff_t memoryPoolOffset,
        size_t memoryPoolSize) NN_NOEXCEPT;

    int CalculateBlockBufferArrayLength(bool isViewDependent) const NN_NOEXCEPT
    {
        int blockCount = isViewDependent ? m_OriginalViewCount : 1;
        return blockCount * m_OriginalBufferingCount;
    }

    int CalculateDynamicVertexBufferCount() const NN_NOEXCEPT;

    void SwitchBlockBuffer() NN_NOEXCEPT;

private:
    virtual void Show() NN_NOEXCEPT;
    virtual void Hide() NN_NOEXCEPT;
    virtual bool IsVisibleByDefault() const NN_NOEXCEPT;

    void UpdateLodLevel(nn::g3d::ResShape* pResShape) NN_NOEXCEPT;
    void InitializeShapeObj(nn::g3d::ResShape* pResShape,
        bool isViewDependent,
        void* pWorkBuffer,
        size_t workBufferSize) NN_NOEXCEPT;

    // @brief 現在のシェイプUBOの値を一時的に保存します
    void SaveShapeBlockData() NN_NOEXCEPT;

    // @brief 一時的に保存したシェイプUBOの値を現在のシェイプUBOにロードします
    void LoadShapeBlockData() NN_NOEXCEPT;

    void ReplaceBlockBuffer() NN_NOEXCEPT;

    void WriteDynamicVertexBufferIndexFromOriginal(ResVertex* pResVertex) NN_NOEXCEPT;

    enum
    {
        ForceMeshLodDisabled = -1
    };

    int                     m_Index;
    nn::g3d::ModelObj*      m_pOwnerModelObj;
    nn::g3d::ShapeObj*      m_pShapeObj;
    nn::g3d::ResShape* m_pOriginalResShape;

    void*                   m_pOriginalBuffer;
    int                     m_OriginalViewCount;
    bool                    m_OriginalIsViewDependent;
    bool                    m_OriginalIsBounding;
    int                     m_OriginalBufferingCount;

    nn::gfx::MemoryPool*    m_pOriginalBlockMemoryPool;
    ptrdiff_t               m_OriginalBlockMemoryPoolOffset;
    size_t                  m_OriginalBlockBufferSize;

    void*                   m_pOriginalMeshArray;
    void*                   m_pMeshArray;
    void*                   m_pOriginalSubMeshBoundingArray;
    void*                   m_pSubMeshBoundingArray;
    nn::gfx::Buffer*        m_pOriginalBlockBufferArray;
    int                     m_OriginalShapeBlockCount;
    nn::gfx::Buffer**       m_pOriginalDynamicVertexBufferPtr;

    bool                    m_ViewDependent;

    EditWorkBuffer    m_WorkBuffer;
    EditWorkBuffer    m_WorkUserAreaBuffer;
    EditWorkBuffer    m_WorkMeshArrayBuffer;
    EditWorkBuffer    m_WorkSubMeshBoundingArrayBuffer;
    EditWorkBufferSwitcher m_BlockBufferObjectManager;

    // シェイプUBOコピーのために使うバッファー
    DynamicArray<ShapeBlock> m_ShapeBlockArray;

    int m_CurrentForceMeshLodLevel;

    // LOD レベル切り替え時などシェイプ更新用のメモリプール管理用
    MemoryPoolManager* m_MemoryPoolManager;

    // オリジナルの ResVertexAttr に設定されていたフラグを管理
    Map<const char*, nn::Bit8> m_OriginalResVertexAttrFlags;

private:
    NN_DISALLOW_COPY(EditShapeObj);
};

}}
}}


