﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditRenderInfoObj.h"



#include <nn/g3d/g3d_MaterialObj.h>
#include <nn/g3d/g3d_ResMaterial.h>
#include "../g3d_Allocator.h"

namespace nn { namespace g3d { namespace viewer { namespace detail {

EditRenderInfoObj::EditRenderInfoObj(
    Allocator* allocator,
    nn::g3d::MaterialObj* ownerMaterialObj) NN_NOEXCEPT
    : m_pAllocator(allocator)
    , m_pOwnerMaterial(ownerMaterialObj)
    , m_pOriginalRenderInfoDic(nullptr)
    , m_pOriginalRenderInfoArray(nullptr)
    , m_RenderInfoDicWorkBuffer(allocator, MaterialObj::Alignment_Buffer, AllocateType_DynamicBuffer)
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(allocator);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(ownerMaterialObj);
    nn::g3d::ResMaterial* pResMaterial = const_cast<nn::g3d::ResMaterial*>(ownerMaterialObj->GetResource());
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pResMaterial);

    m_pOriginalMaterial = pResMaterial;
    m_pOriginalRenderInfoDic = pResMaterial->ToData().pRenderInfoDic.Get();
    m_pOriginalRenderInfoArray = pResMaterial->ToData().pRenderInfoArray.Get();
    m_OriginalRenderInfoCount = pResMaterial->ToData().renderInfoCount;

    m_pUpdatedMaterial = m_pOriginalMaterial;
    m_pUpdatedRenderInfoDic = m_pOriginalRenderInfoDic;
    m_pUpdatedRenderInfoArray = m_pOriginalRenderInfoArray;
}

void
EditRenderInfoObj::Cleanup() NN_NOEXCEPT
{
    m_pUpdatedMaterial->ToData().pRenderInfoDic.Set(static_cast<nn::util::ResDic*>(m_pUpdatedRenderInfoDic));
    m_pUpdatedMaterial->ToData().pRenderInfoArray.Set(static_cast<nn::g3d::ResRenderInfo*>(m_pUpdatedRenderInfoArray));
    m_pOriginalMaterial->ToData().pRenderInfoDic.Set(static_cast<nn::util::ResDic*>(m_pOriginalRenderInfoDic));
    m_pOriginalMaterial->ToData().pRenderInfoArray.Set(static_cast<nn::g3d::ResRenderInfo*>(m_pOriginalRenderInfoArray));
    m_pOriginalMaterial->ToData().renderInfoCount = m_OriginalRenderInfoCount;
}

void
EditRenderInfoObj::Destroy() NN_NOEXCEPT
{
    m_RenderInfoDicWorkBuffer.Clear();
}

void
EditRenderInfoObj::SetUpdateMaterial(nn::g3d::ResMaterial* pResMaterial) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pResMaterial, "%s\n", m_pOriginalMaterial->GetName());
    Cleanup();
    m_pUpdatedRenderInfoDic = pResMaterial->ToData().pRenderInfoDic.Get();
    m_pUpdatedRenderInfoArray = pResMaterial->ToData().pRenderInfoArray.Get();
    m_pUpdatedMaterial = pResMaterial;
}

void
EditRenderInfoObj::Update(
    const void* pRenderInfoUpdateCommandData, size_t renderInfoUpdateCommandDataSize,
    ptrdiff_t renderInfoArrayOffset) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pRenderInfoUpdateCommandData, "%s\n", m_pOriginalMaterial->GetName());
    NN_G3D_VIEWER_ASSERT_DETAIL(renderInfoUpdateCommandDataSize > 0, "%s\n", m_pOriginalMaterial->GetName());
    NN_G3D_VIEWER_ASSERT_DETAIL(renderInfoArrayOffset > 0, "%s\n", m_pOriginalMaterial->GetName());

    Cleanup(); // 一旦元に戻す、必要ない場合はこの処理を削除

    bool result = m_RenderInfoDicWorkBuffer.Resize(renderInfoUpdateCommandDataSize);

    NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pOriginalMaterial, GetName()));
    m_RenderInfoDicWorkBuffer.CopyFrom(pRenderInfoUpdateCommandData, renderInfoUpdateCommandDataSize);

    // リロケート
    nn::util::ResDic* pResDic = reinterpret_cast<nn::util::ResDic*>(m_RenderInfoDicWorkBuffer.GetWorkBufferPtr());
    nn::g3d::ResRenderInfo* pRenderInfoArray = reinterpret_cast<nn::g3d::ResRenderInfo*>(reinterpret_cast<ptrdiff_t>(pResDic) + renderInfoArrayOffset);
    {
        nn::util::ResDicData* pResDicData = &pResDic->ToData();
        int renderInfoCount = pResDicData->count;

        for (int i = 0, end = pResDicData->count + 1; i < end; ++i)
        {
            nn::util::ResDicData::Entry* pEntry = &pResDicData->entries[i];
            pEntry->pKey.Relocate(&pEntry->pKey);
            NN_G3D_VIEWER_DEBUG_PRINT("pResDicData->entries[%d] = %s\n", i, pEntry->pKey.Get()->GetData());
        }

        for (int renderInfoIndex = 0; renderInfoIndex < renderInfoCount; ++renderInfoIndex)
        {
            nn::g3d::ResRenderInfoData* pRenderInfoData = &pRenderInfoArray[renderInfoIndex].ToData();
            pRenderInfoData->pName.Relocate(&pRenderInfoData->pName);
            NN_G3D_VIEWER_DEBUG_PRINT("pRenderInfoData[%d] = %s\n", renderInfoIndex, pRenderInfoData->pName.Get()->GetData());
            switch (static_cast<ResRenderInfo::Type>(pRenderInfoData->type))
            {
            case ResRenderInfo::Type_Int:
                {
                    pRenderInfoData->intValueArray.Relocate(&pRenderInfoData->intValueArray);
                    for (int valueIndex = 0, valueCount = pRenderInfoData->arrayLength; valueIndex < valueCount; ++valueIndex)
                    {
                        NN_G3D_VIEWER_DEBUG_PRINT("intValueArray[%d] = %d\n", valueIndex, pRenderInfoData->intValueArray.Get()[valueIndex]);
                    }
                }
                break;
            case ResRenderInfo::Type_Float:
                {
                    pRenderInfoData->floatValueArray.Relocate(&pRenderInfoData->floatValueArray);
                    for (int valueIndex = 0, valueCount = pRenderInfoData->arrayLength; valueIndex < valueCount; ++valueIndex)
                    {
                        NN_G3D_VIEWER_DEBUG_PRINT("floatValueArray[%d] = %f\n", valueIndex, pRenderInfoData->floatValueArray.Get()[valueIndex]);
                    }
                }
                break;
            case ResRenderInfo::Type_String:
                {
                    pRenderInfoData->stringArray.Relocate(&pRenderInfoData->stringArray);
                    for (int valueIndex = 0, valueCount = pRenderInfoData->arrayLength; valueIndex < valueCount; ++valueIndex)
                    {
                        nn::util::BinPtrToString* pBinStringPtr = &pRenderInfoData->stringArray.Get()[valueIndex];
                        pBinStringPtr->Relocate(pBinStringPtr);
                        NN_G3D_VIEWER_DEBUG_PRINT("stringArray[%d] = %s\n", valueIndex, pBinStringPtr->Get()->GetData());
                    }
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }

    bool successBuildRenderInfoDic = pResDic->Build();
    NN_UNUSED(successBuildRenderInfoDic);
    NN_G3D_VIEWER_ASSERT_DETAIL(successBuildRenderInfoDic, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pOriginalMaterial, GetName()));

    m_UpdateRenderInfoCount = static_cast<uint16_t>(pResDic->GetCount());
    m_pUpdatedMaterial->ToData().pRenderInfoDic.Set(pResDic);
    m_pUpdatedMaterial->ToData().pRenderInfoArray.Set(pRenderInfoArray);
    m_pUpdatedMaterial->ToData().renderInfoCount = m_UpdateRenderInfoCount;
}

}}}} // namespace nn::g3d::viewer::detail


