﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_ViewerServerImpl.h"

#include "g3d_Allocator.h"
#include "g3d_PingMonitor.h"
#include "g3d_CommandMonitor.h"
#include "util/g3d_ViewerUtility.h"
#include "util/g3d_ResourceEditUtility.h"
#include "g3d_HostFileDevice.h"
#include "model/g3d_EditModelObj.h"
#include "shader/g3d_EditShaderArchive.h"
#include "anim/g3d_EditAnimObj.h"
#include "g3d_ViewerKeyManager.h"
#include "g3d_ResourceManager.h"

#include <nn/g3d/g3d_ResFile.h>
#include <nn/diag/text/diag_SdkTextG3dviewer.h>
#include <nn/g3d/g3d_ResShader.h>

using namespace nn::g3d::viewer::detail;

namespace {

bool g_IsRuntimeDebugLogEnabled = false;

} // anonymous namespace

bool nn::g3d::viewer::detail::IsRuntimeDebugLogEnabled() NN_NOEXCEPT
{
    return g_IsRuntimeDebugLogEnabled;
}

namespace nn { namespace g3d { namespace viewer {

ViewerServer::ViewerServerProxy::Impl::Impl(
    nn::gfx::Device* pDevice,
    Allocator* pAllocator,
    const ViewerCallback callbackFunc,
    void* pCallbackUserData,
    const ViewerTextureBindCallback textureBindCallback,
    void* pTextureBindCallbackUserData) NN_NOEXCEPT
    : m_pAllocator(pAllocator)
    , m_CallbackCaller(callbackFunc, pCallbackUserData, textureBindCallback, pTextureBindCallbackUserData)
    , m_PingMonitor(m_pAllocator)
    , m_CommandMonitor(&nn::g3d::viewer::ViewerServer::ViewerServerProxy::Impl::AnalyzeCommand, &nn::g3d::viewer::ViewerServer::ViewerServerProxy::Impl::ProcessCommand)
    , m_EditCommandExecutor(m_pAllocator, &m_CallbackCaller, &m_EditCommandManager)
    , m_pFileDevice(nullptr)
    , m_EditCommandManager(pDevice, m_pAllocator, &m_CallbackCaller)
    , m_EditPickup(m_pAllocator)
    , m_EditAnimControl()
    , m_Mutex(true)
    , m_FileBuffer(nullptr)
    , m_FileBufferSize(0)
    , m_MultiFileState(MULTI_FILE_END)
    , m_IsFileLoading(false)
    , m_IsCloseRequestEnabledOnClose(true)
{
    void* buffer = m_pAllocator->Allocate(sizeof(HostFileDeviceGeneric), Alignment_Default, AllocateType_Communication);
    m_pFileDevice = new (buffer) HostFileDeviceGeneric();
}

ViewerServer::ViewerServerProxy::Impl::~Impl() NN_NOEXCEPT
{
    m_EditCommandManager.ForceDeleteAll();
    ClearFileBuffer();

    m_pFileDevice->~HostFileDeviceBase();
    m_pAllocator->Free(m_pFileDevice);
}

ViewerResult ViewerServer::ViewerServerProxy::Impl::Initialize(const InitializeArg& arg) NN_NOEXCEPT
{
    {
        ViewerResult result = m_PingMonitor.Initialize(arg.m_CodePage);
        if (result != ViewerResult_Success)
        {
            return result;
        }
    }

    {
        CommandMonitor::InitArg commandMonitorInitArg;
        commandMonitorInitArg.pAllocator = m_pAllocator;
        ViewerResult result = m_CommandMonitor.Initialize(commandMonitorInitArg);
        if (result != ViewerResult_Success)
        {
            return result;
        }
    }

    {
        ViewerResult result = m_pFileDevice->Initialize();
        if (result != ViewerResult_Success)
        {
            return result;
        }
    }

    m_EditCommandManager.SetCodePage(arg.m_CodePage);

    return ViewerResult_Success;
}

ViewerResult ViewerServer::ViewerServerProxy::Impl::Open() NN_NOEXCEPT
{
    ScopedLock scopedLock(m_Mutex);
    if (GetPingMonitor().IsClosing())
    {
        return ViewerResult_ServerClosing;
    }

    bool success = true;
    success &= GetCommandMonitor().RequestOpen();
    success &= GetPingMonitor().RequestOpen();
    if (!success)
    {
        return ViewerResult_UnknownError;
    }

    return ViewerResult_Success;
}

void ViewerServer::ViewerServerProxy::Impl::Close() NN_NOEXCEPT
{
    ScopedLock scopedLock(m_Mutex);
    if (!IsOpenRequested())
    {
        return;
    }

    if (GetCommandMonitor().IsConnected())
    {
        ClearState();
        GetCommandMonitor().Close();

        // 3DEditor と接続中の場合は、3DEditor に明示的にクローズされたことを Ping で通知して切断する
        GetPingMonitor().RequestClose();

        // この後、PingMonitor は Poll() が実行されることにより遅延してクローズされるため、
        // アプリケーションが Poll() を呼ばなければ完全にクローズされない
    }
    else
    {
        // 3DEditor と通信不可であれば強制クローズ
        CloseForce();
    }
}

void ViewerServer::ViewerServerProxy::Impl::CloseForce() NN_NOEXCEPT
{
    ScopedLock scopedLock(m_Mutex);
    if (!IsOpenRequested())
    {
        return;
    }

    ClearState();
    GetCommandMonitor().Close();
    GetPingMonitor().Close();
}

void
ViewerServer::ViewerServerProxy::Impl::PollDataCommunication() NN_NOEXCEPT
{
    ScopedLock scopedLock(m_Mutex);

    // ここでは通信処理以外の処理(リソースやオブジェクトの変更など)をしてはいけない
    m_PingMonitor.Poll();
    bool success = m_CommandMonitor.PollDataCommunication();
    if (!success && m_PingMonitor.IsFreezing())
    {
        // PollDataViewer を実行するスレッドが無限ループにならないようにフリーズを解除
        m_PingMonitor.EndFreeze();
        return;
    }

    if (!IsConnected())
    {
        m_PingMonitor.ClearState();
        return;
    }

    // メインスレッドと非同期処理が可能なコマンドキューにためているものを処理
    m_EditCommandManager.ExecuteThreadSafeCommands(&m_CommandMonitor.GetSocket());

    // シェーダアーカイブの更新があれば処理
    GetResourceManager().SendModifiedShaderPrograms(&m_CommandMonitor.GetSocket());

    // ピックアップ情報があった場合は、情報を送信
    if (m_EditPickup.MakePickupPacket())
    {
        m_EditPickup.SendPickup(&m_CommandMonitor.GetSocket());
        m_EditPickup.ClearPickup();
    }
}

void
ViewerServer::ViewerServerProxy::Impl::PollDataEdit() NN_NOEXCEPT
{
    // 遅延破棄が予定されているリソースの破棄
    m_EditCommandManager.TryDestroyUnreferencedResources();

    if (!IsConnected())
    {
        this->ClearState();
        return;
    }

    bool success = m_CommandMonitor.PollDataEdit();
    if (!success)
    {
        return;
    }

    // モデルアニメーションの計算フラグをクリア
    GetResourceManager().ClearCalculateFlagModelAnims();

    // コマンドキューにためているものを処理
    m_EditCommandManager.ExecuteThreadUnsafeCommands();
}

CommandMonitor::AnalyzeCommandResult ViewerServer::ViewerServerProxy::Impl::AnalyzeCommandImpl(CommandMonitor::AnalyzeCommandArg& analyzeCommandArg) NN_NOEXCEPT
{
    NN_G3D_VIEWER_LOG(
        "%sViewerServer::AnalyzeCommand : %s\n",
        GetLogIndent(analyzeCommandArg.command), GetEditCommandString(static_cast<CommandFlag>(analyzeCommandArg.command)));
    switch(analyzeCommandArg.command)
    {
    case SYSTEM_BEGIN_FREEZE_COMMAND_FLAG:
    case SYSTEM_BEGIN_FREEZE_NO_SYNC_COMMAND_FLAG:
        {
            bool sync = analyzeCommandArg.command == SYSTEM_BEGIN_FREEZE_COMMAND_FLAG;
            m_PingMonitor.BeginFreeze(sync);
        }
        return CommandMonitor::AnalyzeCommandResult_Finish;
    case SYSTEM_END_FREEZE_COMMAND_FLAG:
        m_PingMonitor.EndFreeze();
        return CommandMonitor::AnalyzeCommandResult_Finish;
    case SYSTEM_RUNTIME_STATE_NORMAL_COMMAND_FLAG:
        m_PingMonitor.SetRuntimeStateToNormal();
        return CommandMonitor::AnalyzeCommandResult_Finish;
    case SYSTEM_RUNTIME_LOG_COMMAND_FLAG:
        {
            EditValueInfoBlock* block = static_cast<EditValueInfoBlock*>(analyzeCommandArg.pWorkBuffer);
            EditValueBlock* valueBlock = reinterpret_cast<EditValueBlock*>(block + 1);
            g_IsRuntimeDebugLogEnabled = valueBlock->value.bValue;
            if (g_IsRuntimeDebugLogEnabled)
            {
                NN_G3D_VIEWER_LOG("Debug log enabled\n");
            }
            else
            {
                NN_G3D_VIEWER_LOG("Debug log disabled\n");
            }
        }
        return CommandMonitor::AnalyzeCommandResult_Finish;
    case FILEDATA_LOAD_FILE_COMMAND_FLAG:
    case FILEDATA_RELOAD_FILE_COMMAND_FLAG:
    case EDIT_RECV_ATTACH_COMMAND_FLAG:
    case EDIT_RECV_MODIFIED_SHADER_COMMAND_FLAG:
        {
            FileDataBlock* block = static_cast<FileDataBlock*>(analyzeCommandArg.pWorkBuffer);
            ClearFileBuffer();
            size_t alignment = block->fileAlignment;

            NN_G3D_VIEWER_ASSERT(m_FileBuffer == nullptr);

            size_t fileSize = block->fileSize;
            if (fileSize > 0)
            {
                NN_G3D_VIEWER_DEBUG_PRINT("Try Allocate: size = %zu, alignment = %zu\n", fileSize, alignment);
                m_FileBuffer = m_pAllocator->Allocate(fileSize, alignment, AllocateType_Resource);
                if (m_FileBuffer)
                {
                    NN_G3D_VIEWER_DEBUG_PRINT("Allocated: address = 0x%p\n", m_FileBuffer);
                    m_FileBufferSize = fileSize;

                    // キャストしないとビルドエラーになるので
                    const char* filePath = reinterpret_cast<const char*>(block->fileName);

                    NN_G3D_VIEWER_DEBUG_PRINT(
                        "Reading file: path = %s, size = %zu, alignment = %zu\n", filePath, fileSize, alignment);

                    bool result = m_pFileDevice->Open(filePath, HostFileDeviceBase::OpenFlag_ReadOnly);
                    if (!result)
                    {
                        m_EditCommandManager.QueueErrorCommand(RUNTIME_ERROR_CODE_OPEN_FILE_FAILED);
                        return CommandMonitor::AnalyzeCommandResult_Error;
                    }

                    result = m_pFileDevice->ReadASync(m_FileBuffer, fileSize);
                    NN_G3D_VIEWER_ASSERT_DETAIL(result, "Reading %s failed\n", filePath);

                    m_IsFileLoading = result;

                    NN_G3D_VIEWER_DEBUG_PRINT("Complete reading: address = 0x%p, size = %zu, alignment = %zu\n", m_FileBuffer, fileSize, alignment);
                }
                else
                {
                    m_EditCommandManager.QueueErrorCommand(RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY);
                    return CommandMonitor::AnalyzeCommandResult_Error;
                }
            }
            else
            {
                // シェーダーアタッチ時はアタッチするファイルは送られてこない
                // 編集時にシェーダーバイナリーが送られてくる
                m_IsFileLoading = true;
            }
        }
        break;
    case EDIT_RECV_RENDER_INFO_COMMAND_FLAG:
        break;
    case EDIT_SELECT_EDIT_RENDER_INFO_COMMAND_FLAG:
        break;
    case EDIT_RENDER_INFO_ARRAY_SIZE_COMMAND_FLAG:
        break;
    case EDIT_UPDATE_RENDER_INFO_COMMAND_FLAG:
        break;
    case FILEDATA_UNLOAD_FILE_COMMAND_FLAG:
        break;
    case FILEDATA_UNLOAD_ALL_COMMAND_FLAG:
        break;
    case EDIT_MATERIAL_COMMAND_FLAG:
        break;
    case MODEL_ANIMATION_BIND_COMMAND_FLAG:
    case MODEL_ANIMATION_UNBIND_COMMAND_FLAG:
    case SCENE_ANIMATION_BIND_COMMAND_FLAG:
    case SCENE_ANIMATION_UNBIND_COMMAND_FLAG:
        break;
    case MODEL_ANIMATION_EDIT_RETARGET_HOST_MODEL_COMMAND_FLAG:
    case MODEL_ANIMATION_EDIT_MIRRORING_ENABLED_COMMAND_FLAG:
        break;
    case ANIMATION_PLAY_FRAME_CTRL_COMMAND_FLAG:
    case ANIMATION_STOP_FRAME_CTRL_COMMAND_FLAG:
    case ANIMATION_PLAY_POLICY_COMMAND_FLAG:
    case ANIMATION_FRAME_STEP_COMMAND_FLAG:
    case ANIMATION_FRAME_COUNT_COMMAND_FLAG:
    case ANIMATION_START_FRAME_COMMAND_FLAG:
        {
            FrameCtrlBlock* block = static_cast<FrameCtrlBlock*>(analyzeCommandArg.pWorkBuffer);
            switch(analyzeCommandArg.command)
            {
            case ANIMATION_PLAY_FRAME_CTRL_COMMAND_FLAG:
                GetResourceManager().SetFrame(block->frame);
                GetResourceManager().SetAnimPlaying(true);
                break;
            case ANIMATION_STOP_FRAME_CTRL_COMMAND_FLAG:
                GetResourceManager().SetFrame(block->frame);
                GetResourceManager().SetAnimPlaying(false);
                break;
            case ANIMATION_PLAY_POLICY_COMMAND_FLAG:
                GetResourceManager().SetPlayPolicy(static_cast<EditPlayPolicyKind>(block->playPolicy));
                break;
            case ANIMATION_FRAME_STEP_COMMAND_FLAG:
                GetResourceManager().SetFrameStep(block->frameStep);
                break;
            case ANIMATION_FRAME_COUNT_COMMAND_FLAG:
                GetResourceManager().SetFrameCount(block->frameCount);
                break;
            case ANIMATION_START_FRAME_COMMAND_FLAG:
                GetResourceManager().SetStartFrame(block->startFrame);
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
        break;
    case MODEL_ANIMATION_EDIT_CURVE_COMMAND_FLAG:
    case SCENE_ANIMATION_EDIT_CURVE_COMMAND_FLAG:
        break;
    case MODEL_ANIMATION_PLAY_COMMAND_FLAG:
        {
            AnimEditInfoBlock* block = static_cast<AnimEditInfoBlock*>(analyzeCommandArg.pWorkBuffer);
            EditAnimObj* pEditAnimObj = nullptr;
            if (block->modelKey != 0)
            {
                pEditAnimObj = GetResourceManager().FindBoundEditAnimObj(block->modelKey, block->animationKey);
                if (pEditAnimObj != nullptr)
                {
                    EditModelObj* pEditModelObj = GetResourceManager().FindEditModelObj(block->modelKey);
                    pEditAnimObj->SetPauseFlag(pEditModelObj, false, block->fValue);
                }
            }
        }
        break;
    case MODEL_ANIMATION_STOP_COMMAND_FLAG:
        {
            AnimEditInfoBlock* block = static_cast<AnimEditInfoBlock*>(analyzeCommandArg.pWorkBuffer);
            EditAnimObj* pEditAnimObj = nullptr;
            if (block->modelKey != 0)
            {
                pEditAnimObj = GetResourceManager().FindBoundEditAnimObj(block->modelKey, block->animationKey);
                if (pEditAnimObj != nullptr)
                {
                    EditModelObj* pEditModelObj = GetResourceManager().FindEditModelObj(block->modelKey);
                    pEditAnimObj->SetPauseFlag(pEditModelObj, true, block->fValue);
                }
            }
        }
        break;
    case EDIT_SHADER_COMMAND_FLAG:
    case EDIT_BONE_COMMAND_FLAG:
    case EDIT_MODEL_BONE_BIND_COMMAND_FLAG:
    case EDIT_MODEL_LAYOUT_COMMAND_FLAG:
    case EDIT_RECV_MODEL_LAYOUT_COMMAND_FLAG:
    case EDIT_SET_SHAPE_LOD_LEVEL_COMMAND_FLAG:
    case EDIT_RESET_SHAPE_LOD_LEVEL_COMMAND_FLAG:
        break;
    case PICK_RUNTIME_MODEL_COMMAND_FLAG:
    case PICK_RUNTIME_MATERIAL_COMMAND_FLAG:
    case PICK_RUNTIME_BONE_COMMAND_FLAG:
    case PICK_RUNTIME_SHAPE_COMMAND_FLAG:
        break;
    case EDIT_LOAD_SHADER_ARCHIVE_COMMAND_FLAG:
    case EDIT_RESET_SHADER_ARCHIVE_COMMAND_FLAG:
        {
            m_MultiFileState = MULTI_FILE_START;

            // 複数ファイルロードを行うので、ファイルロード中の場合は強制的にクローズ
            if (m_pFileDevice->IsReading())
            {
                m_pFileDevice->Close();
            }
        }
        break;
    case OTHER_EXECUTE_USER_SCRIPT_FLAG:
        break;
    case TEXTURE_BIND_COMMAND_FLAG:
        break;
    default:
        NN_G3D_WARNING(false, "Unkown command detected %d\n", analyzeCommandArg.command);
        break;
    }

    return CommandMonitor::AnalyzeCommandResult_RequestProcessCommand;
} // NOLINT (readability/fn_size)

nn::g3d::viewer::detail::CommandMonitor::ProcessCommandResult
ViewerServer::ViewerServerProxy::Impl::ProcessCommandImpl(CommandMonitor::ProcessCommandArg& processCommandArg) NN_NOEXCEPT
{
    NN_G3D_VIEWER_DEBUG_PRINT("%sViewerServer::ProcessCommand : %s\n",
        GetLogIndent(processCommandArg.command), GetEditCommandString(static_cast<CommandFlag>(processCommandArg.command)));

    switch(processCommandArg.command)
    {
    case FILEDATA_LOAD_FILE_COMMAND_FLAG:
    case FILEDATA_RELOAD_FILE_COMMAND_FLAG:
    case EDIT_RECV_ATTACH_COMMAND_FLAG:
    case EDIT_RECV_MODIFIED_SHADER_COMMAND_FLAG:
        {
            // ファイルロード
            bool success = ExecuteFileLoadCommand(processCommandArg);
            if (!success)
            {
                return CommandMonitor::ProcessCommandResult_Error;
            }
        }
        break;
    case EDIT_LOAD_SHADER_ARCHIVE_COMMAND_FLAG:
    case EDIT_RESET_SHADER_ARCHIVE_COMMAND_FLAG:
        {
            // 複数ファイルロード
            bool success = ExecuteMultiFileLoadCommand(processCommandArg);
            if (!success)
            {
                return CommandMonitor::ProcessCommandResult_Error;
            }
        }
        break;
    case MODEL_ANIMATION_EDIT_RETARGET_HOST_MODEL_COMMAND_FLAG:
        {
            AnimEditInfoBlock* block = static_cast<AnimEditInfoBlock*>(processCommandArg.pWorkBuffer);
            ViewerKeyType retargetHostModelObjKey = block->modelKey;
            ViewerKeyType animationResFileKey = block->animationKey;
            if (retargetHostModelObjKey != 0)
            {
                m_EditCommandManager.SetRetargetingHostModel(animationResFileKey, retargetHostModelObjKey);
            }
            else
            {
                // リターゲッティング解除
                m_EditCommandManager.UnsetRetargetingHostModel(animationResFileKey);
            }
        }
        break;
    case MODEL_ANIMATION_EDIT_MIRRORING_ENABLED_COMMAND_FLAG:
        {
            AnimEditInfoBlock* block = static_cast<AnimEditInfoBlock*>(processCommandArg.pWorkBuffer);
            uint32_t mirroringEnabled = block->iValue;
            ViewerKeyType animationResFileKey = block->animationKey;
            if (mirroringEnabled)
            {
                m_EditCommandManager.SetPlayMotionMirroringEnabled(animationResFileKey, true);
            }
            else
            {
                m_EditCommandManager.SetPlayMotionMirroringEnabled(animationResFileKey, false);
            }
        }
        break;
    case EDIT_RENDER_INFO_ARRAY_SIZE_COMMAND_FLAG:
        {
            // RenderInfo配列サイズ編集処理
            RenderInfoEditInfo* info = static_cast<RenderInfoEditInfo*>(processCommandArg.pWorkBuffer);
            const char* labelName = AddOffset<const char>(info, info->labelOffset - sizeof(PacketHeader));
            ModelObj* pModelObj = ViewerKeyManager::GetInstance().FindData<nn::g3d::ModelObj>(info->modelKey);
            nn::g3d::viewer::detail::EditRenderInfoArraySize(pModelObj, labelName, info);
        }
        break;
    case EDIT_UPDATE_RENDER_INFO_COMMAND_FLAG:
        {
            RenderInfoUpdateBlock* block = static_cast<RenderInfoUpdateBlock*>(processCommandArg.pWorkBuffer);
            detail::EditModelObj* pEditModelObj = GetResourceManager().FindEditModelObj(block->modelKey);
            if (pEditModelObj) // nullptr以外の場合は削除処理を行う
            {
                pEditModelObj->UpdateRenderInfo(
                    block->materialIndex,
                    reinterpret_cast<void*>(block->renderInfoData), static_cast<size_t>(processCommandArg.workBufferSize),
                    static_cast<ptrdiff_t>(block->renderInfoDicDataSize));
            }
        }
        break;
    case EDIT_RECV_RENDER_INFO_COMMAND_FLAG:
        {
            // RenderInfo受信処理
            RenderInfoRecvBlock* pBlock = static_cast<RenderInfoRecvBlock*>(processCommandArg.pWorkBuffer);

            // リロケート
            {
                RelocatePacketBinPtr(&pBlock->ofsMaterialIndexArray);
                RelocatePacketBinPtr(&pBlock->ofsRenderInfoArray);
                SetupRenderInfoData* pSetupRenderInfoDataArray = static_cast<SetupRenderInfoData*>(pBlock->ofsMaterialIndexArray.Get());
                for (int index = 0, end = static_cast<int>(pBlock->numRenderInfo); index < end; ++index)
                {
                    SetupRenderInfoData* setupRenderInfoData = &pSetupRenderInfoDataArray[index];
                    RelocatePacketBinPtr(&setupRenderInfoData->ofsChoice);
                    RelocatePacketBinPtr(&setupRenderInfoData->ofsDefault);
                }
            }

            m_EditCommandExecutor.ExecuteSendRenderInfoDefinition(pBlock, &m_CommandMonitor);
        }
        break;
    case EDIT_SELECT_EDIT_RENDER_INFO_COMMAND_FLAG:
        {
            // RenderInfo選択編集処理
            RenderInfoEditInfo* info = static_cast<RenderInfoEditInfo*>(processCommandArg.pWorkBuffer);
            m_EditCommandExecutor.ExecuteEditRenderInfo(info);
        }
        break;
    case EDIT_MATERIAL_COMMAND_FLAG:
        {
            EditValueInfoBlock* block = static_cast<EditValueInfoBlock*>(processCommandArg.pWorkBuffer);
            m_EditCommandExecutor.ExecuteEditMaterial(block);
        }
        break;
    case FILEDATA_UNLOAD_FILE_COMMAND_FLAG:
        {
            FileDataBlock* block = static_cast<FileDataBlock*>(processCommandArg.pWorkBuffer);
            m_EditCommandExecutor.ExecuteUnloadFile(block);
        }
        break;
    case FILEDATA_UNLOAD_ALL_COMMAND_FLAG:
        m_EditCommandManager.DeleteAll();
        break;
    case MODEL_ANIMATION_BIND_COMMAND_FLAG:
    case MODEL_ANIMATION_UNBIND_COMMAND_FLAG:
    case SCENE_ANIMATION_BIND_COMMAND_FLAG:
    case SCENE_ANIMATION_UNBIND_COMMAND_FLAG:
        {
            BindAnimInfoBlock* block = static_cast<BindAnimInfoBlock*>(processCommandArg.pWorkBuffer);
            m_EditCommandExecutor.ExecuteBindAnimation(processCommandArg.command, block);
        }
        break;
    case MODEL_ANIMATION_EDIT_CURVE_COMMAND_FLAG:
    case SCENE_ANIMATION_EDIT_CURVE_COMMAND_FLAG:
        {
            AnimCurveBlock* block = static_cast<AnimCurveBlock*>(processCommandArg.pWorkBuffer);
            m_EditCommandExecutor.ExecuteEditAnimationCurve(processCommandArg.command, block);
        }
        break;
    case EDIT_SHADER_COMMAND_FLAG:
        {
            EditValueInfoBlock* block = static_cast<EditValueInfoBlock*>(processCommandArg.pWorkBuffer);
            m_EditCommandExecutor.ExecuteUpdateShadingModels(block);
        }
        break;
    case EDIT_BONE_COMMAND_FLAG:
        {
            EditValueInfoBlock* block = static_cast<EditValueInfoBlock*>(processCommandArg.pWorkBuffer);

            switch(block->editTargetKind)
            {
            case EDIT_TARGET_BONE_VISIBILITY:
            case EDIT_TARGET_BONE_BILLBOARD:
                {
                    EditValueBlock* valueBlock = reinterpret_cast<EditValueBlock*>(block + 1);
                    EditBoneArg arg;
                    arg.modelKey = block->key;
                    arg.valueKind = block->valueKind;
                    arg.editTargetKind = block->editTargetKind;
                    arg.value = static_cast<void*>(&valueBlock->value);
                    arg.index = valueBlock->index;
                    arg.indexCount = block->indexCount;

                    GetResourceManager().EditBones(arg);
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
        break;
    case EDIT_MODEL_BONE_BIND_COMMAND_FLAG:
        {
            BondBindEditBlock* block = static_cast<BondBindEditBlock*>(processCommandArg.pWorkBuffer);

            // ボーンバインドの親モデルが nullptr は正常系
            EditModelObj* pParentEditModelObj = GetResourceManager().FindEditModelObj(block->parentModelKey);

            EditModelObj* pChildEditModelObj = GetResourceManager().FindEditModelObj(block->childModelKey);
            if (pChildEditModelObj == nullptr)
            {
                m_EditCommandManager.QueueErrorCommand(RUNTIME_ERROR_CODE_TARGET_MODEL_NOT_FOUND);
                return CommandMonitor::ProcessCommandResult_Error;
            }

            int childModelCount = pChildEditModelObj->GetAttachedModelObjCount();
            for (int childModelIndex = 0; childModelIndex < childModelCount; ++childModelIndex)
            {
                BoneBindUpdatedArg arg;
                arg.pParentModelObj = nullptr;
                if (pParentEditModelObj != nullptr)
                {
                    NN_G3D_VIEWER_ASSERT(pParentEditModelObj->GetTargetModelObj(0));
                    arg.pParentModelObj = pParentEditModelObj->GetTargetModelObj(0);
                }
                arg.pModelObj = pChildEditModelObj->GetTargetModelObj(childModelIndex);
                arg.parentBoneIndex = block->parentBoneIndex;

                m_CallbackCaller.Call(CallbackType_BoneBindUpdated, &arg);
            }
        }
        break;
    case EDIT_MODEL_LAYOUT_COMMAND_FLAG:
        {
            {
                ModelLayoutEditBlock* block = static_cast<ModelLayoutEditBlock*>(processCommandArg.pWorkBuffer);

                EditModelObj* pEditModelObj = GetResourceManager().FindEditModelObj(block->modelKey);
                if (pEditModelObj == nullptr)
                {
                    m_EditCommandManager.QueueErrorCommand(RUNTIME_ERROR_CODE_TARGET_MODEL_NOT_FOUND);
                    return CommandMonitor::ProcessCommandResult_Error;
                }

                for (int modelIndex = 0, modelCount = pEditModelObj->GetAttachedModelObjCount(); modelIndex < modelCount; ++modelIndex)
                {
                    ModelLayoutUpdatedArg arg;
                    arg.pModelObj = pEditModelObj->GetTargetModelObj(modelIndex);
                    nn::util::VectorSet(&arg.scale, block->scale.a[0], block->scale.a[1], block->scale.a[2]);
                    nn::util::VectorSet(&arg.rotate, block->rotate.a[0], block->rotate.a[1], block->rotate.a[2]);
                    nn::util::VectorSet(&arg.translate, block->translate.a[0], block->translate.a[1], block->translate.a[2]);

                    m_CallbackCaller.Call(CallbackType_ModelLayoutUpdated, &arg);
                }
            }
        }
        break;
    case EDIT_RECV_MODEL_LAYOUT_COMMAND_FLAG:
        {
            // 3DEditor からプレビュー配置情報の送信を要求された
            ModelLayoutRecvBlock* block = static_cast<ModelLayoutRecvBlock*>(processCommandArg.pWorkBuffer);
            EditModelObj* pEditModelObj = GetResourceManager().FindEditModelObj(block->modelKey);
            if (pEditModelObj == nullptr)
            {
                m_EditCommandManager.QueueErrorCommand(RUNTIME_ERROR_CODE_TARGET_MODEL_NOT_FOUND);
                return CommandMonitor::ProcessCommandResult_Error;
            }

            NN_G3D_VIEWER_ASSERT(pEditModelObj->GetAttachedModelObjCount() > 0);

            SendModelLayoutRequestedOutArg outArg;
            nn::util::VectorSet(&outArg.scale, 1.f, 1.f, 1.f);
            nn::util::VectorSet(&outArg.rotate, 0.f, 0.f, 0.f);
            nn::util::VectorSet(&outArg.translate, 0.f, 0.f, 0.f);

            // デフォルト値が true だと、
            // クラスコールバックの場合、コールバックが未実装なら配置情報リセットは無効化され、オーバーライドして実装すれば有効化される
            // 関数コールバックの場合、コールバックが未実装でも有効化されるが、接続時にプレビュー配置がリセットされるだけなので、
            // 前者のメリットを優先して true にする
            outArg.isEnabled = true;

            SendModelLayoutRequestedArg inArg;
            inArg.pModelObj = pEditModelObj->GetTargetModelObj(0);
            m_CallbackCaller.Call(&outArg, &inArg, CallbackType_SendModelLayoutRequested);
            if (outArg.isEnabled)
            {
                m_EditCommandManager.AddModelLayoutQueue(
                    block->modelKey,
                    outArg.scale, outArg.rotate, outArg.translate);
            }
        }
        break;
    case EDIT_SET_SHAPE_LOD_LEVEL_COMMAND_FLAG:
    case EDIT_RESET_SHAPE_LOD_LEVEL_COMMAND_FLAG:
        {
            ShapeLodLevelEditBlock* block = static_cast<ShapeLodLevelEditBlock*>(processCommandArg.pWorkBuffer);
            EditModelObj* pEditModelObj = GetResourceManager().FindEditModelObj(block->modelKey);
            if (pEditModelObj == nullptr)
            {
                m_EditCommandManager.QueueErrorCommand(RUNTIME_ERROR_CODE_TARGET_MODEL_NOT_FOUND);
                return CommandMonitor::ProcessCommandResult_Error;
            }

            NN_G3D_VIEWER_ASSERT(pEditModelObj->GetAttachedModelObjCount() > 0);

            if (processCommandArg.command == EDIT_SET_SHAPE_LOD_LEVEL_COMMAND_FLAG)
            {
                pEditModelObj->SetLodLevel(block->lodLevel);
            }
            else
            {
                pEditModelObj->ResetLodLevel();
            }
        }
        break;
    case TEXTURE_BIND_COMMAND_FLAG:
        {
            TextureBindingBlock* pBlock = static_cast<TextureBindingBlock*>(processCommandArg.pWorkBuffer);

            // デバッグ表示
            {
                NN_G3D_VIEWER_DEBUG_PRINT("Update texture bindings: bindTargetKey = %u\n", pBlock->bindTargetKey);
                for (int keyIndex = 0; keyIndex < pBlock->textureKeyArrayCount; ++keyIndex)
                {
                    NN_G3D_VIEWER_DEBUG_PRINT("    textureKeyArrayData[%d] = %u\n", keyIndex, pBlock->textureKeyArrayData[keyIndex]);
                }
            }

            RuntimeErrorCode result = GetResourceManager().UpdateTextureBindings(pBlock);
            if (result != RUNTIME_ERROR_CODE_NO_ERROR)
            {
                m_EditCommandManager.QueueErrorCommand(result);
                return CommandMonitor::ProcessCommandResult_Error;
            }
        }
        break;
    case PICK_RUNTIME_MODEL_COMMAND_FLAG:
    case PICK_RUNTIME_MATERIAL_COMMAND_FLAG:
    case PICK_RUNTIME_BONE_COMMAND_FLAG:
    case PICK_RUNTIME_SHAPE_COMMAND_FLAG:
        {
            EditValueInfoBlock* pBlock = static_cast<EditValueInfoBlock*>(processCommandArg.pWorkBuffer);
            m_EditCommandExecutor.ExecutePickup(processCommandArg.command, pBlock);
        }
        break;

    case OTHER_EXECUTE_USER_SCRIPT_FLAG:
        {
            UserScriptBlock* pBlock = static_cast<UserScriptBlock*>(processCommandArg.pWorkBuffer);
            m_EditCommandExecutor.ExecuteUserScript(pBlock);
        }
        break;

    default:
        break;
    }

    return CommandMonitor::ProcessCommandResult_Finish;
} // NOLINT (readability/fn_size)

bool ViewerServer::ViewerServerProxy::Impl::ExecuteFileLoadCommand(CommandMonitor::ProcessCommandArg& processCommandArg) NN_NOEXCEPT
{
    if (!m_IsFileLoading)
    {
        return true;
    }

    if (m_pFileDevice->IsReading())
    {
        return true;
    }

    m_pFileDevice->Close();

    FileDataBlock* block = nullptr;
    bool isResFile = false;
    if (processCommandArg.command == FILEDATA_LOAD_FILE_COMMAND_FLAG ||
        processCommandArg.command == FILEDATA_RELOAD_FILE_COMMAND_FLAG ||
        processCommandArg.command == EDIT_RECV_ATTACH_COMMAND_FLAG ||
        processCommandArg.command == EDIT_RECV_MODIFIED_SHADER_COMMAND_FLAG)
    {
        block = static_cast<FileDataBlock*>(processCommandArg.pWorkBuffer);
        if (block->kind == detail::FILEDATA_MODEL ||
            block->kind == detail::FILEDATA_TEXTURE ||
            (block->kind >= detail::FILEDATA_SHADER_PARAM_ANIM &&
            block->kind <= detail::FILEDATA_MATERIAL_ANIM))
        {
            isResFile = true;
        }
    }

    // ここに処理が来るときには、block が nullptrであることはない
    // nullptr で処理がくる場合はバグの可能性か、
    // ツール側の通信モジュールとバージョンがあっていない場合
    NN_G3D_VIEWER_ASSERT_NOT_NULL(block);

    if (IsConnected())
    {
        if (isResFile) // ファイル処理対象が、ResFile の場合
        {
            bool success = ExecuteResFileLoadCommand(processCommandArg.command, block, m_FileBuffer, m_FileBufferSize);
            if (!success)
            {
                return false;
            }
        }
        else
        {
            bool success = ExecuteShaderLoadCommand(processCommandArg.command, block);
            if (!success)
            {
                return false;
            }
        }
    }

    m_IsFileLoading = false;
    return true;
}

bool
ViewerServer::ViewerServerProxy::Impl::ExecuteResFileLoadCommand(
    CommandFlag command, const detail::FileDataBlock* block, void* pFileBuffer, size_t fileBufferSize) NN_NOEXCEPT
{
    const ResFileData* pResFileData = static_cast<const ResFileData*>(pFileBuffer);
    size_t align = pResFileData->fileHeader.GetAlignment();
    if (!pResFileData->fileHeader.IsEndianValid())
    {
        nn::util::SwapEndian(&align);
    }

    ExecuteModelFileLoadedArg arg;
    arg.key = block->key;
    arg.loadFileArg.pResFile = static_cast<nn::g3d::ResFile*>(m_FileBuffer);
    arg.loadFileArg.fileSize = fileBufferSize;
    arg.loadFileArg.alignment = align;

    NN_G3D_VIEWER_DEBUG_PRINT("ViewerServer::ExecuteFileLoadCommand : %s [Key:%x]\n",
        GetEditCommandString(static_cast<CommandFlag>(command)),
        block->key);
    switch(command)
    {
    case FILEDATA_LOAD_FILE_COMMAND_FLAG:
        {
            bool success = ExecuteLoadFile(arg, block);
            if (!success)
            {
                return false;
            }
        }
        break;
    case FILEDATA_RELOAD_FILE_COMMAND_FLAG:
        {
            bool success = ExecuteReloadFile(block);
            if (!success)
            {
                return false;
            }
        }
        break;
    case EDIT_RECV_ATTACH_COMMAND_FLAG:
        {
            ModelObj* pAttachedModelObj = ViewerKeyManager::GetInstance().FindData<nn::g3d::ModelObj>(arg.key);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pAttachedModelObj);
            if (!m_EditCommandManager.IsAttaching(pAttachedModelObj))
            {
                // アタッチ中にデタッチがコールされたのでアタッチを中断
                NN_G3D_VIEWER_LOG("Canceled attaching %s(name = %s, key = %d, address = 0x%x)\n",
                    block->fileName, pAttachedModelObj->GetName(), arg.key, pAttachedModelObj);
                m_EditCommandManager.QueueErrorCommand(RUNTIME_ERROR_CODE_ATTACH_CANCELED);
                return false;
            }

            nn::g3d::ResFile* pLoadedResFile = GetResourceManager().LoadResFileWithoutCopyAndRegister(&m_FileBuffer, FILEDATA_MODEL);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pLoadedResFile);

            nn::g3d::ResModel* pLoadedResModel = pLoadedResFile->GetModel(0);
            if (strcmp(pAttachedModelObj->GetName(), pLoadedResModel->GetName()) != 0)
            {
                // 異なる名前の中間ファイルがアタッチされたのでアタッチを中断
                NN_G3D_VIEWER_WARNING_LOG("Attached model name mismatch: %s != %s)\n",
                    pAttachedModelObj->GetName(), pLoadedResModel->GetName());
                m_EditCommandManager.QueueErrorCommand(RUNTIME_ERROR_CODE_INVALID_MODEL_ATTACHED);
                return false;
            }

            const ResModel* pAttachedResModel = pAttachedModelObj->GetResource();
            DynamicArray<ModelObj*> attachedModelObjs(m_pAllocator, DefaultAlignment, nullptr);
            {
                ModelObj* pModelObj = m_EditCommandManager.RemoveAttachingModelObj(pAttachedResModel);
                while (pModelObj != nullptr)
                {
                    attachedModelObjs.PushBack(pModelObj);
                    pModelObj = m_EditCommandManager.RemoveAttachingModelObj(pAttachedResModel);
                }
            }

            RuntimeErrorCode errorCode = GetResourceManager().CreateAttachedEditModelObj(
                attachedModelObjs.GetData(), attachedModelObjs.GetCount(), pLoadedResFile);
            if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR)
            {
                m_EditCommandManager.QueueErrorCommand(errorCode);
                return false;
            }
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    GetResourceManager().PrintAllResFiles();
    ClearFileBuffer();
    return true;
}

bool
ViewerServer::ViewerServerProxy::Impl::ExecuteShaderLoadCommand(CommandFlag command, const detail::FileDataBlock* block) NN_NOEXCEPT
{
    size_t align = nn::g3d::detail::Alignment_Default;
    if (block->kind == detail::FILEDATA_SHADER_PROGRAM)
    {
        SwapShaderArchiveAlignSizeFromFileBuffer(&align);
    }

    switch(command)
    {
    case EDIT_RECV_ATTACH_COMMAND_FLAG:
        {
            // シェーダ定義ファイル、ResShaderArchive が送られてくる順番は保証していないので、
            // まずは、EditShaderArchive が作られているか判定
            EditShaderArchive* pEditShaderArchive = GetResourceManager().FindEditShaderArchive(block->key);
            if (pEditShaderArchive == nullptr)
            {
                ResShaderArchive* pResShaderArchive = ViewerKeyManager::GetInstance().FindData<nn::g3d::ResShaderArchive>(block->key);
                NN_G3D_VIEWER_ASSERT_NOT_NULL(pResShaderArchive);
                pEditShaderArchive = GetResourceManager().CreateAttachEditShaderArchive(pResShaderArchive);
                NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pEditShaderArchive, "%s\n", NN_G3D_VIEWER_RES_NAME(pResShaderArchive, GetName()));
                bool result = pEditShaderArchive->Attach();
                NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(pResShaderArchive, GetName()));
                result = GetResourceManager().AddEditShaderArchive(pEditShaderArchive);
                NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(pResShaderArchive, GetName()));

                {
                    ShaderAttachedArg arg;
                    arg.pShaderArchive = pResShaderArchive;
                    m_CallbackCaller.Call(CallbackType_ShaderAttached, &arg);
                }
            }
        }
        break;
    case EDIT_RECV_MODIFIED_SHADER_COMMAND_FLAG:
        {
            EditShaderArchive* pEditShaderArchive = GetResourceManager().FindEditShaderArchive(block->key);
            if (pEditShaderArchive)
            {
                {
                    size_t alignment = 0;
                    SwapShaderArchiveAlignSizeFromFileBuffer(&alignment);

                    ResShaderArchive* pResShaderArchive = GetResourceManager().LoadResShaderArchiveWithoutCopyAndRegister(&m_FileBuffer);
                    NN_G3D_VIEWER_ASSERT_NOT_NULL(pResShaderArchive);

                    pEditShaderArchive->UpdateShaderProgram(block->shadingModelIndex, block->shaderProgramIndex, pResShaderArchive);
                }

                {
                    ShaderProgramUpdatedArg arg;
                    arg.pShaderArchive = pEditShaderArchive->GetTargetResShaderArchive();
                    arg.shadingModelIndex = block->shadingModelIndex;
                    arg.shaderProgramIndex = block->shaderProgramIndex;
                    m_CallbackCaller.Call(CallbackType_ShaderProgramUpdated, &arg);
                }
            }
            else
            {
                NN_G3D_VIEWER_INTERNAL_WARNING("EditShaderArchive for key %d was not found, but received ModifiedShaderCommand\n", block->key);
                ViewerKeyManager::GetInstance().DumpKeyDataMap();
            }
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return true;
}

bool ViewerServer::ViewerServerProxy::Impl::ExecuteMultiFileLoadCommand(CommandMonitor::ProcessCommandArg& processCommandArg) NN_NOEXCEPT
{
    NN_G3D_VIEWER_DEBUG_PRINT("%s\n", __FUNCTION__);
    if (m_pFileDevice->IsReading())
    {
        return true;
    }

    // このコマンド以外は、処理を終了して抜ける
    if (processCommandArg.command != EDIT_LOAD_SHADER_ARCHIVE_COMMAND_FLAG &&
        processCommandArg.command != EDIT_RESET_SHADER_ARCHIVE_COMMAND_FLAG)
    {
        m_MultiFileState = MULTI_FILE_END;
        return true;
    }

    // パケットをリロケート
    {
        ModelOptimizedShaderBlock* block = static_cast<ModelOptimizedShaderBlock*>(processCommandArg.pWorkBuffer);
        RelocatePacketBinPtr(&block->ofsMaterialShaderIndices);
        RelocatePacketBinPtr(&block->multiFile.ofsFileInfo);
    }

    if (m_MultiFileState == MULTI_FILE_START)
    {
        ModelOptimizedShaderBlock* block = static_cast<ModelOptimizedShaderBlock*>(processCommandArg.pWorkBuffer);
        EditModelObj* pEditModelObj = GetResourceManager().FindEditModelObj(block->modelObjKey);
        if (pEditModelObj == nullptr)
        {
            // 対象が見つからない場合はキャンセル
            m_MultiFileState = MULTI_FILE_END;
            return true;
        }

        if (processCommandArg.command == EDIT_LOAD_SHADER_ARCHIVE_COMMAND_FLAG)
        {
            // マテリアル数分格納する領域を確保する
            pEditModelObj->ResizeAndResetMaterialShaderInfoArray(pEditModelObj->GetMaterialCount());
        }

        // マテリアル情報を初期化
        MaterialShaderInfoArrayData* materialShaderInfoArrayData =
            static_cast<MaterialShaderInfoArrayData*>(block->ofsMaterialShaderIndices.Get());
        for (int idxMat = 0; idxMat < static_cast<int>(materialShaderInfoArrayData->numMaterial); ++idxMat)
        {
            MaterialShaderInfoData receivedInfo = materialShaderInfoArrayData->arrayData[idxMat];

            NN_G3D_VIEWER_DEBUG_PRINT(
                "idxMat = %d (%s), shaderArchiveIndex = %d, isOptimized = %s\n",
                idxMat, pEditModelObj->GetResModel()->GetMaterialName(idxMat),
                receivedInfo.shaderArchiveIndex,
                receivedInfo.isOptimized ? "true" : "false");
            ShaderAssignUpdatedArg::MaterialShaderInfo info;
            info.isOptimized = receivedInfo.isOptimized != 0 ? true : false;
            info.isOptimizationSkipped = receivedInfo.isOptimizationSkipped != 0 ? true : false;
            info.pResShaderArchive = nullptr;
            pEditModelObj->SetMaterialShaderInfo(idxMat, info);
        }

        m_MultiFileState = MULTI_FILE_LOOPING;
    }

    bool isLoadingSkipped = false;
    do
    {
        if (m_MultiFileState == MULTI_FILE_LOOPING)
        {
            ModelOptimizedShaderBlock* block = static_cast<ModelOptimizedShaderBlock*>(processCommandArg.pWorkBuffer);
            uint16_t &loopIndex = block->multiFile.loopCount; // 送信された時に roopCount は 0 が入っている前提
            FileInfoData* fileInfoData = static_cast<FileInfoData*>(block->multiFile.ofsFileInfo.Get());
            FileInfoData::OffsetFileInfo* fileInfo = &(fileInfoData->fileInfo[loopIndex]);

            ClearFileBuffer();

            if (fileInfo->fileSize > 0)
            {
                const char* fileName = AddOffset<const char>(block, fileInfo->ofsFileName);

                size_t alignment = fileInfo->fileAlignment;
                m_FileBuffer = m_pAllocator->Allocate(fileInfo->fileSize, alignment, AllocateType_Resource);
                if (m_FileBuffer == nullptr)
                {
                    m_EditCommandManager.QueueErrorCommand(RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY);
                    m_MultiFileState = MULTI_FILE_END;
                    return false;
                }

                m_FileBufferSize = fileInfo->fileSize;

                NN_G3D_VIEWER_DEBUG_PRINT(
                    "Reading file: path = %s, size = %zu, alignment = %zu\n", fileName, fileInfo->fileSize, alignment);

                // キャストしないとビルドエラーになるので
                bool result = m_pFileDevice->Open(fileName, HostFileDeviceBase::OpenFlag_ReadOnly);
                NN_G3D_VIEWER_ASSERT_DETAIL(result, "Opening %s failed\n", fileName);
                result = m_pFileDevice->ReadASync(m_FileBuffer, m_FileBufferSize);
                NN_G3D_VIEWER_ASSERT_DETAIL(result, "Reading %s failed\n", fileName);

                NN_G3D_VIEWER_DEBUG_PRINT("Complete reading\n");
            }
            else
            {
                isLoadingSkipped = true;
            }

            m_MultiFileState = MULTI_FILE_LOADING;
        }

        if (m_MultiFileState == MULTI_FILE_LOADING)
        {
            if (!isLoadingSkipped)
            {
                if (m_pFileDevice->IsReading()) // 読み込み中の場合は処理を一旦抜ける
                {
                    return true;
                }
                m_pFileDevice->Close();
            }

            ModelOptimizedShaderBlock* block = static_cast<ModelOptimizedShaderBlock*>(processCommandArg.pWorkBuffer);
            EditModelObj* pEditModelObj = GetResourceManager().FindEditModelObj(block->modelObjKey);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pEditModelObj);

            uint16_t &loopIndex = block->multiFile.loopCount; // 送信された時に roopCount は 0 が入っている前提
            FileInfoData::OffsetFileInfo* fileInfo = &(
                static_cast<FileInfoData*>(block->multiFile.ofsFileInfo.Get())->fileInfo[loopIndex]);
            MaterialShaderInfoArrayData* materialShaderInfoArrayData = static_cast<MaterialShaderInfoArrayData*>(block->ofsMaterialShaderIndices.Get());

            switch(fileInfo->fileDataKind)
            {
            case FILEDATA_SHADER_ARCHIVE:
                {
                    ResShaderArchive* pResShaderArchive = nullptr;
                    if (m_FileBuffer != nullptr)
                    {
                        pResShaderArchive = GetResourceManager().LoadResShaderArchiveWithoutCopyAndRegister(&m_FileBuffer);
                        NN_G3D_VIEWER_ASSERT_NOT_NULL(pResShaderArchive);
                    }

                    NN_G3D_VIEWER_ASSERT(static_cast<int>(materialShaderInfoArrayData->numMaterial) == pEditModelObj->GetShaderCount());

                    uint16_t &idxShaderArchive = block->multiFile.loopCount;
                    for (int idxMat = 0; idxMat < static_cast<int>(materialShaderInfoArrayData->numMaterial); ++idxMat)
                    {
                        MaterialShaderInfoData receivedInfo = materialShaderInfoArrayData->arrayData[idxMat];
                        if (receivedInfo.shaderArchiveIndex == idxShaderArchive)
                        {
                            ShaderAssignUpdatedArg::MaterialShaderInfo info;
                            info.isOptimized = receivedInfo.isOptimized != 0 ? true : false;
                            info.isOptimizationSkipped = receivedInfo.isOptimizationSkipped != 0 ? true : false;
                            info.pResShaderArchive = pResShaderArchive;
                            pEditModelObj->SetMaterialShaderInfo(idxMat, info);
                        }
                    }
                }
                break;
            case FILEDATA_MODEL:
                {
                    NN_G3D_VIEWER_ASSERT_NOT_NULL(m_FileBuffer);
                    const ResFileData* resFileData = static_cast<const ResFileData*>(m_FileBuffer);
                    size_t align = resFileData->fileHeader.GetAlignment();
                    if (!resFileData->fileHeader.IsEndianValid())
                    {
                        nn::util::SwapEndian(&align);
                    }

                    nn::g3d::ResFile* pReservedResFile = GetResourceManager().LoadResFileWithoutCopyAndRegister(&m_FileBuffer, FILEDATA_MODEL);
                    NN_G3D_VIEWER_ASSERT_NOT_NULL(pReservedResFile);

                    pEditModelObj->SetUpdateResFile(pReservedResFile);
                }
                break;
            default:
                NN_G3D_VIEWER_ASSERT_DETAIL(NN_STATIC_CONDITION(false), "invalid file.\n");
                break;
            }

            ++loopIndex;
            if (loopIndex >= block->multiFile.numFile)
            {
                m_MultiFileState = MULTI_FILE_LOADED_ALL;
            }
            else
            {
                m_MultiFileState = MULTI_FILE_LOOPING;
            }
        }
    } while(m_MultiFileState == MULTI_FILE_LOOPING);

    if (m_MultiFileState == MULTI_FILE_LOADED_ALL)
    {
        ModelOptimizedShaderBlock* block = static_cast<ModelOptimizedShaderBlock*>(processCommandArg.pWorkBuffer);
        EditModelObj* pEditModelObj = GetResourceManager().FindEditModelObj(block->modelObjKey);
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pEditModelObj);

        bool useShaders = true;
        if (processCommandArg.command == EDIT_RESET_SHADER_ARCHIVE_COMMAND_FLAG)
        {
            useShaders = false;
        }

        RuntimeErrorCode errorCode = pEditModelObj->UpdateShaders(useShaders);
        if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR)
        {
            m_EditCommandManager.QueueErrorCommand(errorCode);
            m_MultiFileState = MULTI_FILE_END;
            return false;
        }

        if (processCommandArg.command == EDIT_RESET_SHADER_ARCHIVE_COMMAND_FLAG)
        {
            pEditModelObj->DestroyShaders();
        }

        GetResourceManager().ForceBindExistingTextures(pEditModelObj);

        GetResourceManager().ScheduleDestroyUnusedResFiles();

        m_MultiFileState = MULTI_FILE_END;
    }

    return true;
} // NOLINT (readability/fn_size)

bool
ViewerServer::ViewerServerProxy::Impl::ExecuteLoadFile(const ExecuteModelFileLoadedArg& arg, const FileDataBlock* block) NN_NOEXCEPT
{
    NN_G3D_VIEWER_DEBUG_PRINT("%s\n", __FUNCTION__);
    FileDataKind kind = static_cast<FileDataKind>(block->kind);
    switch(kind)
    {
    case FILEDATA_MODEL:
        {
            // コールバックを実行し、ユーザから管理対象の ResFile を受け取る
            nn::g3d::ResFile* pOriginalResFile;
            {
                ResFile* pArgResFile = arg.loadFileArg.pResFile;
                NN_G3D_VIEWER_DEBUG_PRINT(
                    "Loaded Model ResFile: name = %s, size = %zu, alignment = %zu\n",
                    pArgResFile->ToData().fileHeader.GetFileName() != nullptr?
                        pArgResFile->ToData().fileHeader.GetFileName().data() : "nullptr",
                    pArgResFile->ToData().fileHeader.GetFileSize(),
                    pArgResFile->ToData().fileHeader.GetAlignment());

                ModelFileLoadedArg inArg;
                inArg.alignment = arg.loadFileArg.alignment;
                inArg.fileSize = arg.loadFileArg.fileSize;
                inArg.pResFile = pArgResFile;
                ModelFileLoadedOutArg outArg;
                outArg.pResFile = nullptr;
                m_CallbackCaller.Call(&outArg, &inArg, CallbackType_ModelFileLoaded);
                pOriginalResFile = outArg.pResFile;
                if (pOriginalResFile == nullptr)
                {
                    // ModelFileLoaded コールバックの出力引数が設定されていない
                    // メモリ不足か、アプリケーション側のコールバックの実装の問題
                    m_EditCommandManager.QueueErrorCommand(RUNTIME_ERROR_CODE_LOAD_FILE_FAILED);
                    return false;
                }

                GetResourceManager().BindExistingTextures(pOriginalResFile);
            }

            nn::g3d::ResFile* pFirstLoadResFile = GetResourceManager().LoadResFileWithoutCopyAndRegister(&m_FileBuffer, FILEDATA_MODEL);
            NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pFirstLoadResFile, "%s", arg.loadFileArg.pResFile->GetModelName(0));

            EditModelObj* pEditModelObj = nullptr;
            RuntimeErrorCode errorCode = GetResourceManager().CreateEditModelObj(&pEditModelObj, pOriginalResFile);
            if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR)
            {
                {
                    ModelFileUnloadedArg unloadArg;
                    unloadArg.pResFile = pOriginalResFile;
                    m_CallbackCaller.Call(CallbackType_ModelFileUnloaded, &unloadArg);
                }

                m_EditCommandManager.QueueErrorCommand(errorCode);
                return false;
            }

            bool success;
            nn::g3d::ResModel* pOriginalResModel = pOriginalResFile->GetModel(0);

            ViewerKeyType originalResFileKey = ViewerKeyManager::GetInstance().Register(pOriginalResFile, ViewerKeyManager::DataType_ModelResFile);
            ViewerKeyType originalResModelKey = ViewerKeyManager::GetInstance().FindKey(pOriginalResModel);
            if (originalResModelKey == InvalidKey)
            {
                // ユーザが ModelFileLoaded コールバックの外で遅延初期化(アタッチ)する場合は、ここでキーを登録する
                // コールバック内でアタッチした場合は、アタッチ API の中でキーが登録されている
                originalResModelKey = ViewerKeyManager::GetInstance().Register(pOriginalResModel, ViewerKeyManager::DataType_ResModel);
            }

            pEditModelObj->SetOriginalResFileManagedByUser(pOriginalResFile);

            // ユーザが ModelObj をアタッチするまで、3DEditor から送られてきた ResFile をアタッチすることができないので、
            // キーをコマンドに記録しておいて、後でアタッチするときに参照する
            ViewerKeyType firstLoadedModelFileKey = ViewerKeyManager::GetInstance().FindKey(pFirstLoadResFile);
            NN_G3D_VIEWER_ASSERT(firstLoadedModelFileKey != InvalidKey);

            success = m_EditCommandManager.AddModelFileLoadedQueue(originalResFileKey, originalResModelKey, firstLoadedModelFileKey, arg.key);
            NN_G3D_VIEWER_ASSERT(success);

            NN_G3D_VIEWER_DEBUG_PRINT("Loaded Model: %s\n", GetResName(pOriginalResFile, kind));
        }
        break;
    case FILEDATA_TEXTURE:
        {
            nn::gfx::ResTextureFile* pResTextureFile = GetResourceManager().LoadResTextureFileWithoutCopyAndRegister(&m_FileBuffer);
            NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pResTextureFile, "%s", pResTextureFile->GetResTexture(0)->GetName());

            GetResourceManager().ForceBindTextureForAllObjs(pResTextureFile);

            ViewerKeyType textureKey = ViewerKeyManager::GetInstance().FindKey(pResTextureFile);
            NN_G3D_VIEWER_ASSERT_VALID_KEY(textureKey);
            bool result = m_EditCommandManager.AddFileLoadedQueue(kind, textureKey, arg.key);
            NN_G3D_VIEWER_ASSERT(result);

            NN_G3D_VIEWER_DEBUG_PRINT("Loaded Texture Name: %s\n", pResTextureFile->GetResTexture(0)->GetName());
        }
        break;
    case FILEDATA_SHADER_PARAM_ANIM:
    case FILEDATA_SKELETAL_ANIM:
    case FILEDATA_MAT_COLOR_ANIM:
    case FILEDATA_TEXTURE_SRT_ANIM:
    case FILEDATA_TEXTURE_PATTERN_ANIM:
    case FILEDATA_BONE_VISIBILITY_ANIM:
    case FILEDATA_MAT_VISIBILITY_ANIM:
    case FILEDATA_SHAPE_ANIM:
    case FILEDATA_MATERIAL_ANIM:
        {
            // アニメーションはリソースだけここで用意
            // モデルバインド時にEditAnimObjを作る
            nn::g3d::ResFile* pResFile = GetResourceManager().LoadAnimResFileWithoutCopyAndRegister(&m_FileBuffer, kind);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pResFile);

            EditAnimObj* pEditAnimObj = GetResourceManager().AddEditAnimObj(pResFile, kind);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pEditAnimObj);

            GetResourceManager().ForceBindExistingTextures(pEditAnimObj);

            ViewerKeyType resFileKey = ViewerKeyManager::GetInstance().FindKey(pResFile);
            NN_G3D_VIEWER_ASSERT_VALID_KEY(resFileKey);
            bool result = m_EditCommandManager.AddFileLoadedQueue(kind, resFileKey, arg.key);
            NN_G3D_VIEWER_ASSERT(result);

            NN_G3D_VIEWER_DEBUG_PRINT("Loaded Anim Name: %s\n", GetResName(pResFile, kind));
        }
        break;
    case FILEDATA_SCENE_ANIM:
        {
            nn::g3d::ResFile* pResFile = GetResourceManager().LoadResFileWithoutCopyAndRegister(&m_FileBuffer, kind);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pResFile);

            EditSceneAnimObj* pEditSceneAnimObj = GetResourceManager().AddEditSceneAnimObj(pResFile);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pEditSceneAnimObj);

            bool result = m_EditCommandManager.AddSceneAnimFileLoadedQueue(pEditSceneAnimObj, arg.key);
            NN_G3D_VIEWER_ASSERT(result);

            NN_G3D_VIEWER_DEBUG_PRINT("Loaded Scene Anim Name: %s\n", GetResName(pResFile, kind));
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return true;
} // NOLINT (readability/fn_size)

bool
ViewerServer::ViewerServerProxy::Impl::ExecuteReloadFile(const FileDataBlock* block) NN_NOEXCEPT
{
    NN_G3D_VIEWER_DEBUG_PRINT("%s\n", __FUNCTION__);
    FileDataKind kind = static_cast<FileDataKind>(block->kind);

    switch(block->kind)
    {
    case FILEDATA_MODEL:
        {
            nn::g3d::ResFile* pResFile = GetResourceManager().LoadResFileWithoutCopyAndRegister(&m_FileBuffer, kind);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pResFile);

            EditModelObj* pEditModelObj = GetResourceManager().FindEditModelObj(block->key);
            if (pEditModelObj == nullptr)
            {
                m_EditCommandManager.QueueErrorCommand(RUNTIME_ERROR_CODE_TARGET_MODEL_NOT_FOUND);
                return false;
            }

            NN_G3D_VIEWER_DEBUG_PRINT("Reloading Model Name: %s\n", GetResName(pResFile, kind));
            RuntimeErrorCode errorCode = pEditModelObj->ReloadResFile(pResFile);
            if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR)
            {
                m_EditCommandManager.QueueErrorCommand(errorCode);
                m_EditCommandManager.ScheduleDestructEditModelObj(block->key);
                return false;
            }

            GetResourceManager().ForceBindExistingTextures(pEditModelObj);

            GetResourceManager().ScheduleDestroyUnusedResFiles();
        }
        break;
    case FILEDATA_TEXTURE:
        {
            nn::gfx::ResTextureFile* pResTextureFile = GetResourceManager().LoadResTextureFileWithoutCopy(&m_FileBuffer);
            NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pResTextureFile, "%s", pResTextureFile->GetResTexture(0)->GetName());

            NN_G3D_VIEWER_DEBUG_PRINT("Reloading Texture Name: %s\n", pResTextureFile->GetResTexture(0)->GetName());
            NN_G3D_VIEWER_DEBUG_PRINT("Reload key = %u, resFileKey = %u\n", block->key, block->resFileKey);
            RuntimeErrorCode errorCode = m_EditCommandManager.ReloadTextureResource(block->resFileKey, pResTextureFile);
            if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR)
            {
                NN_G3D_VIEWER_INTERNAL_WARNING("Reloading %s failed\n", pResTextureFile->GetResTexture(0)->GetName());
                m_EditCommandManager.QueueErrorCommand(errorCode);
                return false;
            }
        }
        break;
    case FILEDATA_SHADER_PARAM_ANIM:
    case FILEDATA_MAT_COLOR_ANIM:
    case FILEDATA_TEXTURE_SRT_ANIM:
    case FILEDATA_TEXTURE_PATTERN_ANIM:
    case FILEDATA_SKELETAL_ANIM:
    case FILEDATA_BONE_VISIBILITY_ANIM:
    case FILEDATA_MAT_VISIBILITY_ANIM:
    case FILEDATA_SHAPE_ANIM:
    case FILEDATA_MATERIAL_ANIM:
        {
            nn::g3d::ResFile* pResFile = GetResourceManager().LoadAnimResFileWithoutCopy(&m_FileBuffer);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pResFile);

            NN_G3D_VIEWER_DEBUG_PRINT("Reloading Anim Name: %s\n", GetResName(pResFile, kind));

            RuntimeErrorCode errorCode = m_EditCommandManager.ReloadAnimResource(block->resFileKey, pResFile, kind);
            if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR)
            {
                NN_G3D_VIEWER_INTERNAL_WARNING("Reloading %s failed\n", GetResName(pResFile, kind));
                m_EditCommandManager.QueueErrorCommand(errorCode);
                return false;
            }

            ViewerKeyType resFileKey = ViewerKeyManager::GetInstance().FindKey(pResFile);
            NN_G3D_VIEWER_ASSERT_VALID_KEY(resFileKey);
            EditAnimObj* pEditAnimObj = GetResourceManager().FindEditAnimObj(resFileKey);
            GetResourceManager().ForceBindExistingTextures(pEditAnimObj);
        }
        break;
    case FILEDATA_SCENE_ANIM:
        {
            nn::g3d::ResFile* resFile = GetResourceManager().LoadResFileWithoutCopyAndRegister(&m_FileBuffer, kind);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(resFile);

            NN_G3D_VIEWER_DEBUG_PRINT("Reloading Scene Anim Name: %s\n", GetResName(resFile, kind));

            RuntimeErrorCode errorCode = m_EditCommandManager.ReloadEditSceneAnimObj(block->resFileKey, resFile);
            if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR)
            {
                NN_G3D_VIEWER_INTERNAL_WARNING("Reloading %s failed\n", GetResName(resFile, kind));
                m_EditCommandManager.QueueErrorCommand(errorCode);
                return false;
            }
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return true;
}

void
ViewerServer::ViewerServerProxy::Impl::SwapShaderArchiveAlignSizeFromFileBuffer(size_t* outAlignSize) const NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(outAlignSize);
    *outAlignSize = nn::g3d::detail::Alignment_Default;

    const ResShaderFile* pResShaderFile = static_cast<const ResShaderFile*>(m_FileBuffer);
    *outAlignSize = pResShaderFile->GetFileHeader()->GetAlignment();
    if (!pResShaderFile->GetFileHeader()->IsEndianValid())
    {
        nn::util::SwapEndian(outAlignSize);
    }
}

void
ViewerServer::ViewerServerProxy::Impl::ClearState() NN_NOEXCEPT
{
    m_IsFileLoading = false;

    if (m_pFileDevice->IsFileOpened())
    {
        m_pFileDevice->Close();
    }

    m_MultiFileState = MULTI_FILE_END;

    m_EditCommandManager.ClearEditCommandManager();
    if (m_pFileDevice->IsFileOpened())
    {
        m_pFileDevice->Close();
    }

    ClearFileBuffer();
}

}}} // namespace nn::g3d::viewer


