﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/viewer/g3d_ViewerDefine.h>

#include "util/g3d_SynchronizedMap.h"

namespace nn { namespace g3d { namespace viewer { namespace detail {

class ViewerKeyManager
{
    NN_DISALLOW_COPY(ViewerKeyManager);
public:
    enum DataType
    {
        DataType_ModelObj,
        DataType_ResModel,
        DataType_ModelResFile,
        DataType_ResTextureFile,
        DataType_AnimResFile,
        DataType_SceneAnimResFile,
        DataType_ResShaderFile,
        DataType_ResShaderArchive,
        DataType_NotSpecified,
    };

    static const char* GetDataTypeAsString(DataType type) NN_NOEXCEPT
    {
        switch (type)
        {
        case DataType_ModelObj: return "ModelObj";
        case DataType_ResModel: return "ResModel";
        case DataType_ModelResFile: return "ModelResFile";
        case DataType_ResTextureFile: return "ResTextureFile";
        case DataType_AnimResFile: return "AnimResFile";
        case DataType_SceneAnimResFile: return "SceneAnimResFile";
        case DataType_ResShaderFile: return "ResShaderFile";
        case DataType_ResShaderArchive: return "ResShaderArchive";
        case DataType_NotSpecified: return "NotSpecified";
        default: return "Unknown";
        }
    }

    // TODO: 移行円滑化のために static にしているので、いつか static でなくす
    static void Initialize(Allocator* pAllocator) NN_NOEXCEPT;
    static ViewerKeyManager& GetInstance() NN_NOEXCEPT;
    static void Finalize() NN_NOEXCEPT;
    static bool IsInitialized() NN_NOEXCEPT;

    template<typename T>
    ViewerKeyType Register(T* pData) NN_NOEXCEPT
    {
        return Register(pData, DataType_NotSpecified);
    }

    template<typename T>
    ViewerKeyType Register(T* pData, DataType dataType) NN_NOEXCEPT
    {
        ScopedLock lock(m_KeyDataMap);
        ViewerKeyType key = GetUniqueKey();
        if (key == InvalidKey)
        {
            return InvalidKey;
        }

        bool success = m_KeyDataMap.Register(key, reinterpret_cast<const void*>(pData));
        if (!success)
        {
            return InvalidKey;
        }

        success = m_KeyDataTypeMap.Register(key, dataType);
        if (!success)
        {
            m_KeyDataMap.Unregister(key);
            return InvalidKey;
        }

        NN_G3D_VIEWER_DEBUG_PRINT(
            "Key %d has been registered for data address 0x%p(%s)\n",
            key, pData, GetDataTypeAsString(dataType));

        return key;
    }

    template<typename T>
    bool Unregister(T* pData) NN_NOEXCEPT
    {
        ScopedLock lock(m_KeyDataMap);
        ViewerKeyType key = m_KeyDataMap.FindKey(reinterpret_cast<const void*>(pData));
        if (key == InvalidKey)
        {
            NN_G3D_VIEWER_DEBUG_PRINT("Failed to unregister: data = 0x%p\n", pData);
            return false;
        }

        return UnregisterImpl(key);
    }

    bool Unregister(ViewerKeyType key) NN_NOEXCEPT
    {
        ScopedLock lock(m_KeyDataMap);
        return UnregisterImpl(key);
    }

    //! @biref 指定したキーに紐づくデータを更新します。
    template<typename T>
    bool UpdateData(ViewerKeyType key, T* pData) NN_NOEXCEPT
    {
        return m_KeyDataMap.UpdateValue(key, pData);
    }

    template<typename T>
    bool IsRegistered(T* pData) const NN_NOEXCEPT
    {
        return m_KeyDataMap.IsValueRegistered(reinterpret_cast<const void*>(pData));
    }

    bool IsKeyRegistered(ViewerKeyType key) const NN_NOEXCEPT
    {
        return m_KeyDataMap.IsKeyRegistered(key);
    }

    template<typename T>
    ViewerKeyType FindKey(const T* pData) const NN_NOEXCEPT
    {
        void* pCasted = reinterpret_cast<void*>(const_cast<T*>(pData));
        return m_KeyDataMap.FindKey(pCasted);
    }

    template<typename T>
    T* FindData(ViewerKeyType key) NN_NOEXCEPT
    {
        return static_cast<T*>(const_cast<void*>(m_KeyDataMap.FindValue(key)));
    }

    template<typename T>
    const T* FindData(ViewerKeyType key) const NN_NOEXCEPT
    {
        return reinterpret_cast<const T*>(m_KeyDataMap.FindValue(key));
    }

    void UnregisterAllKeys() NN_NOEXCEPT
    {
        ScopedLock lock(m_KeyDataMap);
        while (m_KeyDataMap.GetCount() > 0)
        {
            ViewerKeyType key = m_KeyDataMap.GetKey(0);
            m_KeyDataMap.Unregister(key);
        }

        // TODO: ランタイムだけであればキーの生成値をここでリセットできるが、
        // 3DEditor が切断時もシェーダ編集のキーを保持しているためリセットできない？要確認
        // m_LastGeneratedKey = InitialKey;
    }

    int GetRegisteredKeyCount() const NN_NOEXCEPT
    {
        return m_KeyDataMap.GetCount();
    }

    ViewerKeyType GetKey(int index) const NN_NOEXCEPT
    {
        return m_KeyDataMap.GetKey(index);
    }

    DataType FindDataType(ViewerKeyType key) const NN_NOEXCEPT
    {
        return m_KeyDataTypeMap.FindValue(key);
    }

    void DumpKeyDataMap() const NN_NOEXCEPT
    {
        NN_G3D_VIEWER_LOG("Key Data Map:\n");
        for (int i = 0, count = m_KeyDataMap.GetCount(); i < count; ++i)
        {
            NN_G3D_VIEWER_LOG(
                "[%d] key = %d, data address = 0x%p\n", i, m_KeyDataMap.GetKey(i), m_KeyDataMap.GetValue(i));
        }
    }

private:
    explicit ViewerKeyManager(Allocator* pAllocator) NN_NOEXCEPT
        : m_KeyDataMap(pAllocator, InvalidKey, nullptr)
        , m_KeyDataTypeMap(pAllocator, InvalidKey, DataType_NotSpecified)
        , m_pAllocator(pAllocator)
        , m_LastGeneratedKey(InitialKey)
    {
    }

    ViewerKeyType GetUniqueKey() NN_NOEXCEPT
    {
        for (ViewerKeyType key = m_LastGeneratedKey + 1; key < INT_MAX; ++key)
        {
            if (key == InvalidKey)
            {
                continue;
            }

            const void* pData = m_KeyDataMap.FindValue(key);
            if (pData == nullptr)
            {
                m_LastGeneratedKey = key;
                return key;
            }
        }

        return InvalidKey;
    }

    bool UnregisterImpl(ViewerKeyType key) NN_NOEXCEPT
    {
        DataType dataType = m_KeyDataTypeMap.FindValue(key);
        NN_UNUSED(dataType);
        bool success = m_KeyDataMap.Unregister(key);
        success &= m_KeyDataTypeMap.Unregister(key);
        if (!success)
        {
            NN_G3D_VIEWER_DEBUG_PRINT("Failed to unregister: key = %d\n", key);
            return false;
        }

        NN_G3D_VIEWER_DEBUG_PRINT("Key %d has been unregistered(%s)\n", key, GetDataTypeAsString(dataType));

        return true;
    }

private:
    static const ViewerKeyType InitialKey = 0;
    SynchronizedMap<ViewerKeyType, const void*> m_KeyDataMap;
    SynchronizedMap<ViewerKeyType, DataType> m_KeyDataTypeMap;
    Allocator* m_pAllocator;
    ViewerKeyType m_LastGeneratedKey;
};

}}}} // namespace nn::g3d::viewer::detail


