﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditPickup.h"
#include "g3d_EditSocket.h"
#include "util/g3d_ViewerUtility.h"
#include "g3d_ViewerDetailDefine.h"
#include "g3d_ViewerKeyManager.h"

#include <nn/g3d/detail/g3d_Inlines.h>

namespace nn { namespace g3d { namespace viewer { namespace detail {

ViewerResult
EditPickup::PushMaterialPickup(const ModelObj* modelObj, int materialIndex) NN_NOEXCEPT
{
    size_t bufferSize = sizeof(MaterialPickup);
    void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_Communication);
    if (buffer == nullptr)
    {
        return ViewerResult_MemoryAllocationFailed;
    }

    // マテリアルピックアップ登録作業
    {
        MaterialPickup* pickup = new (buffer) MaterialPickup(modelObj, materialIndex);
        if (!m_MaterialPickups.PushBack(pickup))
        {
            m_pAllocator->Free(pickup);
            return ViewerResult_MemoryAllocationFailed;
        }
    }
    return ViewerResult_Success;
}

void
EditPickup::ClearPickup() NN_NOEXCEPT
{
    ScopedLock scopedLock(m_MaterialPickups);
    {
        int size = m_MaterialPickups.GetCount();
        for (int i = 0;i < size; ++i)
        {
            MaterialPickup* pickup = m_MaterialPickups.UnsafeAt(i);
            m_pAllocator->Free(pickup);
        }
        m_MaterialPickups.Clear();
    }
}

bool
EditPickup::MakePickupPacket() NN_NOEXCEPT
{
    size_t bufferSize = CalculateWorkBufferSize();
    if (bufferSize == 0)
    {
        return false;
    }

    if (!m_WorkBuffer.Resize(bufferSize))
    {
        return false;
    }

    m_PickupBufferSize = bufferSize;
    MakePickupPacketBuffer();
    return true;
}

bool
EditPickup::SendPickup(EditSocketBase* socket) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(socket);

    return socket->WriteSync(m_WorkBuffer.GetWorkBufferPtr(), m_PickupBufferSize);
}

size_t
EditPickup::CalculateWorkBufferSize() NN_NOEXCEPT
{
    int materialPickupCount = m_MaterialPickups.GetCount();

    if (materialPickupCount <= 0)
    {
        return 0;
    }

    size_t size = sizeof(PacketHeader);
    size += sizeof(PickupSendInfo);

    size_t materialPickupBufferSize = sizeof(MaterialPickup) * materialPickupCount;
    size += materialPickupBufferSize;

    return size;
}

void
EditPickup::MakePickupPacketBuffer() NN_NOEXCEPT
{
    ScopedLock scopedLock(m_MaterialPickups);

    void* buffer = m_WorkBuffer.GetWorkBufferPtr();
    NN_G3D_VIEWER_ASSERT_NOT_NULL(buffer);

    PacketHeader* header = static_cast<PacketHeader*>(buffer);
    header->magic = NN_G3D_EDIT_MAGIC;
    header->verWord = NN_G3D_EDIT_VERSION;
    header->command = PICK_TOOL_MATERIAL_COMMAND_FLAG;
    // TODO: 後でsize_tに直す
    header->dataSize = static_cast<int>(m_PickupBufferSize - sizeof(PacketHeader));

    PickupSendInfo* sendInfo = AddOffset<PickupSendInfo>(buffer, sizeof(PacketHeader));

    int materialPickupCount = m_MaterialPickups.GetCount();

    sendInfo->materialPickupNum = materialPickupCount;
    sendInfo->padding[0] = 0;
    sendInfo->padding[1] = 0;
    sendInfo->padding[2] = 0;

    size_t materialPickupInfoSize = sizeof(MaterialPickupInfo);
    MaterialPickupInfo* firstInfo = AddOffset<MaterialPickupInfo>(sendInfo, sizeof(PickupSendInfo));
    {
        for (int i = 0; i < materialPickupCount; ++i)
        {
            MaterialPickupInfo* info = AddOffset<MaterialPickupInfo>(firstInfo, i * materialPickupInfoSize);
            MaterialPickup* pickup = m_MaterialPickups.UnsafeAt(i);

            info->modelObjKey = ViewerKeyManager::GetInstance().FindKey(pickup->m_pModelObj);
            info->materialIndex = pickup->m_MaterialIndex;
        }
    }
}

}}}} // namespace nn::g3d::viewer::detail


