﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/viewer/g3d_ViewerDefine.h>

#include "g3d_CallbackCaller.h"
#include "util/g3d_SynchronizedDynamicPtrArray.h"
#include "util/g3d_DynamicArray.h"
#include "util/g3d_SynchronizedDynamicRingBuffer.h"
#include "util/g3d_DynamicLengthString.h"
#include "util/g3d_ViewerUtility.h"
#include "command/g3d_IViewerCommand.h"
#include "g3d_ViewerKeyManager.h"
#include "command/g3d_CommandTypes.h"
#include "command/g3d_FileLoadedCommand.h"

#include <nn/g3d/viewer/g3d_ViewerServer.h>
#include <nn/g3d/viewer/g3d_ViewerCallback.h>
#include <nn/gfx/gfx_Core.h>
#include <nn/gfx/gfx_Types.h>
#include <nn/g3d/g3d_ResCommon.h>

namespace nn { namespace g3d {

class ModelObj;
class ModelAnimObj;

class ResModel;
class ResShaderArchive;
class ResFile;
class ResShaderFile;

namespace viewer {

class ViewerServer;

namespace detail {

class Allocator;
class EditSocketBase;
class EditModelObj;
class EditAnimObj;
class EditShaderParamAnimObj;
class EditTexPatternAnimObj;
class EditMatVisibilityAnimObj;
class EditBoneVisibilityAnimObj;
class EditSceneAnimObj;
class EditShaderArchive;
class EditShapeAnimObj;
class EditMaterialAnimObj;
class AttachCommand;
class CommandMonitor;
class SendRenderInfoDefinitionCommand;
class ResourceManager;

class EditCommandManager
{
    NN_DISALLOW_COPY(EditCommandManager) NN_NOEXCEPT;

    friend class AttachCommand;
public:
    EditCommandManager(nn::gfx::Device* pDevice, Allocator* allocator, CallbackCaller* editCallback) NN_NOEXCEPT;
    ~EditCommandManager() NN_NOEXCEPT;

    void SetCodePage(int codePage) NN_NOEXCEPT
    {
        m_CodePage = codePage;
    }

    bool IsAttaching(const ModelObj* pTarget) const NN_NOEXCEPT
    {
        // キューにたまっているものか判定
        // nullptrの場合は、遅延処理に積まれてないので存在しない。
        if (FindSameAttachModelObjQueue(pTarget) != nullptr)
        {
            return true;
        }

        return m_AttachingModelObjArray.IndexOf(pTarget) >= 0;
    }

    // @brief 対象をアタッチ予約リストから削除して次のイテレータを返します。
    Iter<ModelObj*> RemoveAttachingModelObj(const ModelObj* pTarget) NN_NOEXCEPT
    {
        int index = m_AttachingModelObjArray.IndexOf(pTarget);
        NN_G3D_VIEWER_ASSERT(index >= 0);
        return m_AttachingModelObjArray.Erase(m_AttachingModelObjArray.Begin() + index);
    }

    // @brief 対象をアタッチ予約リストから削除して削除された ModelObj を返します。
    ModelObj* RemoveAttachingModelObj(const ResModel* pResModel) NN_NOEXCEPT;

    // @brief 対象のリソースを保有する ModelObj をアタッチ予約リストからすべて削除します。
    void RemoveAttachingModelObjs(const ResModel* pResModel) NN_NOEXCEPT;

    void ScheduleDestructEditModelObj(ViewerKeyType key) NN_NOEXCEPT;

    int GetActiveEditAnimIndex(const ModelObj* modelObj, int boundAnimIndex) const NN_NOEXCEPT;
    int GetModelAnimBoundCount(const ModelObj* modelObj) const NN_NOEXCEPT;

    ResourceManager* GetResourceManager()
    {
        return m_pResourceManager;
    }

    const ResourceManager* GetResourceManager() const
    {
        return m_pResourceManager;
    }

    ViewerResult AddModelLayoutQueue(
        ViewerKeyType modelKey,
        const nn::util::Vector3fType& scale,
        const nn::util::Vector3fType& rotate,
        const nn::util::Vector3fType& translate) NN_NOEXCEPT;

    bool AddModelFileLoadedQueue(ViewerKeyType resFileKey, ViewerKeyType resModelKey, ViewerKeyType firstLoadedModelFileKey, ViewerKeyType toolKey) NN_NOEXCEPT;

    bool AddFileLoadedQueue(FileDataKind kind, ViewerKeyType resFileKey, ViewerKeyType toolKey) NN_NOEXCEPT;
    RuntimeErrorCode ReloadAnimResource(ViewerKeyType resFileKey, nn::g3d::ResFile* resFile, FileDataKind kind) NN_NOEXCEPT;
    RuntimeErrorCode ReloadTextureResource(ViewerKeyType resFileKey, nn::gfx::ResTextureFile* pResTextureFile) NN_NOEXCEPT;
    bool AddAnimFileReloadedQueue(FileDataKind kind, ViewerKeyType newResFileKey, ViewerKeyType oldResFileKey) NN_NOEXCEPT;
    bool AddTextureFileReloadedQueue(ViewerKeyType newResFileKey, ViewerKeyType oldResFileKey) NN_NOEXCEPT;

    RuntimeErrorCode ReloadEditSceneAnimObj(ViewerKeyType resFileKey, nn::g3d::ResFile* resFile) NN_NOEXCEPT;
    bool AddSceneAnimFileLoadedQueue(const EditSceneAnimObj* editSceneAnimObj, ViewerKeyType toolKey) NN_NOEXCEPT;
    bool AddSceneAnimFileReloadedQueue(const EditSceneAnimObj* editAnimObj, ViewerKeyType resFileKey) NN_NOEXCEPT;

    bool IsShaderArchiveAttaching(const nn::g3d::ResShaderArchive* pResShaderArchive) const NN_NOEXCEPT;

    void Reset() NN_NOEXCEPT;
    void ClearEditCommandManager() NN_NOEXCEPT;

    void DeleteAll() NN_NOEXCEPT;
    void ForceDeleteAll() NN_NOEXCEPT;

    ViewerResult QueueAttachModel(nn::g3d::ModelObj** targetModel, int modelCount, const char* path) NN_NOEXCEPT;
    ViewerResult QueueDetachModel(const nn::g3d::ModelObj* targetModel) NN_NOEXCEPT;
    bool QueueDeleteEditModelObj(EditModelObj* editModelObj) NN_NOEXCEPT;

    void ExecuteThreadUnsafeCommands() NN_NOEXCEPT;
    void ExecuteThreadSafeCommands(EditSocketBase* pSocket) NN_NOEXCEPT;

    ViewerResult QueueAttachShaderArchive(nn::g3d::ResShaderArchive* resShaderArchive, const char* path) NN_NOEXCEPT;
    ViewerResult QueueDetachShaderArchive(const nn::g3d::ResShaderArchive* resShaderArchive) NN_NOEXCEPT;
    bool QueueDeleteEditShaderArchive(EditShaderArchive* editShaderArchive) NN_NOEXCEPT;
    bool QueueErrorCommand(RuntimeErrorCode errorCode) NN_NOEXCEPT;

    SendRenderInfoDefinitionCommand* QueueSendRenderInfoDefinitionCommand(CommandMonitor* pCommandMonitor) NN_NOEXCEPT;

    //! @brief ユーザメッセージを送信するコマンドをエンキューします。
    ViewerResult QueueNotifyMessageCommand(
        const nn::util::string_view& message,
        nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageType messageType,
        nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageDestination messageDestination) NN_NOEXCEPT;

    //! @brief 破棄予定の未参照リソースの破棄を試みます。
    void TryDestroyUnreferencedResources() NN_NOEXCEPT;

    //! @brief animResFileKeyを持つアニメーションのリターゲッティングホストモデルを設定します。
    void SetRetargetingHostModel(ViewerKeyType animResFileKey, ViewerKeyType retargetHostModelObjKey) NN_NOEXCEPT;

    //! @brief animResFileKeyを持つアニメーションのリターゲッティングを解除します。
    void UnsetRetargetingHostModel(ViewerKeyType animResFileKey) NN_NOEXCEPT;

    void SetPlayMotionMirroringEnabled(ViewerKeyType animResFileKey, bool isEnabled) NN_NOEXCEPT;

    bool AddModelFileReloadedQueue(const EditModelObj* editModelObj, ViewerKeyType resFileKey) NN_NOEXCEPT;

    void ExecuteCommandQueueImpl(EditSocketBase* pSocket) NN_NOEXCEPT;

    void UnloadResTextureFile(ViewerKeyType resFileKey) NN_NOEXCEPT;

private:

    AttachCommand* FindSameAttachResModelCommand(const ResModel* pResModel) NN_NOEXCEPT;

    AttachCommand* FindSameAttachModelObjQueue(const ModelObj* modelObj) const NN_NOEXCEPT;
    AttachCommand* FindSameAttachShaderArchiveQueue(const nn::g3d::ResShaderArchive* resShaderArchive) const NN_NOEXCEPT;

    void ExecuteAttachCommandQueueThreadUnsafe() NN_NOEXCEPT;
    void ExecuteAttachCommandQueue(EditSocketBase* pSocket) NN_NOEXCEPT;
    int FindFileLoadedCommandIndex(ViewerKeyType resModelKey) NN_NOEXCEPT
    {
        for (int commandIndex = 0, fileBlockEnd = m_CommandQueue.GetCount(); commandIndex < fileBlockEnd; ++commandIndex)
        {
            IViewerCommand* pCommand = m_CommandQueue[commandIndex];
            if (pCommand->Type() != static_cast<int>(CommandType_FileLoaded))
            {
                continue;
            }

            FileLoadedCommand* pFileLoadedCommand = static_cast<FileLoadedCommand*>(pCommand);
            if (resModelKey == static_cast<ViewerKeyType>(pFileLoadedCommand->GetResModelKey()))
            {
                return commandIndex;
            }
        }

        return -1;
    }

private:
    volatile bool m_IsWriteStarted;
    nn::gfx::Device* m_pDevice;
    Allocator* m_pAllocator;
    CallbackCaller* m_pViewerCallback;
    int m_CodePage;

    SynchronizedDynamicRingBuffer<IViewerCommand*> m_CommandQueue;

    // アタッチコマンドは通信スレッドだけでは処理できないのでその他のコマンドとはキューを分ける
    SynchronizedDynamicRingBuffer<AttachCommand*> m_AttachCommandQueue;

    // デタッチが来た時にアタッチを中断できるようにアタッチ中のモデルを保持しておく
    DynamicPtrArray<ModelObj> m_AttachingModelObjArray;

    ResourceManager* m_pResourceManager;
};

}}}} // namespace nn::g3d::edit::detail


