﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/viewer/g3d_ViewerDefine.h>



#include <nn/g3d/viewer/detail/g3d_ViewerPacketDefine.h>
#include "g3d_EditSocket.h"
#include "util/g3d_EditWorkBuffer.h"
#include "g3d_Allocator.h"

namespace nn {
namespace g3d {
namespace viewer {
namespace detail {

class CommandMonitor
{
    NN_DISALLOW_COPY(CommandMonitor);

public:
    struct InitArg
    {
        nn::g3d::viewer::detail::Allocator*         pAllocator;
        explicit InitArg() NN_NOEXCEPT
            : pAllocator(nullptr)
        {}
    };

    struct ProcessCommandArg
    {
        CommandFlag command;
        void* pWorkBuffer;
        int workBufferSize;
    };

    struct AnalyzeCommandArg
    {
        CommandFlag command;
        void* pWorkBuffer;
        int workBufferSize;
    };

    enum AnalyzeCommandResult
    {
        // 失敗しました
        AnalyzeCommandResult_Error,

        // ProcessCommand を必要としない場合(リソースの編集などメインスレッドと排他的な操作が必要ない場合)、
        // コマンドの解析時に必要な処理を終わらせて、処理を終了します(メインスレッドの処理待ちがなくなります)
        AnalyzeCommandResult_Finish,

        // ProcessCommand を次に実行することを要求します。
        AnalyzeCommandResult_RequestProcessCommand,
    };

    enum ProcessCommandResult
    {
        ProcessCommandResult_Error,
        ProcessCommandResult_Finish,
    };

public:
    explicit CommandMonitor(
        AnalyzeCommandResult (*pAnalyzeCommandCallback)(AnalyzeCommandArg&),
        ProcessCommandResult (*pProcessCommandCallback)(ProcessCommandArg&)) NN_NOEXCEPT;
    ~CommandMonitor() NN_NOEXCEPT;

    ViewerResult Initialize(const InitArg& arg) NN_NOEXCEPT;

    bool RequestOpen() NN_NOEXCEPT;
    void Close() NN_NOEXCEPT;

    bool IsConnected() const NN_NOEXCEPT;

    bool IsOpened() const NN_NOEXCEPT
    {
        return m_pSocket->IsOpened();
    }

    // TODO: GetSocketで行っている処理をこのクラスが行うようにする
    EditSocketBase& GetSocket() NN_NOEXCEPT
    {
        return *m_pSocket;
    }

    bool PollDataCommunication() NN_NOEXCEPT;
    bool PollDataEdit() NN_NOEXCEPT;

    bool SendBeginFreeze() NN_NOEXCEPT;
    bool SendEndFreeze() NN_NOEXCEPT;

private:
    // Poll 処理用状態の列挙型
    enum PollState
    {
        kReady,
        kPacketAnalyzing,
        kCommandAnalyzing,
        kCommandProcessing
    };

    // ファイル処理用状態の列挙型
    enum FileLoadState
    {
        kBegin,     // 処理開始
        kLooping,   // 複数ファイルループ処理中
        kLoading,   // ファイルロード中
        kLoadedAll, // 全ての複数ファイルロード終了
        kEnd        // 処理終了
    };

private:
    void TryResizeWorkBuffer(int bufferSize) NN_NOEXCEPT;
    void ClearState() NN_NOEXCEPT;
    void ClearWorkBuffer() NN_NOEXCEPT;
    void PollSocket() NN_NOEXCEPT;
    bool AnalyzePacket() NN_NOEXCEPT;

private:
    nn::g3d::viewer::detail::Allocator* m_pAllocator;
    nn::g3d::viewer::detail::EditWorkBuffer* m_pWorkBuffer;

    EditSocketBase*  m_pSocket;
    PacketHeader     m_PacketHeader;
    PacketHeader      m_PacketErrorHeader;
    PacketHeader      m_SendPacketHeader;

    volatile PollState        m_PollState;
    volatile FileLoadState    m_FileLoadState;
    volatile int                    m_FileLoadIndex;
    volatile bool           m_IsReadStarted;
    volatile bool           m_IsCommandAnalyzing;

    AnalyzeCommandResult (*m_pAnalyzeCommandCallback)(AnalyzeCommandArg&);
    ProcessCommandResult (*m_pProcessCommandCallback)(ProcessCommandArg&);
};

}}}} // namespace evfl::viewer::detail

