﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/viewer/g3d_ViewerDefine.h>
#include "g3d_ViewerDetailDefine.h"


#include <nn/nn_Allocator.h>

namespace nn { namespace g3d { namespace viewer { namespace detail {

// 開発目的でメモリ使用内訳を調べるために利用
enum AllocateType
{
    AllocateType_Resource,      //!< リソース
    AllocateType_Obj,           //!< g3d のオブジェクト
    AllocateType_MemoryPool,    //!< メモリプール
    AllocateType_EditObj,       //!< リアルタイム編集用のオブジェクト
    AllocateType_Communication, //!< 通信が終わったら破棄される通信用のメモリ
    AllocateType_DynamicBuffer, //!< 動的にリサイズされるバッファ
    AllocateType_Other,         //!< その他
};

class IAllocator
{
public:
    virtual void* Allocate(size_t size, size_t alignment, AllocateType type) = 0;
    virtual void Free(void* ptr) = 0;
};

//! @brief 内部実装用のメモリアロケータです。
class Allocator : public IAllocator
{
public:
    Allocator(
        const nn::AlignedAllocateFunctionWithUserData allocateFunc,
        const nn::FreeFunctionWithUserData freeFunc,
        void* pAllocateUserData,
        void* pFreeUserData) NN_NOEXCEPT;
    ~Allocator() NN_NOEXCEPT;

    //! @brief メモリを確保します。
    void* Allocate(size_t size, size_t alignment, AllocateType type) NN_NOEXCEPT;

    //! @brief メモリを解放します。
    void Free(void* ptr) NN_NOEXCEPT;

    const nn::FreeFunctionWithUserData GetFreeFunction() const NN_NOEXCEPT;

    void* GetFreeUserData() NN_NOEXCEPT;

    //! @brief デバッグ情報を記録するときにスタックトレースを記録する深さを指定します。デバッグビルド時のみ有効です。
    void SetTraceBackStackDepth(int minDepth, int maxDepth) NN_NOEXCEPT;

    //! @brief リークしているメモリー確保数を調べます。メモリーリークがあった場合、スタックトレースをログ出力します。
    //!        デバッグビルド時のみ有効です。
    int FindMemoryLeakCount() const NN_NOEXCEPT;

    //! @brief  メモリを確保した際の情報をログ出力します。スタックトレースが存在する場合、スタックトレースもログ出力します。
    //!         デバッグビルド時のみ有効です。
    bool PrintAllocateInfo(void* pAddress) const NN_NOEXCEPT;

private:
    class Impl;
    Impl* m_Impl;
};

}}}} // namespace nn::g3d::viewer::detail

