﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_Allocator.h"
#include "util\g3d_SynchronizedDynamicArray.h"
#include "util\g3d_ScopedLock.h"
#include "util\g3d_ScopedAllocator.h"

// メモリーリーク箇所のスタックトレースログを有効にするためのプロプロセッサです。
// Windows 版のみ利用可能です。
//#define NN_G3D_VIEWER_DETAIL_ENABLE_STACK_TRACE_LOG

#ifdef NN_G3D_VIEWER_DETAIL_ENABLE_STACK_TRACE_LOG
#ifdef _WIN32
#include <windows.h>
#pragma warning( push )
#pragma warning ( disable : 4091 )
#include <imagehlp.h>
#pragma warning( pop )
#endif
#endif

#define NN_G3D_VIEWER_DETAIL_CASE_RETURN_STRING(value) case value: return #value


using namespace nn::g3d::viewer::detail;

namespace {
    const char* GetAllocateTypeAsString(nn::g3d::viewer::detail::AllocateType type)
    {
        switch (type)
        {
            NN_G3D_VIEWER_DETAIL_CASE_RETURN_STRING(AllocateType_Resource);
            NN_G3D_VIEWER_DETAIL_CASE_RETURN_STRING(AllocateType_Obj);
            NN_G3D_VIEWER_DETAIL_CASE_RETURN_STRING(AllocateType_MemoryPool);
            NN_G3D_VIEWER_DETAIL_CASE_RETURN_STRING(AllocateType_EditObj);
            NN_G3D_VIEWER_DETAIL_CASE_RETURN_STRING(AllocateType_Communication);
            NN_G3D_VIEWER_DETAIL_CASE_RETURN_STRING(AllocateType_DynamicBuffer);
            NN_G3D_VIEWER_DETAIL_CASE_RETURN_STRING(AllocateType_Other);
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    class SimpleAllocator : public nn::g3d::viewer::detail::IAllocator
    {
    public:
        SimpleAllocator(
            const nn::AlignedAllocateFunctionWithUserData allocateFunc,
            const nn::FreeFunctionWithUserData freeFunc,
            void* pAllocateUserData,
            void* pFreeUserData) NN_NOEXCEPT
            : m_AllocateFunc(allocateFunc)
            , m_FreeFunc(freeFunc)
            , m_AllocateUserData(pAllocateUserData)
            , m_FreeUserData(pFreeUserData)
        {
        }

        //! @brief メモリを確保します。
        void* Allocate(size_t size, size_t alignment, AllocateType type) NN_NOEXCEPT
        {
            const size_t LargeMemorySizeThreshold = 1024 * 100;
            if (type != AllocateType_Resource && size > LargeMemorySizeThreshold)
            {
                // リソース以外は 100 KB に収まる想定なので、超えてしまったら内部警告メッセージを出しておく
                NN_G3D_VIEWER_INTERNAL_WARNING("Allocated large memory for debug data\n");
            }

            void* buffer = m_AllocateFunc(size, alignment, m_AllocateUserData);
            return buffer;
        }

        //! @brief メモリを解放します。
        void Free(void* ptr) NN_NOEXCEPT
        {
            m_FreeFunc(ptr, m_FreeUserData);
        }

        const nn::FreeFunctionWithUserData GetFreeFunction() const NN_NOEXCEPT
        {
            return m_FreeFunc;
        }

        void* GetFreeUserData() NN_NOEXCEPT
        {
            return m_FreeUserData;
        }

    private:
        const nn::AlignedAllocateFunctionWithUserData m_AllocateFunc;
        const nn::FreeFunctionWithUserData m_FreeFunc;
        void* m_AllocateUserData;
        void* m_FreeUserData;
    };

    struct AllocateInfo
    {
        void* pAddress;
        size_t allocSize;
        size_t allocAlignment;
        AllocateType allocType;
        char* allocatedFunctionName;

        AllocateInfo()
            : pAddress(nullptr)
            , allocatedFunctionName(nullptr)
        {
        }

        AllocateInfo(void* ptr, size_t size, size_t alignment, AllocateType type)
            : pAddress(ptr)
            , allocSize(size)
            , allocAlignment(alignment)
            , allocType(type)
            , allocatedFunctionName(nullptr)
        {
        }
    };
}

class nn::g3d::viewer::detail::Allocator::Impl
{
public:
    Impl(const nn::AlignedAllocateFunctionWithUserData allocateFunc,
        const nn::FreeFunctionWithUserData freeFunc,
        void* pAllocateUserData,
        void* pFreeUserData,
        bool isDebugModeEnabled,
        bool isAllocationLogEnabled)
        : m_Allocator(allocateFunc, freeFunc, pAllocateUserData, pFreeUserData)
        , m_AllocatedAddressList(&m_Allocator, nn::DefaultAlignment, AllocateInfo())
        , m_TraceBackStackDepthMin(0)
        , m_TraceBackStackDepthMax(6)
        , m_IsDebugModeEnabled(isDebugModeEnabled)
        , m_IsAllocationLogEnabled(isAllocationLogEnabled)
    {
    }

    //! @brief メモリを確保します。
    void* Allocate(size_t size, size_t alignment, AllocateType type) NN_NOEXCEPT
    {
        void* buffer = m_Allocator.Allocate(size, alignment, type);

        if (m_IsDebugModeEnabled)
        {
            RegisterDebugInfo(buffer, size, alignment, type);
        }
        return buffer;
    }

    //! @brief メモリを解放します。
    void Free(void* ptr) NN_NOEXCEPT
    {
        if (m_IsDebugModeEnabled)
        {
            UnregisterDebugInfo(ptr);
        }

        m_Allocator.Free(ptr);
    }

    const nn::FreeFunctionWithUserData GetFreeFunction() const NN_NOEXCEPT
    {
        return m_Allocator.GetFreeFunction();
    }

    void* GetFreeUserData() NN_NOEXCEPT
    {
        return m_Allocator.GetFreeUserData();
    }

    int FindMemoryLeakCount() const NN_NOEXCEPT
    {
        if (m_AllocatedAddressList.GetCount() == 0)
        {
            return 0;
        }

        NN_G3D_VIEWER_LOG("Detected memory leaks:\n");
        for (Iter<const AllocateInfo> iter = m_AllocatedAddressList.Begin(), end = m_AllocatedAddressList.End(); iter != end; ++iter)
        {
            const AllocateInfo* pInfo = &(*iter);
            const char* allocateTypeStr = GetAllocateTypeAsString(pInfo->allocType);
            NN_UNUSED(allocateTypeStr);
            NN_G3D_VIEWER_LOG("address = %p, type = %s, size = %zu, alignment = %zu\n",
                pInfo->pAddress, allocateTypeStr, pInfo->allocSize, pInfo->allocAlignment);
            if (pInfo->allocatedFunctionName != nullptr)
            {
                NN_G3D_VIEWER_LOG("StackTrace:\n%s\n", pInfo->allocatedFunctionName);
            }
        }

        return m_AllocatedAddressList.GetCount();
    }

    //! @brief デバッグ情報を記録するときにスタックトレースを記録する深さを指定します。
    void SetTraceBackStackDepth(int minDepth, int maxDepth)
    {
        m_TraceBackStackDepthMin = minDepth;
        m_TraceBackStackDepthMax = maxDepth;
    }

    bool PrintAllocateInfo(void* pAddress) const NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pAddress);

        if (m_AllocatedAddressList.GetCount() == 0)
        {
            return false;
        }

        for (Iter<const AllocateInfo> iter = m_AllocatedAddressList.Begin(), end = m_AllocatedAddressList.End(); iter != end; ++iter)
        {
            const AllocateInfo* pInfo = &(*iter);
            if (pInfo->pAddress != pAddress)
            {
                continue;
            }

            const char* allocateTypeStr = GetAllocateTypeAsString(pInfo->allocType);
            NN_UNUSED(allocateTypeStr);

            NN_G3D_VIEWER_LOG("address = %p, type = %s, size = %zu, alignment = %zu\n",
                pInfo->pAddress, allocateTypeStr, pInfo->allocSize, pInfo->allocAlignment);

            if (pInfo->allocatedFunctionName != nullptr)
            {
                NN_G3D_VIEWER_LOG("StackTrace:\n%s\n", pInfo->allocatedFunctionName);
            }

            return true;
        }

        return false;
    }

private:
#ifdef NN_G3D_VIEWER_DETAIL_ENABLE_STACK_TRACE_LOG
#ifdef _WIN32
    std::string GetCallstackOutputString(char* pOutSymbolNames, int nameLengthMax, int symbolCount)
    {
        std::string outCallstack;
        for (int i = 0; i < symbolCount; ++i)
        {
            char* pFuncName = pOutSymbolNames + nameLengthMax * i;
            outCallstack += std::string("    ") + std::string(pFuncName) + std::string("\n");
        }

        return outCallstack;
    }

    void GetCalledSymbolName(char* pOutSymbolNames, int nameLengthMax, int minSymbolDepth, int maxSymbolDepth)
    {
        HANDLE process = GetCurrentProcess();
        SymInitialize(process, nullptr, TRUE);

        void* stack[100];
        CaptureStackBackTrace(0, 100, stack, nullptr);
        SYMBOL_INFO* pSymbol = (SYMBOL_INFO *)calloc(sizeof(SYMBOL_INFO) + 256 * sizeof(char), 1);
        pSymbol->MaxNameLen = 255;
        pSymbol->SizeOfStruct = sizeof(SYMBOL_INFO);

        std::string symbolName;
        for (int i = minSymbolDepth; i <= maxSymbolDepth; ++i)
        {
            int calledStackLevel = 2 + i;
            SymFromAddr(process, (DWORD64)(stack[calledStackLevel]), 0, pSymbol);

            int bufferIndex = i - minSymbolDepth;
            char* pDst = pOutSymbolNames + nameLengthMax * bufferIndex;
            strcpy(pDst, pSymbol->Name);
        }

        free(pSymbol);
    }
#endif
#endif

    void CreateStackTraceInfo(char** pOutFunctionName)
    {
        NN_UNUSED(pOutFunctionName);
#ifdef NN_G3D_VIEWER_DETAIL_ENABLE_STACK_TRACE_LOG
#ifdef _WIN32
        const int functionNameLengthMax = 128;
        const int defaultStackDepth = 3;
        int traceBackDepthMin = m_TraceBackStackDepthMin + defaultStackDepth;
        int traceBackDepthMax = m_TraceBackStackDepthMax + defaultStackDepth;
        const int traceBackStackCount = (traceBackDepthMax + 1) - traceBackDepthMin;
        size_t bufferSize = functionNameLengthMax * traceBackStackCount;
        {
            ScopedAllocator scopedAllocator(m_Allocator, bufferSize, nn::DefaultAlignment, AllocateType_Other);
            char* calledFunctionNames = reinterpret_cast<char*>(scopedAllocator.GetAllocatedBuffer());
            *pOutFunctionName = reinterpret_cast<char*>(m_Allocator.Allocate(bufferSize, nn::DefaultAlignment, AllocateType_Other));
            GetCalledSymbolName(calledFunctionNames, functionNameLengthMax, traceBackDepthMin, traceBackDepthMax);
            strncpy(*pOutFunctionName,
                GetCallstackOutputString(calledFunctionNames, functionNameLengthMax, traceBackStackCount).c_str(),
                bufferSize);
        }
#else
        // Windows 版以外はスタックトレースの記録はサポート外
#endif
#endif
    }

    void DeleteStackTraceInfo(char** pFunctionName)
    {
        if (*pFunctionName == nullptr)
        {
            return;
        }

        m_Allocator.Free(*pFunctionName);
        (*pFunctionName) = nullptr;
    }

    void RegisterDebugInfo(void* ptr, size_t size, size_t alignment, AllocateType type) NN_NOEXCEPT
    {
        AllocateInfo info(ptr, size, alignment, type);
        CreateStackTraceInfo(&info.allocatedFunctionName);
        if (m_IsAllocationLogEnabled)
        {
            const char* allocateTypeStr = GetAllocateTypeAsString(type);
            NN_UNUSED(allocateTypeStr);
            NN_G3D_VIEWER_LOG("Allocate %p: size %zu, alignment %zu, type %s\n", ptr, size, alignment, allocateTypeStr);
        }
        m_AllocatedAddressList.PushBack(info);
    }

    void UnregisterDebugInfo(void* ptr) NN_NOEXCEPT
    {
        ScopedLock scopedLock(m_AllocatedAddressList);
        for (Iter<AllocateInfo> iter = m_AllocatedAddressList.Begin(), end = m_AllocatedAddressList.End(); iter != end; ++iter)
        {
            AllocateInfo* pInfo = &(*iter);
            if (pInfo->pAddress == ptr)
            {
                DeleteStackTraceInfo(&pInfo->allocatedFunctionName);
                if (m_IsAllocationLogEnabled)
                {
                    const char* allocateTypeStr = GetAllocateTypeAsString(pInfo->allocType);
                    NN_UNUSED(allocateTypeStr);
                    NN_G3D_VIEWER_LOG("Free %p, size = %zu, alignment = %zu, type = %s\n",
                        ptr, pInfo->allocSize, pInfo->allocAlignment, allocateTypeStr);
                }
                m_AllocatedAddressList.Erase(iter);
                return;
            }
        }
    }

    SimpleAllocator m_Allocator;
    nn::g3d::viewer::detail::SynchronizedDynamicArray<AllocateInfo> m_AllocatedAddressList;
    int m_TraceBackStackDepthMin;
    int m_TraceBackStackDepthMax;
    bool m_IsDebugModeEnabled;
    bool m_IsAllocationLogEnabled;
};

nn::g3d::viewer::detail::Allocator::Allocator(
    const nn::AlignedAllocateFunctionWithUserData allocateFunc,
    const nn::FreeFunctionWithUserData freeFunc,
    void* pAllocateUserData,
    void* pFreeUserData) NN_NOEXCEPT
    : m_Impl(nullptr)
{
#if defined(NN_SDK_BUILD_DEBUG)
    // デバッグビルド時はメモリーリークチェックをする
    const bool IsDebugModeEnabled = true;
#else
    const bool IsDebugModeEnabled = false;
#endif
    // 必要な時は書き換えて有効にする
    const bool IsAllocationLogEnabled = false;

    void* buffer = allocateFunc(sizeof(nn::g3d::viewer::detail::Allocator::Impl), nn::DefaultAlignment, pAllocateUserData);
    m_Impl = new (buffer) nn::g3d::viewer::detail::Allocator::Impl(
        allocateFunc, freeFunc, pAllocateUserData, pFreeUserData, IsDebugModeEnabled, IsAllocationLogEnabled);
}

Allocator::~Allocator() NN_NOEXCEPT
{
    const nn::FreeFunctionWithUserData freeFunc = m_Impl->GetFreeFunction();
    void* pFreeUserData = m_Impl->GetFreeUserData();
    m_Impl->~Impl();
    freeFunc(m_Impl, pFreeUserData);
}

void* nn::g3d::viewer::detail::Allocator::Allocate(size_t size, size_t alignment, AllocateType type) NN_NOEXCEPT
{
    return m_Impl->Allocate(size, alignment, type);
}

void nn::g3d::viewer::detail::Allocator::Free(void* ptr) NN_NOEXCEPT
{
    m_Impl->Free(ptr);
}

int nn::g3d::viewer::detail::Allocator::FindMemoryLeakCount() const NN_NOEXCEPT
{
    return m_Impl->FindMemoryLeakCount();
}

bool nn::g3d::viewer::detail::Allocator::PrintAllocateInfo(void * pAddress) const NN_NOEXCEPT
{
    return m_Impl->PrintAllocateInfo(pAddress);
}

void nn::g3d::viewer::detail::Allocator::SetTraceBackStackDepth(int minDepth, int maxDepth) NN_NOEXCEPT
{
    return m_Impl->SetTraceBackStackDepth(minDepth, maxDepth);
}

const nn::FreeFunctionWithUserData nn::g3d::viewer::detail::Allocator::GetFreeFunction() const NN_NOEXCEPT
{
    return m_Impl->GetFreeFunction();
}

void* nn::g3d::viewer::detail::Allocator::GetFreeUserData() NN_NOEXCEPT
{
    return m_Impl->GetFreeUserData();
}
