﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/viewer/g3d_ViewerDefine.h>



#include "g3d_EditAnimObj.h"
#include "g3d_EditAnimCurve.h"

#include <nn/g3d/g3d_BoneVisibilityAnimObj.h>

namespace nn { namespace g3d {

class ModelAnimObj;
class ResFile;

namespace viewer {
namespace detail {

class Allocator;

class EditVisibilityAnimObj : public EditAnimObj
{
public:
    virtual ~EditVisibilityAnimObj() NN_NOEXCEPT
    {
    }

    virtual void DestroyDataForEditingAnimCurve() NN_NOEXCEPT;

    virtual void EditCurve(
        int curveIndex,
        const nn::g3d::ResAnimCurve* resAnimCurve) = 0;

protected:
    explicit EditVisibilityAnimObj(Allocator* pAllocator, nn::g3d::ResFile* pResFile, ViewerAnimKind animKind) NN_NOEXCEPT
        : EditAnimObj(pAllocator, pResFile, animKind)
        , m_pEditAnimCurveArray(nullptr)
        , m_EditAnimCurveCount(0)
    {
    }

    nn::g3d::ResAnimCurve* GetCurve(ResBoneVisibilityAnim* pResAnim, int index) NN_NOEXCEPT
    {
        return pResAnim->GetCurve(index);
    }

    nn::g3d::ResAnimCurve* GetCurve(ResMaterialAnim* pResAnim, int index) NN_NOEXCEPT
    {
        int totalCurveIndex = 0;
        for (int perMaterialAnimIndex = 0, perMaterialAnimCount = pResAnim->GetPerMaterialAnimCount();
            perMaterialAnimIndex < perMaterialAnimCount; ++perMaterialAnimIndex)
        {
            ResPerMaterialAnim* pResPerMaterialAnim = pResAnim->GetPerMaterialAnim(perMaterialAnimIndex);
            for (int curveIndex = 0, curveCount = pResPerMaterialAnim->GetCurveCount();
                curveIndex < curveCount; ++curveIndex, ++totalCurveIndex)
            {
                if (totalCurveIndex == index)
                {
                    NN_G3D_VIEWER_DEBUG_PRINT("Bind Material Visibility Anim: materialAnimIndex = %d\n", perMaterialAnimIndex);
                    return pResPerMaterialAnim->GetCurve(curveIndex);
                }
            }
        }

        return nullptr;
    }

    template<typename TResAnim>
    void EditCurveImpl(
        int curveIndex,
        const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT
    {
        TResAnim* pResAnim = static_cast<TResAnim*>(GetResAnim());
        NN_G3D_VIEWER_ASSERT_DETAIL(curveIndex != -1, "%s\n", NN_G3D_VIEWER_RES_NAME(pResAnim, GetName()));

        int curveCount = pResAnim->GetCurveCount();
        if (curveIndex >= curveCount)
        {
            NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(curveIndex, curveCount);
            return;
        }

        EditAnimCurve* editAnimCurve = &m_pEditAnimCurveArray[curveIndex];
        editAnimCurve->Edit(resAnimCurve);
    }

    template<typename TAnimObj, typename TResAnim>
    BindResult SetupInternalImpl(EditModelObj* pBindTargetEditModelObj, ModelAnimObj* pBoundAnimObj) NN_NOEXCEPT
    {
        BindResult result;

        // TODO: clangでエラーが出ないように対応する
        TAnimObj* pAnimObj = static_cast<TAnimObj*>(pBoundAnimObj);
        TResAnim* pResAnim = static_cast<TResAnim*>(GetResAnim());

        typename TAnimObj::Builder builder;
        builder.SetContextDisabled();
        builder.Reserve(pBindTargetEditModelObj->GetResource());
        builder.Reserve(pResAnim);
        builder.CalculateMemorySize();
        size_t bufferSize = builder.GetWorkMemorySize();
        void* buffer = m_pAllocator->Allocate(bufferSize, TAnimObj::Alignment_Buffer, AllocateType_Obj);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NN_G3D_VIEWER_RES_NAME(pBindTargetEditModelObj->GetResource(), GetName()));//今は止める
        bool success = builder.Build(pAnimObj, buffer, bufferSize);
        NN_G3D_VIEWER_ASSERT(success);
        pAnimObj->SetResource(pResAnim);
        result = pAnimObj->Bind(pBindTargetEditModelObj->GetResource());

        return result;
    }

    template <typename TResAnim>
    bool CreateDataForEditingAnimCurveImpl() NN_NOEXCEPT
    {
        TResAnim* pResAnim = static_cast<TResAnim*>(GetResAnim());
        int curveCount = pResAnim->GetCurveCount();
        size_t bufferSize = sizeof(EditAnimCurve) * curveCount;

        void* buffer = nullptr;
        if (bufferSize > 0)
        {
            buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
            if (buffer == nullptr)
            {
                return false;
            }
        }

        m_pEditAnimCurveArray = static_cast<EditAnimCurve*>(buffer);
        m_EditAnimCurveCount = curveCount;

        for(int i = 0; i < m_EditAnimCurveCount; ++i)
        {
            nn::g3d::ResAnimCurve* pResAnimCurve = GetCurve(pResAnim, i);
            NN_G3D_VIEWER_ASSERT_NOT_NULL(pResAnimCurve);

            EditAnimCurve* editAnimCurve = new (&m_pEditAnimCurveArray[i]) EditAnimCurve(m_pAllocator, pResAnimCurve);
            editAnimCurve->Setup();
        }
        return true;
    }

protected:
    EditAnimCurve* m_pEditAnimCurveArray;
    int m_EditAnimCurveCount;

private:
    NN_DISALLOW_COPY(EditVisibilityAnimObj);
};

class EditBoneVisibilityAnimObj : public EditVisibilityAnimObj
{
public:
    explicit EditBoneVisibilityAnimObj(Allocator* allocator, nn::g3d::ResFile* pResFile) NN_NOEXCEPT
        : EditVisibilityAnimObj(allocator, pResFile, ViewerAnimKind_BoneVisibilityAnim)
    {
    }

    virtual ~EditBoneVisibilityAnimObj() NN_NOEXCEPT
    {
        UnbindAllModelObjs();
        DestroyDataForEditingAnimCurve();
    }

    virtual bool CreateDataForEditingAnimCurve() NN_NOEXCEPT
    {
        return CreateDataForEditingAnimCurveImpl<ResBoneVisibilityAnim>();
    }

    virtual const char* GetName() const NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT (m_pResFile->GetBoneVisibilityAnimCount() > 0);
        return m_pResFile->GetBoneVisibilityAnim(0)->GetName();
    }

    virtual void EditCurve(
        int curveIndex,
        const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT
    {
        EditCurveImpl<ResBoneVisibilityAnim>(curveIndex, resAnimCurve);
    }

    virtual ModelAnimObj* CreateAnimObj() NN_NOEXCEPT
    {
        void* buffer = m_pAllocator->Allocate(sizeof(BoneVisibilityAnimObj), nn::g3d::detail::Alignment_Default, AllocateType_Obj);
        BoneVisibilityAnimObj* pAnimObj = new (buffer) BoneVisibilityAnimObj();
        return pAnimObj;
    }

protected:
    virtual void ResetToOriginalValue(EditModelObj* pBoundModelObj) NN_NOEXCEPT;
    virtual BindResult SetupInternal(EditModelObj* pBindTargetEditModelObj, ModelAnimObj* pBoundAnimObj) NN_NOEXCEPT
    {
        return SetupInternalImpl<BoneVisibilityAnimObj, ResBoneVisibilityAnim>(pBindTargetEditModelObj, pBoundAnimObj);
    }

private:
    NN_DISALLOW_COPY(EditBoneVisibilityAnimObj);
};

class EditMatVisibilityAnimObj : public EditVisibilityAnimObj
{
public:
    explicit EditMatVisibilityAnimObj(Allocator* allocator, nn::g3d::ResFile* pResFile) NN_NOEXCEPT
        : EditVisibilityAnimObj(allocator, pResFile, ViewerAnimKind_MaterialVisibilityAnim)
    {
    }

    virtual ~EditMatVisibilityAnimObj() NN_NOEXCEPT
    {
        UnbindAllModelObjs();
        DestroyDataForEditingAnimCurve();
    }

    virtual bool CreateDataForEditingAnimCurve() NN_NOEXCEPT
    {
        return CreateDataForEditingAnimCurveImpl<ResMaterialAnim>();
    }

    virtual const char* GetName() const NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT (HasMaterialVisibilityAnim(m_pResFile));
        return GetResName(m_pResFile, nn::g3d::viewer::detail::FILEDATA_MAT_VISIBILITY_ANIM);
    }

    virtual void EditCurve(
        int curveIndex,
        const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT
    {
        EditCurveImpl<ResMaterialAnim>(curveIndex, resAnimCurve);
    }

    virtual ModelAnimObj* CreateAnimObj() NN_NOEXCEPT
    {
        void* buffer = m_pAllocator->Allocate(sizeof(MaterialAnimObj), nn::g3d::detail::Alignment_Default, AllocateType_Obj);
        MaterialAnimObj* pAnimObj = new (buffer) MaterialAnimObj();
        return pAnimObj;
    }

protected:
    virtual void ResetToOriginalValue(EditModelObj* pBoundModelObj) NN_NOEXCEPT;
    virtual BindResult SetupInternal(EditModelObj* pBindTargetEditModelObj, ModelAnimObj* pBoundAnimObj) NN_NOEXCEPT
    {
        return SetupInternalImpl<MaterialAnimObj, ResMaterialAnim>(pBindTargetEditModelObj, pBoundAnimObj);
    }

private:
    NN_DISALLOW_COPY(EditMatVisibilityAnimObj);
};

}}}} // namespace nn::g3d::viewer::detail


