﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditTexPatternAnimObj.h"



#include "../g3d_Allocator.h"
#include "g3d_EditAnimCurve.h"
#include <nn/gfx/gfx_ResTexture.h>

namespace nn { namespace g3d { namespace viewer { namespace detail {

    class EditTexPatternAnimObj::EditTexPatternMatAnim
    {
    public:
        explicit EditTexPatternMatAnim(Allocator* allocator, ResPerMaterialAnim* resMatAnim) NN_NOEXCEPT
            : m_pAllocator(allocator)
            , m_pResMatAnim(resMatAnim)
            , m_pEditAnimCurveArray(nullptr)
            , m_EditAnimCurveCount(0)
        {
            NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(allocator, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pResMatAnim, GetName()));
            NN_G3D_VIEWER_ASSERT_NOT_NULL(resMatAnim);
        }

        bool Init() NN_NOEXCEPT
        {
            int curveCount = m_pResMatAnim->GetCurveCount();
            size_t bufferSize = sizeof(EditAnimCurve) * curveCount;

            void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
            if (buffer == nullptr)
            {
                return false;
            }

            m_pEditAnimCurveArray = static_cast<EditAnimCurve*>(buffer);
            m_EditAnimCurveCount = curveCount;

            for(int i = 0; i < m_EditAnimCurveCount; ++i)
            {
                nn::g3d::ResAnimCurve* resAnimCurve = m_pResMatAnim->GetCurve(i);
                EditAnimCurve* editAnimCurve = new (&m_pEditAnimCurveArray[i]) EditAnimCurve(m_pAllocator, resAnimCurve);
                editAnimCurve->Setup();
            }

            return true;
        }

        void Destroy() NN_NOEXCEPT
        {
            for(int i = 0; i < m_EditAnimCurveCount; ++i)
            {
                EditAnimCurve* editAnimCurve = &m_pEditAnimCurveArray[i];
                editAnimCurve->Cleanup();
                editAnimCurve->~EditAnimCurve();
            }

            if (m_pEditAnimCurveArray != nullptr)
            {
                m_pAllocator->Free(m_pEditAnimCurveArray);
                m_pEditAnimCurveArray = nullptr;
            }
        }

        EditAnimCurve* GetEditAnimCurve(int index) NN_NOEXCEPT
        {
            return &m_pEditAnimCurveArray[index];
        }

        int GetEditAnimCurveCount() const NN_NOEXCEPT
        {
            return m_EditAnimCurveCount;
        }

    private:
        Allocator*             m_pAllocator;
        ResPerMaterialAnim*   m_pResMatAnim;
        EditAnimCurve*          m_pEditAnimCurveArray;
        int m_EditAnimCurveCount;

    private:
        NN_DISALLOW_COPY(EditTexPatternMatAnim);
    };

    BindResult EditTexPatternAnimObj::SetupInternal(EditModelObj* pBindTargetEditModelObj, ModelAnimObj* pBoundAnimObj) NN_NOEXCEPT
    {
        BindResult result;
        MaterialAnimObj* pAnimObj = static_cast<MaterialAnimObj*>(pBoundAnimObj);
        ResMaterialAnim* pResAnim = GetResAnim<ResMaterialAnim>();
        MaterialAnimObj::Builder builder;
        builder.SetContextDisabled();
        builder.Reserve(pBindTargetEditModelObj->GetResource());
        builder.Reserve(pResAnim);
        builder.CalculateMemorySize();
        size_t bufferSize = builder.GetWorkMemorySize();
        void* buffer = m_pAllocator->Allocate(bufferSize, MaterialAnimObj::Alignment_Buffer, AllocateType_Obj);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NN_G3D_VIEWER_RES_NAME(pBindTargetEditModelObj->GetResource(), GetName()));//今は止める
        bool success = builder.Build(pAnimObj, buffer, bufferSize);
        NN_G3D_VIEWER_ASSERT(success);
        pAnimObj->SetResource(pResAnim);
        result = pAnimObj->Bind(pBindTargetEditModelObj->GetResource());

        if (!result.IsComplete())
        {
            // 不具合でバインドできない場合のために念のためログを出しておく
            NN_G3D_VIEWER_LOG("Binding \"%s\" to \"%s\" has not been completed\n", pResAnim->GetName(), pBindTargetEditModelObj->GetName());
        }

        return result;
    }

    void EditTexPatternAnimObj::ResetToOriginalValue(EditModelObj* pBoundModelObj) NN_NOEXCEPT
    {
        if (!IsModelBound(pBoundModelObj))
        {
            return;
        }
    }

    bool EditTexPatternAnimObj::CreateDataForEditingAnimCurve() NN_NOEXCEPT
    {
        ResMaterialAnim* pResAnim = GetResAnim<ResMaterialAnim>();
        int matAnimCount = pResAnim->GetPerMaterialAnimCount();
        size_t bufferSize = sizeof(EditTexPatternAnimObj::EditTexPatternMatAnim) * matAnimCount;

        void* buffer = nullptr;
        if (bufferSize > 0)
        {
            buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
            if (buffer == nullptr)
            {
                return false;
            }
        }

        m_pMatAnimArray = static_cast<EditTexPatternAnimObj::EditTexPatternMatAnim*>(buffer);
        m_MatAnimCount = matAnimCount;

        bool isFailed = false;
        for(int i = 0; i < m_MatAnimCount; ++i)
        {
            nn::g3d::ResPerMaterialAnim* resMatAnim = pResAnim->GetPerMaterialAnim(i);

            EditTexPatternAnimObj::EditTexPatternMatAnim* editMatAnim =
                new (&m_pMatAnimArray[i]) EditTexPatternAnimObj::EditTexPatternMatAnim(m_pAllocator, resMatAnim);
            isFailed = !editMatAnim->Init();
        }

        // 初期化失敗のものが一つでもあった場合は、インスタンスを破棄して失敗にする。
        if (isFailed)
        {
            DestroyDataForEditingAnimCurve();
            return false;
        }

        return true;
    }

    void EditTexPatternAnimObj::DestroyDataForEditingAnimCurve() NN_NOEXCEPT
    {
        for(int i = 0; i < m_MatAnimCount; ++i)
        {
            EditTexPatternAnimObj::EditTexPatternMatAnim* editMatAnim = &m_pMatAnimArray[i];
            editMatAnim->Destroy();
            editMatAnim->~EditTexPatternMatAnim();
        }

        m_MatAnimCount = 0;

        if (m_pMatAnimArray != nullptr)
        {
            m_pAllocator->Free(m_pMatAnimArray);
            m_pMatAnimArray = nullptr;
        }
    }

    void EditTexPatternAnimObj::EditCurve(
        int matAnimIndex,
        int curveIndex,
        const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_DETAIL(curveIndex != -1, "%s\n", NN_G3D_VIEWER_RES_NAME(GetResAnim<ResMaterialAnim>(), GetName()));

        if (matAnimIndex >= m_MatAnimCount)
        {
            NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(matAnimIndex, m_MatAnimCount);
            return;
        }

        EditTexPatternAnimObj::EditTexPatternMatAnim* editMatAnim = &m_pMatAnimArray[matAnimIndex];

        int curveCount = editMatAnim->GetEditAnimCurveCount();
        if (curveIndex >= curveCount)
        {
            NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(curveIndex, curveCount);
            return;
        }

        EditAnimCurve* editAnimCurve = editMatAnim->GetEditAnimCurve(curveIndex);
        editAnimCurve->Edit(resAnimCurve);
    }

    bool EditTexPatternAnimObj::ExaminesTextureFileUsed(const nn::gfx::ResTextureFile* pResTextureFile) const NN_NOEXCEPT
    {
        return nn::g3d::viewer::detail::ExaminesTextureFileUsed(GetResAnim<const ResMaterialAnim>(), pResTextureFile);
    }

    void EditTexPatternAnimObj::ReloadTexture(
        const nn::gfx::ResTextureFile* pOldResTextureFile, const nn::gfx::ResTextureFile* pNewResTextureFile, CallbackCaller& callback) NN_NOEXCEPT
    {
        if (!ExaminesTextureFileUsed(pOldResTextureFile))
        {
            return;
        }

        ForceBindTexture(pNewResTextureFile, callback);
    }

    void EditTexPatternAnimObj::ForceBindTexture(const nn::gfx::ResTextureFile* pResTextureFile, CallbackCaller& callback) NN_NOEXCEPT
    {
        for (int boundModelIndex = 0, boundModelCount = GetBoundEditModelObjCount(); boundModelIndex < boundModelCount; ++boundModelIndex)
        {
            nn::g3d::MaterialAnimObj* pAnimObj = GetAnimObj<MaterialAnimObj>(boundModelIndex);
            nn::g3d::viewer::detail::ForceBindTexture(pAnimObj, pResTextureFile, callback);
        }

        ResMaterialAnim* pResMatAnim = GetResAnim<ResMaterialAnim>();
        nn::g3d::viewer::detail::ForceBindTexture(pResMatAnim, pResTextureFile, callback);
    }

    void EditTexPatternAnimObj::UnbindTexture(const nn::gfx::ResTextureFile* pResTextureFile, CallbackCaller& callback) NN_NOEXCEPT
    {
        if (!ExaminesTextureFileUsed(pResTextureFile))
        {
            return;
        }

        for (int boundModelIndex = 0, boundModelCount = GetBoundEditModelObjCount(); boundModelIndex < boundModelCount; ++boundModelIndex)
        {
            nn::g3d::MaterialAnimObj* pAnimObj = GetAnimObj<MaterialAnimObj>(boundModelIndex);
            ForceUnbindTexture(pAnimObj, pResTextureFile, callback);
        }

        ResMaterialAnim* pResMatAnim = GetResAnim<ResMaterialAnim>();
        ForceUnbindTexture(pResMatAnim, pResTextureFile, callback);
    }

    ModelAnimObj* EditTexPatternAnimObj::CreateAnimObj() NN_NOEXCEPT
    {
        void* buffer = m_pAllocator->Allocate(sizeof(MaterialAnimObj), nn::g3d::detail::Alignment_Default, AllocateType_Obj);
        MaterialAnimObj* pAnimObj = new (buffer) MaterialAnimObj();
        return pAnimObj;
    }

}}}} // namespace nn::g3d::viewer::detail


