﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditShaderParamAnimObj.h"



#include "g3d_EditAnimCurve.h"


namespace nn { namespace g3d { namespace viewer { namespace detail {

    class EditShaderParamAnimObj::EditShaderParamMatAnim
    {
    public:
        explicit EditShaderParamMatAnim(Allocator* allocator, ResPerMaterialAnim* resMatAnim) NN_NOEXCEPT
            : m_pAllocator(allocator)
            , m_pResMatAnim(resMatAnim)
            , m_pEditAnimCurveArray(nullptr)
            , m_EditAnimCurveCount(0)
        {
            NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(allocator, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pResMatAnim, GetName()));
            NN_G3D_VIEWER_ASSERT_NOT_NULL(resMatAnim);
        }

        bool Init() NN_NOEXCEPT
        {
            int curveCount = m_pResMatAnim->GetCurveCount();
            size_t bufferSize = sizeof(EditAnimCurve) * curveCount;

            if (bufferSize > 0)
            {
                void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
                if (buffer == nullptr)
                {
                    return false;
                }
                m_pEditAnimCurveArray = static_cast<EditAnimCurve*>(buffer);
            }

            m_EditAnimCurveCount = curveCount;

            for(int i = 0; i < m_EditAnimCurveCount; ++i)
            {
                nn::g3d::ResAnimCurve* resAnimCurve = m_pResMatAnim->GetCurve(i);

                EditAnimCurve* editAnimCurve = new(&m_pEditAnimCurveArray[i]) EditAnimCurve(m_pAllocator, resAnimCurve);
                editAnimCurve->Setup();
            }
            return true;
        }

        void Destroy() NN_NOEXCEPT
        {
            for(int i = 0; i < m_EditAnimCurveCount; ++i)
            {
                EditAnimCurve* editAnimCurve = &m_pEditAnimCurveArray[i];
                editAnimCurve->Cleanup();
                editAnimCurve->~EditAnimCurve();
            }

            if (m_pEditAnimCurveArray)
            {
                m_pAllocator->Free(m_pEditAnimCurveArray);
                m_pEditAnimCurveArray = nullptr;
            }
        }

        EditAnimCurve* GetEditAnimCurve(int index) NN_NOEXCEPT
        {
            return &m_pEditAnimCurveArray[index];
        }

        int GetEditAnimCurveCount() const NN_NOEXCEPT
        {
            return m_EditAnimCurveCount;
        }

    private:
        Allocator*             m_pAllocator;
        ResPerMaterialAnim*  m_pResMatAnim;
        EditAnimCurve* m_pEditAnimCurveArray;
        int m_EditAnimCurveCount;

    private:
        NN_DISALLOW_COPY(EditShaderParamMatAnim);
    };

    /*virtual*/void
        EditShaderParamAnimObj::ResetToOriginalValue(EditModelObj* pBoundEditModelObj) NN_NOEXCEPT
    {
        if (!IsModelBound(pBoundEditModelObj))
        {
            return;
        }

        for (int modelIndex = 0, modelCount = pBoundEditModelObj->GetAttachedModelObjCount(); modelIndex < modelCount; ++modelIndex)
        {
            int materialCount = pBoundEditModelObj->GetMaterialCount();
            for (int matIndex = 0; matIndex < materialCount; ++matIndex)
            {
                EditMaterialObj* pEditMaterialObj = pBoundEditModelObj->GetEditMaterialObj(modelIndex, matIndex);
                pEditMaterialObj->RestoreEditedShaderParam();
            }
        }
    }

    /*virtual*/bool
        EditShaderParamAnimObj::CreateDataForEditingAnimCurve() NN_NOEXCEPT
    {
        ResMaterialAnim* pResAnim = static_cast<ResMaterialAnim*>(GetResAnim());
        int matAnimCount = pResAnim->GetPerMaterialAnimCount();
        size_t bufferSize = sizeof(EditShaderParamAnimObj::EditShaderParamMatAnim) * matAnimCount;

        void* buffer = nullptr;
        if (bufferSize > 0)
        {
            buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
            if (buffer == nullptr)
            {
                return false;
            }
        }

        m_pMatAnimArray = static_cast<EditShaderParamAnimObj::EditShaderParamMatAnim*>(buffer);
        m_MatAnimCount = matAnimCount;

        bool isFailed = false;
        for(int i = 0; i < m_MatAnimCount; ++i)
        {
            nn::g3d::ResPerMaterialAnim* resMatAnim = pResAnim->GetPerMaterialAnim(i);

            EditShaderParamAnimObj::EditShaderParamMatAnim* editMatAnim = new(&m_pMatAnimArray[i]) EditShaderParamAnimObj::EditShaderParamMatAnim(m_pAllocator, resMatAnim);
            isFailed = !editMatAnim->Init();
        }

        // 初期化失敗のものが一つでもあった場合は、インスタンスを破棄して失敗にする。
        if (isFailed)
        {
            DestroyDataForEditingAnimCurve();
            return false;
        }

        return true;
    }

    /*virtual*/void
        EditShaderParamAnimObj::DestroyDataForEditingAnimCurve() NN_NOEXCEPT
    {
        for(int i = 0; i < m_MatAnimCount; ++i)
        {
            EditShaderParamAnimObj::EditShaderParamMatAnim* editMatAnim = &m_pMatAnimArray[i];
            editMatAnim->Destroy();
            editMatAnim->~EditShaderParamMatAnim();
        }

        m_MatAnimCount = 0;

        if (m_pMatAnimArray != nullptr)
        {
            m_pAllocator->Free(m_pMatAnimArray);
            m_pMatAnimArray = nullptr;
        }
    }

    /*virtual*/BindResult
        EditShaderParamAnimObj::SetupInternal(EditModelObj* pBindTargetEditModelObj, ModelAnimObj* pBoundAnimObj) NN_NOEXCEPT
    {
        MaterialAnimObj* pAnimObj = static_cast<MaterialAnimObj*>(pBoundAnimObj);
        ResMaterialAnim* pResAnim = static_cast<ResMaterialAnim*>(GetResAnim());

        MaterialAnimObj::Builder builder;
        builder.SetContextDisabled();
        builder.Reserve(pBindTargetEditModelObj->GetResource());
        builder.Reserve(pResAnim);
        builder.CalculateMemorySize();
        size_t bufferSize = builder.GetWorkMemorySize();
        void* buffer = m_pAllocator->Allocate(bufferSize, MaterialAnimObj::Alignment_Buffer, AllocateType_Obj);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NN_G3D_VIEWER_RES_NAME(pBindTargetEditModelObj->GetResource(), GetName()));//今は止める
        bool success = builder.Build(pAnimObj, buffer, bufferSize);
        NN_G3D_VIEWER_ASSERT(success);

        pAnimObj->SetResource(pResAnim);
        BindResult result;
        result = pAnimObj->Bind(pBindTargetEditModelObj->GetResource());

        return result;
    }

    void EditShaderParamAnimObj::EditCurve(
        int matAnimIndex,
        int curveIndex,
        const ResAnimCurve* resAnimCurve) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_DETAIL(curveIndex != -1, "%s\n", NN_G3D_VIEWER_RES_NAME(static_cast<ResMaterialAnim*>(GetResAnim()), GetName()));

        if (matAnimIndex >= m_MatAnimCount)
        {
            NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(matAnimIndex, m_MatAnimCount);
            return;
        }

        EditShaderParamAnimObj::EditShaderParamMatAnim* editMatAnim = &m_pMatAnimArray[matAnimIndex];

        int curveCount = editMatAnim->GetEditAnimCurveCount();
        if (curveIndex >= curveCount)
        {
            NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(curveIndex, curveCount);
            return;
        }

        EditAnimCurve* editAnimCurve = editMatAnim->GetEditAnimCurve(curveIndex);
        editAnimCurve->Edit(resAnimCurve);
    }

}}}} // namespace nn::g3d::viewer::detail


