﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditSceneAnimObj.h"



#include <nn/g3d/g3d_ResFile.h>
#include "../g3d_Allocator.h"

namespace nn { namespace g3d { namespace viewer { namespace detail {

bool
EditSceneAnimObj::ReloadSceneAnimObj(nn::g3d::ResFile* resFile) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(resFile);

    NN_G3D_VIEWER_ASSERT_DETAIL(resFile->GetSceneAnimCount() > 0, "%s\n", NN_G3D_VIEWER_RES_NAME(resFile, GetName()));
    nn::g3d::ResSceneAnim* resAnim = resFile->GetSceneAnim(0);
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(resAnim, "%s\n", NN_G3D_VIEWER_RES_NAME(resFile, GetName()));

    Detach();
    m_pResAnim = resAnim;
    m_pResFile = resFile;

    bool result = Attach();
    if (result)
    {
        result = Setup();
        if (!result)
        {
            Detach();
        }
    }
    return result;
}

bool
EditSceneAnimObj::Attach() NN_NOEXCEPT
{
    int cameraAnimCount = m_pResAnim->GetCameraAnimCount();
    int lightAnimCount = m_pResAnim->GetLightAnimCount();
    int fogAnimCount = m_pResAnim->GetFogAnimCount();

    bool isFailed = false;
    // コールバックに渡すために、配列を作成
    {
        if (cameraAnimCount > 0)
        {
            size_t bufferSize = FixedSizeArray<CameraAnimObj*>::CalculateBufferSize(cameraAnimCount);
            void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_Obj);
            if (buffer != nullptr) // バッファ確保失敗
            {
                m_CameraAnimObjPtrArray.SetArrayBuffer(buffer, cameraAnimCount);
            }
            else
            {
                isFailed = true;
            }
        }

        if (lightAnimCount > 0)
        {
            size_t bufferSize = FixedSizeArray<LightAnimObj*>::CalculateBufferSize(lightAnimCount);
            void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_Obj);
            if (buffer != nullptr) // バッファ確保失敗
            {
                m_LightAnimObjPtrArray.SetArrayBuffer(buffer, lightAnimCount);
            }
            else
            {
                isFailed = true;
            }
        }

        if (fogAnimCount > 0)
        {
            size_t bufferSize = FixedSizeArray<FogAnimObj*>::CalculateBufferSize(fogAnimCount);
            void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_Obj);
            if (buffer != nullptr) // バッファ確保失敗
            {
                m_FogAnimObjPtrArray.SetArrayBuffer(buffer, fogAnimCount);
            }
            else
            {
                isFailed = true;
            }
        }
    }
    // 配列作成が一つでも失敗していた場合は、解放して失敗
    if (isFailed)
    {
        Detach();
        return false;
    }

    isFailed = false;
    // Edit***AnimObj を生成
    {
        if (cameraAnimCount > 0)
        {
            size_t bufferSize = FixedSizeArray<EditCameraAnimObj>::CalculateBufferSize(cameraAnimCount);
            void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
            if (buffer != nullptr) // バッファ確保失敗
            {
                 m_EditCameraAnimArray.SetArrayBuffer(buffer, cameraAnimCount);
            }
            else
            {
                isFailed = true;
            }
        }

        if (lightAnimCount > 0)
        {
            size_t bufferSize = FixedSizeArray<EditLightAnimObj>::CalculateBufferSize(lightAnimCount);
            void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
            if (buffer != nullptr) // バッファ確保失敗
            {
                m_EditLightAnimArray.SetArrayBuffer(buffer, lightAnimCount);
            }
            else
            {
                isFailed = true;
            }
        }

        if (fogAnimCount > 0)
        {
            size_t bufferSize = FixedSizeArray<EditFogAnimObj>::CalculateBufferSize(fogAnimCount);
            void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
            if (buffer != nullptr) // バッファ確保失敗
            {
                m_EditFogAnimArray.SetArrayBuffer(buffer, fogAnimCount);
            }
            else
            {
                isFailed = true;
            }
        }
    }
    // Edit***AnimObj生成が一つでも失敗していた場合は、解放して失敗
    if (isFailed)
    {
        Detach();
        return false;
    }

    isFailed = false;
    {
        for(int i = 0, end = cameraAnimCount; i < end; ++i)
        {
            ResCameraAnim* resAnim = m_pResAnim->GetCameraAnim(i);
            CameraAnimObj** animObj = m_CameraAnimObjPtrArray.UnsafeAt(i);
            EditCameraAnimObj* editAnimObj = m_EditCameraAnimArray.UnsafeAt(i);
            new (editAnimObj) EditCameraAnimObj(m_pAllocator, resAnim);
            *animObj = editAnimObj->GetAnimObj();
            isFailed |= !editAnimObj->Init();
        }
    }

    {
        for(int i = 0, end = lightAnimCount; i < end; ++i)
        {
            ResLightAnim* resAnim = m_pResAnim->GetLightAnim(i);
            LightAnimObj** animObj = m_LightAnimObjPtrArray.UnsafeAt(i);
            EditLightAnimObj* editAnimObj = m_EditLightAnimArray.UnsafeAt(i);
            new (editAnimObj) EditLightAnimObj(m_pAllocator, resAnim);
            *animObj = editAnimObj->GetAnimObj();
            isFailed |= !editAnimObj->Init();
        }
    }

    {
        for(int i = 0, end = fogAnimCount; i < end; ++i)
        {
            ResFogAnim* resAnim = m_pResAnim->GetFogAnim(i);
            FogAnimObj** animObj = m_FogAnimObjPtrArray.UnsafeAt(i);
            EditFogAnimObj* editAnimObj = m_EditFogAnimArray.UnsafeAt(i);
            new (editAnimObj) EditFogAnimObj(m_pAllocator, resAnim);
            *animObj = editAnimObj->GetAnimObj();
            isFailed |= !editAnimObj->Init();
        }
    }

    // Init に一つでも失敗していた場合は、解放して失敗
    if (isFailed)
    {
        Detach();
        return false;
    }

    return true;
} // NOLINT (readability/fn_size)

void
EditSceneAnimObj::Detach() NN_NOEXCEPT
{
    {
        for(int i = 0, end = m_EditFogAnimArray.GetCount(); i < end; ++i)
        {
            EditFogAnimObj* editAnimObj = m_EditFogAnimArray.UnsafeAt(i);
            editAnimObj->Destroy();
        }

        void* buffer = m_EditFogAnimArray.GetBufferPtr();
        if (buffer != nullptr)
        {
            m_pAllocator->Free(buffer);
            m_EditFogAnimArray.Clear();
        }
    }

    {
        void* buffer = m_FogAnimObjPtrArray.GetBufferPtr();
        if (buffer != nullptr)
        {
            m_pAllocator->Free(buffer);
            m_FogAnimObjPtrArray.Clear();
        }
    }
    {
        for(int i = 0, end = m_EditLightAnimArray.GetCount(); i < end; ++i)
        {
            EditLightAnimObj* editAnimObj = m_EditLightAnimArray.UnsafeAt(i);
            editAnimObj->Destroy();
        }

        void* buffer = m_EditLightAnimArray.GetBufferPtr();
        if (buffer != nullptr)
        {
            m_pAllocator->Free(buffer);
            m_EditLightAnimArray.Clear();
        }
    }

    {
        void* buffer = m_LightAnimObjPtrArray.GetBufferPtr();
        if (buffer != nullptr)
        {
            m_pAllocator->Free(buffer);
            m_LightAnimObjPtrArray.Clear();
        }
    }

    {
        for(int i = 0, end = m_EditCameraAnimArray.GetCount(); i < end; ++i)
        {
            EditCameraAnimObj* editCameraAnimObj = m_EditCameraAnimArray.UnsafeAt(i);
            editCameraAnimObj->Destroy();
        }

        void* buffer = m_EditCameraAnimArray.GetBufferPtr();
        if (buffer != nullptr)
        {
            m_pAllocator->Free(buffer);
            m_EditCameraAnimArray.Clear();
        }
    }

    {
        void* buffer = m_CameraAnimObjPtrArray.GetBufferPtr();
        if (buffer != nullptr)
        {
            m_pAllocator->Free(buffer);
            m_CameraAnimObjPtrArray.Clear();
        }
    }
}

void
EditSceneAnimObj::EditCameraAnimCurve(
    int cameraIndex,
    int curveIndex,
    const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT
{
    if (cameraIndex >= m_pResAnim->GetCameraAnimCount())
    {
        NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(cameraIndex, m_pResAnim->GetCameraAnimCount());
        return;
    }

    EditCameraAnimObj* editAnimObj = m_EditCameraAnimArray.UnsafeAt(cameraIndex);
    editAnimObj->EditCurve(curveIndex, resAnimCurve);
}

void
EditSceneAnimObj::EditLightAnimCurve(
    int lightIndex,
    int curveIndex,
    const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT
{
    if (lightIndex >= m_pResAnim->GetLightAnimCount())
    {
        NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(lightIndex, m_pResAnim->GetLightAnimCount());
        return;
    }
    EditLightAnimObj* editAnimObj = m_EditLightAnimArray.UnsafeAt(lightIndex);
    editAnimObj->EditCurve(curveIndex, resAnimCurve);
}

void
EditSceneAnimObj::EditFogAnimCurve(
    int fogIndex,
    int curveIndex,
    const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT
{
    if (fogIndex >= m_pResAnim->GetFogAnimCount())
    {
        NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(fogIndex, m_pResAnim->GetFogAnimCount());
        return;
    }
    EditFogAnimObj* editAnimObj = m_EditFogAnimArray.UnsafeAt(fogIndex);
    editAnimObj->EditCurve(curveIndex, resAnimCurve);
}

void
EditSceneAnimObj::Calculate() NN_NOEXCEPT
{
    {
        int animCount = m_CameraAnimObjPtrArray.GetCount();
        for (int i = 0; i < animCount; ++i)
        {
            CameraAnimObj** animObj = m_CameraAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->Calculate();
            (*animObj)->GetFrameCtrl().UpdateFrame();
        }
    }
    {
        int animCount = m_LightAnimObjPtrArray.GetCount();
        for (int i = 0; i < animCount; ++i)
        {
            LightAnimObj** animObj = m_LightAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->Calculate();
            (*animObj)->GetFrameCtrl().UpdateFrame();
        }
    }
    {
        int animCount = m_FogAnimObjPtrArray.GetCount();
        for (int i = 0; i < animCount; ++i)
        {
            FogAnimObj** animObj = m_FogAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->Calculate();
            (*animObj)->GetFrameCtrl().UpdateFrame();
        }
    }
}

void
EditSceneAnimObj::SetFrame(float frame) NN_NOEXCEPT
{
    {
        int animCount = m_CameraAnimObjPtrArray.GetCount();
        for (int i = 0; i < animCount; ++i)
        {
            CameraAnimObj** animObj = m_CameraAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->InvalidateContext();
            (*animObj)->GetFrameCtrl().SetFrame(frame);
        }
    }
    {
        int animCount = m_LightAnimObjPtrArray.GetCount();
        for (int i = 0; i < animCount; ++i)
        {
            LightAnimObj** animObj = m_LightAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->InvalidateContext();
            (*animObj)->GetFrameCtrl().SetFrame(frame);
        }
    }
    {
        int animCount = m_FogAnimObjPtrArray.GetCount();
        for (int i = 0; i < animCount; ++i)
        {
            FogAnimObj** animObj = m_FogAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->InvalidateContext();
            (*animObj)->GetFrameCtrl().SetFrame(frame);
        }
    }
}

void
EditSceneAnimObj::SetStep(float step) NN_NOEXCEPT
{
    {
        int animCount = m_CameraAnimObjPtrArray.GetCount();
        for (int i = 0; i < animCount; ++i)
        {
            CameraAnimObj** animObj = m_CameraAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->GetFrameCtrl().SetStep(step);
        }
    }
    {
        int animCount = m_LightAnimObjPtrArray.GetCount();
        for (int i = 0; i < animCount; ++i)
        {
            LightAnimObj** animObj = m_LightAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->GetFrameCtrl().SetStep(step);
        }
    }
    {
        int animCount = m_FogAnimObjPtrArray.GetCount();
        for (int i = 0; i < animCount; ++i)
        {
            FogAnimObj** animObj = m_FogAnimObjPtrArray.UnsafeAt(i);
            (*animObj)->GetFrameCtrl().SetStep(step);
        }
    }
}

void
EditSceneAnimObj::SetPlayPolicy(bool isLoopAnim) NN_NOEXCEPT
{
    {
        int animCount = m_EditCameraAnimArray.GetCount();
        for (int i = 0; i < animCount; ++i)
        {
            EditCameraAnimObj* editAnim = m_EditCameraAnimArray.UnsafeAt(i);
            editAnim->SetPlayPolicy(isLoopAnim);
        }
    }
    {
        int animCount = m_EditLightAnimArray.GetCount();
        for (int i = 0; i < animCount; ++i)
        {
            EditLightAnimObj* editAnim = m_EditLightAnimArray.UnsafeAt(i);
            editAnim->SetPlayPolicy(isLoopAnim);
        }
    }
    {
        int animCount = m_EditFogAnimArray.GetCount();
        for (int i = 0; i < animCount; ++i)
        {
            EditFogAnimObj* editAnim = m_EditFogAnimArray.UnsafeAt(i);
            editAnim->SetPlayPolicy(isLoopAnim);
        }
    }
    m_IsLoopAnim = isLoopAnim;
}

bool
EditSceneAnimObj::Setup() NN_NOEXCEPT
{
    bool isFailed = false;
    {
        for(int i = 0, end = m_EditCameraAnimArray.GetCount(); i < end; ++i)
        {
            EditCameraAnimObj* editAnimObj = m_EditCameraAnimArray.UnsafeAt(i);
            isFailed |= !editAnimObj->Setup();
        }
    }
    {
        for(int i = 0, end = m_EditLightAnimArray.GetCount(); i < end; ++i)
        {
            EditLightAnimObj* editAnimObj = m_EditLightAnimArray.UnsafeAt(i);
            isFailed |= !editAnimObj->Setup();
        }
    }
    {
        for(int i = 0, end = m_EditFogAnimArray.GetCount(); i < end; ++i)
        {
            EditFogAnimObj* editAnimObj = m_EditFogAnimArray.UnsafeAt(i);
            isFailed |= !editAnimObj->Setup();
        }
    }
    return !isFailed;
}

bool
EditCameraAnimObj::Init() NN_NOEXCEPT
{
    int curveCount = m_pResAnim->GetCurveCount();

    if (curveCount > 0)
    {
        size_t bufferSize = FixedSizeArray<EditAnimCurve>::CalculateBufferSize(curveCount);

        void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
        if (buffer == nullptr) // バッファ確保失敗
        {
            return false;
        }
        m_EditAnimCurveArray.SetArrayBuffer(buffer, curveCount);
        for(int i = 0, end = m_EditAnimCurveArray.GetCount(); i < end; ++i)
        {
            nn::g3d::ResAnimCurve* resAnimCurve = m_pResAnim->GetCurve(i);
            EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(i);
            new (editAnimCurve) EditAnimCurve(m_pAllocator, resAnimCurve);
            editAnimCurve->Setup();
        }
    }
    return true;
}

void
EditCameraAnimObj::Destroy() NN_NOEXCEPT
{
    for(int i = 0, end = m_EditAnimCurveArray.GetCount(); i < end; ++i)
    {
        EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(i);
        editAnimCurve->Cleanup();
        editAnimCurve->~EditAnimCurve();
    }

    m_pAllocator->Free(m_EditAnimCurveArray.GetBufferPtr());

    m_AnimObj.ResetResource();
    m_WorkBuffer.Clear();
}

bool
EditCameraAnimObj::Setup() NN_NOEXCEPT
{
    CameraAnimObj::Builder builder;
    builder.SetContextDisabled();
    builder.Reserve(m_pResAnim);
    builder.CalculateMemorySize();
    size_t bufferSize = builder.GetWorkMemorySize();
    bool success = m_WorkBuffer.Resize(bufferSize);

    NN_G3D_VIEWER_ASSERT_DETAIL(success, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pResAnim, GetName()));//今は止める

    success = builder.Build(&m_AnimObj, m_WorkBuffer.GetWorkBufferPtr(), m_WorkBuffer.GetSize());
    NN_G3D_VIEWER_ASSERT(success);

    m_AnimObj.SetResource(m_pResAnim);
    SetPlayPolicy(true);
    return true;
}

void
EditCameraAnimObj::EditCurve(
    int curveIndex,
    const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_DETAIL(curveIndex != -1, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pResAnim, GetName()));

    int curveCount = m_pResAnim->GetCurveCount();
    if (curveIndex >= curveCount)
    {
        NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(curveIndex, curveCount);
        return;
    }

    EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(curveIndex);
    editAnimCurve->Edit(resAnimCurve);
}

void
EditCameraAnimObj::SetPlayPolicy(bool isLoopAnim) NN_NOEXCEPT
{
    float start = m_AnimObj.GetDefaultFrameCtrl().GetStartFrame();
    float end = m_AnimObj.GetDefaultFrameCtrl().GetEndFrame();

    if (isLoopAnim)
    {
        m_AnimFrameCtrl.Initialize(start, end, AnimFrameCtrl::PlayLoop);
    }
    else
    {
        m_AnimFrameCtrl.Initialize(start, end, AnimFrameCtrl::PlayOneTime);
    }
    m_AnimObj.SetFrameCtrl(&m_AnimFrameCtrl);
}

bool
EditLightAnimObj::Init() NN_NOEXCEPT
{
    int curveCount = m_pResAnim->GetCurveCount();
    if (curveCount > 0)
    {
        size_t bufferSize = FixedSizeArray<EditAnimCurve>::CalculateBufferSize(curveCount);

        void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
        if (buffer == nullptr) // バッファ確保失敗
        {
            return false;
        }
        m_EditAnimCurveArray.SetArrayBuffer(buffer, curveCount);
        for(int i = 0, end = m_EditAnimCurveArray.GetCount(); i < end; ++i)
        {
            nn::g3d::ResAnimCurve* resAnimCurve = m_pResAnim->GetCurve(i);
            EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(i);
            new (editAnimCurve) EditAnimCurve(m_pAllocator, resAnimCurve);
            editAnimCurve->Setup();
        }
    }
    return true;
}

void
EditLightAnimObj::Destroy() NN_NOEXCEPT
{
    for(int i = 0, end = m_EditAnimCurveArray.GetCount(); i < end; ++i)
    {
        EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(i);
        editAnimCurve->Cleanup();
        editAnimCurve->~EditAnimCurve();
    }

    m_pAllocator->Free(m_EditAnimCurveArray.GetBufferPtr());

    m_AnimObj.ResetResource();
    m_WorkBuffer.Clear();
}

bool
EditLightAnimObj::Setup() NN_NOEXCEPT
{
    LightAnimObj::Builder builder;
    builder.SetContextDisabled();
    builder.Reserve(m_pResAnim);

    builder.CalculateMemorySize();
    size_t bufferSize = builder.GetWorkMemorySize();
    bool result = m_WorkBuffer.Resize(bufferSize);
    NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pResAnim, GetName()));//今は止める

    bool success = builder.Build(&m_AnimObj, m_WorkBuffer.GetWorkBufferPtr(), m_WorkBuffer.GetSize());
    NN_G3D_VIEWER_ASSERT(success);
    m_AnimObj.SetResource(m_pResAnim);
    SetPlayPolicy(true);
    return true;
}

void
EditLightAnimObj::EditCurve(
    int curveIndex,
    const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_DETAIL(curveIndex != -1, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pResAnim, GetName()));

    int curveCount = m_pResAnim->GetCurveCount();
    if (curveIndex >= curveCount)
    {
        NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(curveIndex, curveCount);
        return;
    }

    EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(curveIndex);
    editAnimCurve->Edit(resAnimCurve);
}

void
EditLightAnimObj::SetPlayPolicy(bool isLoopAnim) NN_NOEXCEPT
{
    float start = m_AnimObj.GetDefaultFrameCtrl().GetStartFrame();
    float end = m_AnimObj.GetDefaultFrameCtrl().GetEndFrame();

    if (isLoopAnim)
    {
        m_AnimFrameCtrl.Initialize(start, end, AnimFrameCtrl::PlayLoop);
    }
    else
    {
        m_AnimFrameCtrl.Initialize(start, end, AnimFrameCtrl::PlayOneTime);
    }
    m_AnimObj.SetFrameCtrl(&m_AnimFrameCtrl);
}

bool
EditFogAnimObj::Init() NN_NOEXCEPT
{
    int curveCount = m_pResAnim->GetCurveCount();

    if (curveCount > 0)
    {
        size_t bufferSize = FixedSizeArray<EditAnimCurve>::CalculateBufferSize(curveCount);

        void* buffer = m_pAllocator->Allocate(bufferSize, nn::g3d::detail::Alignment_Default, AllocateType_EditObj);
        if (buffer == nullptr) // バッファ確保失敗
        {
            return false;
        }
        m_EditAnimCurveArray.SetArrayBuffer(buffer, curveCount);
        for(int i = 0, end = m_EditAnimCurveArray.GetCount(); i < end; ++i)
        {
            nn::g3d::ResAnimCurve* resAnimCurve = m_pResAnim->GetCurve(i);
            EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(i);
            new (editAnimCurve) EditAnimCurve(m_pAllocator, resAnimCurve);
            editAnimCurve->Setup();
        }
    }
    return true;
}

void
EditFogAnimObj::Destroy() NN_NOEXCEPT
{
    for(int i = 0, end = m_EditAnimCurveArray.GetCount(); i < end; ++i)
    {
        EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(i);
        editAnimCurve->Cleanup();
        editAnimCurve->~EditAnimCurve();
    }

    m_pAllocator->Free(m_EditAnimCurveArray.GetBufferPtr());

    m_AnimObj.ResetResource();
    m_WorkBuffer.Clear();
}

bool
EditFogAnimObj::Setup() NN_NOEXCEPT
{
    FogAnimObj::Builder builder;
    builder.SetContextDisabled();
    builder.Reserve(m_pResAnim);

    //FogAnimObj::InitArg initArg;
    //initArg.SetContextDisabled();
    //initArg.Reserve(m_pResAnim);

    builder.CalculateMemorySize();
    size_t bufferSize = builder.GetWorkMemorySize();
    //size_t bufferSize = FogAnimObj::CalculateBufferSize(initArg);
    bool result = m_WorkBuffer.Resize(bufferSize);
    NN_G3D_VIEWER_ASSERT_DETAIL(result, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pResAnim, GetName()));//今は止める

    bool success = builder.Build(&m_AnimObj, m_WorkBuffer.GetWorkBufferPtr(), m_WorkBuffer.GetSize());
    NN_G3D_VIEWER_ASSERT(success);
    //m_AnimObj.Init(initArg, m_WorkBuffer.GetWorkBufferPtr(), m_WorkBuffer.GetSize());
    m_AnimObj.SetResource(m_pResAnim);
    SetPlayPolicy(true);
    return true;
}

void
EditFogAnimObj::EditCurve(
    int curveIndex,
    const nn::g3d::ResAnimCurve* resAnimCurve) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_DETAIL(curveIndex != -1, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pResAnim, GetName()));

    int curveCount = m_pResAnim->GetCurveCount();
    if (curveIndex >= curveCount)
    {
        NN_G3D_VIEWER_ASSERT_INDEX_BOUNDS(curveIndex, curveCount);
        return;
    }

    EditAnimCurve* editAnimCurve = m_EditAnimCurveArray.UnsafeAt(curveIndex);
    editAnimCurve->Edit(resAnimCurve);
}

void
EditFogAnimObj::SetPlayPolicy(bool isLoopAnim) NN_NOEXCEPT
{
    float start = m_AnimObj.GetDefaultFrameCtrl().GetStartFrame();
    float end = m_AnimObj.GetDefaultFrameCtrl().GetEndFrame();

    if (isLoopAnim)
    {
        m_AnimFrameCtrl.Initialize(start, end, AnimFrameCtrl::PlayLoop);
    }
    else
    {
        m_AnimFrameCtrl.Initialize(start, end, AnimFrameCtrl::PlayOneTime);
    }
    m_AnimObj.SetFrameCtrl(&m_AnimFrameCtrl);
}

bool EditSceneAnimObj::ContainsLoopAnimInCameraAnim() const NN_NOEXCEPT
{
    for(int i = 0, end = m_EditCameraAnimArray.GetCount(); i < end; ++i)
    {
        const EditCameraAnimObj* editAnimObj = m_EditCameraAnimArray.UnsafeAt(i);
        if (editAnimObj->IsLoopAnim())
        {
            return true;
        }
    }

    return false;
}

bool EditSceneAnimObj::ContainsLoopAnimInLightAnim() const NN_NOEXCEPT
{
    for(int i = 0, end = m_EditLightAnimArray.GetCount(); i < end; ++i)
    {
        const EditLightAnimObj* editAnimObj = m_EditLightAnimArray.UnsafeAt(i);
        if (editAnimObj->IsLoopAnim())
        {
            return true;
        }
    }

    return false;
}

bool EditSceneAnimObj::ContainsLoopAnimInFogAnim() const NN_NOEXCEPT
{
    for(int i = 0, end = m_EditFogAnimArray.GetCount(); i < end; ++i)
    {
        const EditFogAnimObj* editAnimObj = m_EditFogAnimArray.UnsafeAt(i);
        if (editAnimObj->IsLoopAnim())
        {
            return true;
        }
    }

    return false;
}

}}}} // namespace nn::g3d::edit::detail


