﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/viewer/g3d_ViewerDefine.h>

#include "../model/g3d_EditModelObj.h"
#include "../g3d_IEditTargetObj.h"
#include "../util/g3d_ViewerUtility.h"
#include "../util/g3d_DynamicArray.h"
#include "../g3d_ViewerKeyManager.h"

#include <nn/g3d/g3d_ModelObj.h>
#include <nn/g3d/g3d_SkeletalAnimObj.h>
#include <nn/g3d/g3d_MaterialAnimObj.h>
#include <nn/g3d/g3d_ShapeAnimObj.h>
#include <nn/g3d/g3d_ResFile.h>

#include <stdint.h>

namespace nn { namespace g3d {

class ModelAnimObj;
class ResFile;

namespace viewer {
namespace detail {

class Allocator;

//! @brief アニメーションを編集するためのクラスです。
// このクラスではリソースひとつに対して複数のModelObjとModelAnimObjをバインドします。
class EditAnimObj : IEditTargetObj
{
public:
    explicit EditAnimObj(
        Allocator* pAllocator,
        nn::g3d::ResFile* pResFile,
        ViewerAnimKind animKind) NN_NOEXCEPT
        : m_SetupFlag(false)
        , m_pAllocator(pAllocator)
        , m_pResFile(pResFile)
        , m_BoundModelInfos(pAllocator, nn::g3d::detail::Alignment_Default, nullptr)
        , m_IsLoopAnim(false)
        , m_AnimKind(animKind)
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pResFile);
        NN_G3D_VIEWER_ASSERT(IsValidResFile(pResFile));
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pAllocator, "%s\n", NN_G3D_VIEWER_RES_NAME(m_pResFile, GetName()));
    }

    virtual ~EditAnimObj() NN_NOEXCEPT
    {
        for (int animIndex = 0, animCount = m_BoundModelInfos.GetCount(); animIndex < animCount; ++animIndex)
        {
            DestroyAnimObj(animIndex);
        }
    }

    //! @brief モデルをバインドします。
    BindResult BindModelObj(EditModelObj* pBindTargetEditModelObj) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_DETAIL(!IsModelBound(pBindTargetEditModelObj), "%s", pBindTargetEditModelObj->GetResource()->GetName());
        AddBoundModelObj(pBindTargetEditModelObj);
        ModelAnimObj* pBoundAnimObj = m_BoundModelInfos[m_BoundModelInfos.GetCount() - 1]->pAnimObj;
        BindResult result = SetupInternal(pBindTargetEditModelObj, pBoundAnimObj);
        ResetPlayPolicy();
        m_SetupFlag = true;
        return result;
    }

    //! @brief モデルをアンバインドします。
    void UnbindModelObj(EditModelObj* pEditModelObj) NN_NOEXCEPT
    {
        int index = IndexOf(pEditModelObj);
        NN_G3D_VIEWER_ASSERT_DETAIL(index >= 0, "%s is not bound", pEditModelObj->GetResource()->GetName());

        ResetToOriginalValue(pEditModelObj);

        DestroyAnimObj(index);
        BoundModelInfo* pInfo = m_BoundModelInfos[index];
        pInfo->~BoundModelInfo();
        m_pAllocator->Free(pInfo);
        m_BoundModelInfos.EraseByIndex(index);
    }

    //! @brief アニメーションカーブ編集用のデータを作成します。
    virtual bool CreateDataForEditingAnimCurve() = 0;

    //! @brief アニメーションカーブ編集用のデータを破棄します。
    virtual void DestroyDataForEditingAnimCurve() = 0;

    //! @biref バインドされているモデルにアニメーションを適用します。
    virtual void ApplyAnimTo(EditModelObj* pEditModelObj) NN_NOEXCEPT
    {
        ModelAnimObj* pAssociatedAnimObj = GetAssociatedModelAnimObj(pEditModelObj);
        for (int modelIndex = 0, modelCount = pEditModelObj->GetAttachedModelObjCount(); modelIndex < modelCount; ++modelIndex)
        {
            ModelObj* pModelObj = pEditModelObj->GetTargetModelObj(modelIndex);
            pAssociatedAnimObj->ApplyTo(pModelObj);
        }
    }

    //! @brief アニメーションリソースの名前を取得します。
    const char* GetName() const NN_NOEXCEPT
    {
        return GetResName(m_pResFile, ConvertToFileDataKind(m_AnimKind));
    }

    //! @brief アニメーションの種類を取得します。
    ViewerAnimKind GetAnimKind() const NN_NOEXCEPT
    {
        return m_AnimKind;
    }

    //! @brief アニメーションリソースをリロードします。
    bool ReloadResource(nn::g3d::ResFile* resFile) NN_NOEXCEPT;

    void InvalidateContext(EditModelObj* pEditModelObj) NN_NOEXCEPT
    {
        ModelAnimObj* pAnimObj = GetAssociatedAnimObj(pEditModelObj);
        if (pAnimObj == nullptr)
        {
            return;
        }

        pAnimObj->InvalidateContext();
    }

    bool IsResFileBound(ResFile* pResFile) const NN_NOEXCEPT
    {
        return m_pResFile == pResFile;
    }

    float GetFrameCount() const NN_NOEXCEPT
    {
        if (m_BoundModelInfos.GetCount() == 0)
        {
            return 0.0f;
        }

        return m_BoundModelInfos[0]->pAnimObj->GetFrameCtrl().GetEndFrame();
    }

    bool IsLoopAnim() const NN_NOEXCEPT
    {
        if (m_BoundModelInfos.GetCount() == 0)
        {
            return m_IsLoopAnim;
        }

        return m_BoundModelInfos[0]->pAnimObj->GetDefaultFrameCtrl().GetPlayPolicy() == AnimFrameCtrl::PlayLoop;
    }

    bool IsCalculated(EditModelObj* pEditModelObj) const NN_NOEXCEPT
    {
        int index = IndexOf(pEditModelObj);
        NN_G3D_VIEWER_ASSERT_DETAIL(index >= 0, "%s is not bound", pEditModelObj->GetResource()->GetName());

        return m_BoundModelInfos[index]->isCalculated;
    }

    bool IsPaused(EditModelObj* pEditModelObj) const NN_NOEXCEPT
    {
        int index = IndexOf(pEditModelObj);
        NN_G3D_VIEWER_ASSERT_DETAIL(index >= 0, "%s is not bound", pEditModelObj->GetResource()->GetName());
        return m_BoundModelInfos[index]->isPaused;
    }

    float GetPauseFrame(EditModelObj* pEditModelObj) const NN_NOEXCEPT
    {
        int index = IndexOf(pEditModelObj);
        NN_G3D_VIEWER_ASSERT_DETAIL(index >= 0, "%s is not bound", pEditModelObj->GetResource()->GetName());
        return m_BoundModelInfos[index]->pauseFrame;
    }

    void Calculate(EditModelObj* pEditModelObj) NN_NOEXCEPT
    {
        int index = IndexOf(pEditModelObj);
        NN_G3D_VIEWER_ASSERT_DETAIL(index >= 0, "%s is not bound", pEditModelObj->GetResource()->GetName());

        if (m_BoundModelInfos[index]->isCalculated)
        {
            return;
        }

        ModelAnimObj* pAnimObj = m_BoundModelInfos[index]->pAnimObj;
        pAnimObj->Calculate();
        pAnimObj->GetFrameCtrl().UpdateFrame();
        m_BoundModelInfos[index]->isCalculated = true;
    }

    void SetFrame(EditModelObj* pEditModelObj, float frame) NN_NOEXCEPT
    {
        int index = IndexOf(pEditModelObj);
        NN_G3D_VIEWER_ASSERT_DETAIL(index >= 0, "%s is not bound", pEditModelObj->GetResource()->GetName());
        m_BoundModelInfos[index]->pAnimObj->GetFrameCtrl().SetFrame(frame);
    }

    void SetFrameForAllBoundModels(float frame) NN_NOEXCEPT
    {
        for (int animIndex = 0, animCount = m_BoundModelInfos.GetCount(); animIndex < animCount; ++animIndex)
        {
            ModelAnimObj* pAnimObj = m_BoundModelInfos[animIndex]->pAnimObj;
            pAnimObj->GetFrameCtrl().SetFrame(frame);
        }
    }

    void SetStep(float step) NN_NOEXCEPT
    {
        for (int animIndex = 0, animCount = m_BoundModelInfos.GetCount(); animIndex < animCount; ++animIndex)
        {
            ModelAnimObj* pAnimObj = m_BoundModelInfos[animIndex]->pAnimObj;
            pAnimObj->GetFrameCtrl().SetStep(step);
        }
    }

    void SetPlayPolicy(bool isLoopAnim) NN_NOEXCEPT
    {
        m_IsLoopAnim = isLoopAnim;
        ResetPlayPolicy();
    }

    void ResetPlayPolicy() NN_NOEXCEPT
    {
        if (m_BoundModelInfos.GetCount() == 0)
        {
            return;
        }

        for (int index = 0, size = m_BoundModelInfos.GetCount(); index < size; ++index)
        {
            ModelAnimObj* pAnimObj = m_BoundModelInfos[index]->pAnimObj;
            float start = pAnimObj->GetDefaultFrameCtrl().GetStartFrame();
            float end = pAnimObj->GetDefaultFrameCtrl().GetEndFrame();
            if (m_IsLoopAnim)
            {
                pAnimObj->GetFrameCtrl().Initialize(start, end, AnimFrameCtrl::PlayLoop);
            }
            else
            {
                pAnimObj->GetFrameCtrl().Initialize(start, end, AnimFrameCtrl::PlayOneTime);
            }
        }
    }

    void ClearCalculateFlag() NN_NOEXCEPT
    {
        for (int index = 0, end = m_BoundModelInfos.GetCount(); index < end; ++index)
        {
            m_BoundModelInfos[index]->isCalculated = false;
        }
    }

    void SetPauseFlag(EditModelObj* pEditModelObj, bool enable, float frame) NN_NOEXCEPT
    {
        int index = IndexOf(pEditModelObj);
        NN_G3D_VIEWER_ASSERT_DETAIL(index >= 0, "%s is not bound", pEditModelObj->GetResource()->GetName());

        BoundModelInfo* pInfo = m_BoundModelInfos[index];
        pInfo->isPaused = enable;
        pInfo->pauseFrame = frame;
    }

    EditModelObj* GetBoundEditModelObj(int index) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_REQUIRES_RANGE(index, 0, m_BoundModelInfos.GetCount());
        return m_BoundModelInfos[index]->pBoundEditModelObj;
    }

    int GetBoundEditModelObjCount() const NN_NOEXCEPT
    {
        return m_BoundModelInfos.GetCount();
    }

    ViewerKeyType GetResFileKey() const NN_NOEXCEPT
    {
        return ViewerKeyManager::GetInstance().FindKey(m_pResFile);
    }

    void PrintAnimInfo() const NN_NOEXCEPT
    {
        ResFile* pResFile = m_pResFile;
        NN_UNUSED(pResFile);
        NN_G3D_VIEWER_ASSERT_NOT_NULL(pResFile);

        NN_G3D_VIEWER_DEBUG_PRINT("%s (%s)\n",
            GetResName(pResFile, ConvertToFileDataKind(this->GetAnimKind())),
            GetAnimKindString(this->GetAnimKind()));

        NN_G3D_VIEWER_DEBUG_PRINT("  Bound to:\n");
        for (int i = 0, end = m_BoundModelInfos.GetCount(); i < end; ++i)
        {
            EditModelObj* pEditModelObj = m_BoundModelInfos[i]->pBoundEditModelObj;
            const ResModel* pResModel = pEditModelObj->GetResource();
            NN_UNUSED(pEditModelObj);
            NN_UNUSED(pResModel);
            NN_G3D_VIEWER_DEBUG_PRINT("    %s\n", pResModel->GetName());
        }
    }

    bool IsModelBound(EditModelObj* pTargetEditModelObj) const NN_NOEXCEPT
    {
        for (int i = 0, end = m_BoundModelInfos.GetCount(); i < end; ++i)
        {
            EditModelObj* pBoundEditModelObj = m_BoundModelInfos[i]->pBoundEditModelObj;
            if (pBoundEditModelObj == pTargetEditModelObj)
            {
                return true;
            }
        }

        return false;
    }

    bool HasBoundModel() const NN_NOEXCEPT
    {
        return m_BoundModelInfos.GetCount() > 0;
    }

protected:
    virtual BindResult SetupInternal(EditModelObj* pBindTargetEditModelObj, ModelAnimObj* pBoundAnimObj) = 0;
    virtual ModelAnimObj* CreateAnimObj() = 0;
    virtual void ResetToOriginalValue(EditModelObj* pBoundEditModelObj) = 0;

    void UnbindAllModelObjs() NN_NOEXCEPT
    {
        while (m_BoundModelInfos.GetCount() > 0)
        {
            UnbindModelObj(m_BoundModelInfos[0]->pBoundEditModelObj);
        }
    }

    bool IsValidResFile(ResFile* pResFile) NN_NOEXCEPT
    {
        switch (GetAnimKind())
        {
        case ViewerAnimKind_SkeletalAnim:
            return pResFile->GetSkeletalAnimCount() > 0;
        case ViewerAnimKind_ShaderParamAnim:
        case ViewerAnimKind_ColorAnim:
        case ViewerAnimKind_TextureSrtAnim:
        case ViewerAnimKind_TexturePatternAnim:
        case ViewerAnimKind_MaterialVisibilityAnim:
        case ViewerAnimKind_MaterialAnim:
            return pResFile->GetMaterialAnimCount() > 0;
        case ViewerAnimKind_BoneVisibilityAnim:
            return pResFile->GetBoneVisibilityAnimCount() > 0;
        case ViewerAnimKind_ShapeAnim:
            return pResFile->GetShapeAnimCount() > 0;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void* GetResAnim() NN_NOEXCEPT
    {
        switch (GetAnimKind())
        {
        case ViewerAnimKind_SkeletalAnim:
            return m_pResFile->GetSkeletalAnim(0);
        case ViewerAnimKind_ShaderParamAnim:
        case ViewerAnimKind_ColorAnim:
        case ViewerAnimKind_TextureSrtAnim:
        case ViewerAnimKind_TexturePatternAnim:
        case ViewerAnimKind_MaterialVisibilityAnim:
        case ViewerAnimKind_MaterialAnim:
            return m_pResFile->GetMaterialAnim(0);
        case ViewerAnimKind_BoneVisibilityAnim:
            return m_pResFile->GetBoneVisibilityAnim(0);
        case ViewerAnimKind_ShapeAnim:
            return m_pResFile->GetShapeAnim(0);
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void* GetResAnim() const NN_NOEXCEPT
    {
        switch (GetAnimKind())
        {
        case ViewerAnimKind_SkeletalAnim:
            return m_pResFile->GetSkeletalAnim(0);
        case ViewerAnimKind_ShaderParamAnim:
        case ViewerAnimKind_ColorAnim:
        case ViewerAnimKind_TextureSrtAnim:
        case ViewerAnimKind_TexturePatternAnim:
        case ViewerAnimKind_MaterialVisibilityAnim:
        case ViewerAnimKind_MaterialAnim:
            return m_pResFile->GetMaterialAnim(0);
        case ViewerAnimKind_BoneVisibilityAnim:
            return m_pResFile->GetBoneVisibilityAnim(0);
        case ViewerAnimKind_ShapeAnim:
            return m_pResFile->GetShapeAnim(0);
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    template <typename T>
    T* GetResAnim() NN_NOEXCEPT
    {
        return static_cast<T*>(GetResAnim());
    }

    template <typename T>
    const T* GetResAnim() const NN_NOEXCEPT
    {
        return static_cast<const T*>(GetResAnim());
    }

    ModelAnimObj* GetAnimObj(int index) NN_NOEXCEPT
    {
        return m_BoundModelInfos[index]->pAnimObj;
    }

    template <typename T>
    T* GetAnimObj(int index) NN_NOEXCEPT
    {
        return static_cast<T*>(m_BoundModelInfos[index]->pAnimObj);
    }

    ModelAnimObj* GetAssociatedAnimObj(EditModelObj* pEditModelObj) NN_NOEXCEPT
    {
        int index = IndexOf(pEditModelObj);
        if (index == -1)
        {
            return nullptr;
        }

        return m_BoundModelInfos[index]->pAnimObj;
    }

    ModelAnimObj* GetAssociatedModelAnimObj(EditModelObj* pEditModelObj) NN_NOEXCEPT
    {
        int index = IndexOf(pEditModelObj);
        NN_G3D_VIEWER_ASSERT_DETAIL(index >= 0, "%s is not bound", pEditModelObj->GetResource()->GetName());

       return m_BoundModelInfos[index]->pAnimObj;
    }

    bool                    m_SetupFlag;
    Allocator*             m_pAllocator;
    nn::g3d::ResFile*  m_pResFile;

private:
    struct BoundModelInfo
    {
        EditModelObj* pBoundEditModelObj;
        ModelAnimObj* pAnimObj;
        AnimFrameCtrl animFrameCtrl;
        float  pauseFrame;
        bool isPaused;
        bool isCalculated;

        BoundModelInfo() NN_NOEXCEPT
            : pBoundEditModelObj(nullptr)
            , pAnimObj(nullptr)
            , pauseFrame(0.0f)
            , isPaused(false)
            , isCalculated(false)
        {
        }

        BoundModelInfo(const BoundModelInfo& rhs) NN_NOEXCEPT
        {
            operator = (rhs);
        }

        BoundModelInfo& operator=(const BoundModelInfo& rhs) NN_NOEXCEPT
        {
            pBoundEditModelObj = rhs.pBoundEditModelObj;
            pAnimObj = rhs.pAnimObj;
            pauseFrame = rhs.pauseFrame;
            isPaused = rhs.isPaused;
            isCalculated = rhs.isCalculated;
            animFrameCtrl.Initialize(
                rhs.animFrameCtrl.GetStartFrame(),
                rhs.animFrameCtrl.GetEndFrame(),
                rhs.animFrameCtrl.GetPlayPolicy());

            return *this;
        }
    };

    void FreeAnimObjBuffers() NN_NOEXCEPT;

    void SetResource(nn::g3d::ResFile* resFile) NN_NOEXCEPT
    {
        m_pResFile = resFile;
    }

    void GetBoundEditModelObjs(DynamicArray<BoundModelInfo>* pDestination) NN_NOEXCEPT
    {
        for (int i = 0, end = GetBoundEditModelObjCount(); i < end; ++i)
        {
            (*pDestination).PushBack(*m_BoundModelInfos[i]);
        }
    }

    void AddBoundModelObjs(DynamicArray<BoundModelInfo>& source) NN_NOEXCEPT
    {
        for (int i = 0, end = GetBoundEditModelObjCount(); i < end; ++i)
        {
            AddBoundModelObj((source)[i].pBoundEditModelObj);
        }
    }

    void AddBoundModelObj(EditModelObj* pEditModelObj) NN_NOEXCEPT
    {
        BoundModelInfo* pInfo;
        {
            void* buffer = m_pAllocator->Allocate(sizeof(BoundModelInfo), nn::g3d::detail::Alignment_Default, AllocateType_Other);
            pInfo = new (buffer) BoundModelInfo();
            NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pInfo, "%s\n", pEditModelObj->GetResource()->GetName());
            m_BoundModelInfos.PushBack(pInfo);
        }
        pInfo->pBoundEditModelObj = pEditModelObj;

        ModelAnimObj* pModelAnimObj = CreateAnimObj();
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pModelAnimObj, "%s\n", pEditModelObj->GetResource()->GetName());

        pModelAnimObj->SetFrameCtrl(&pInfo->animFrameCtrl);

        pInfo->pAnimObj = pModelAnimObj;
    }

    void DestroyAnimObj(int index) NN_NOEXCEPT
    {
        ModelAnimObj* pAnimObj = m_BoundModelInfos[index]->pAnimObj;
        void* buffer = pAnimObj->GetBufferPtr();
        if (buffer)
        {
            m_pAllocator->Free(pAnimObj->GetBufferPtr());
        }
        pAnimObj->~ModelAnimObj();
        m_pAllocator->Free(pAnimObj);
    }

    int IndexOf(EditModelObj* pEditModelObj) const NN_NOEXCEPT
    {
        for (int index = 0, size = m_BoundModelInfos.GetCount(); index < size; ++index)
        {
            if (m_BoundModelInfos[index]->pBoundEditModelObj == pEditModelObj)
            {
                return index;
            }
        }

        return -1;
    }

    DynamicArray<BoundModelInfo*> m_BoundModelInfos;
    bool          m_IsLoopAnim;
    ViewerAnimKind m_AnimKind;

private:
    NN_DISALLOW_COPY(EditAnimObj);
};

}}}} // namespace nn::g3d::edit::detail


