﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_ShaderObj.h>
#include <nn/gfx/util/gfx_ObjectDebugLabel.h>
#include <algorithm>

namespace nn { namespace g3d {

void ShadingModelObj::InitializeArgument::CalculateMemorySize() NN_NOEXCEPT
{
    const ResShadingModel* pRes = GetResource();
    int dirtyFlagCount = pRes->GetStaticOptionCount();

    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_MemoryBlock[blockIndex].Initialize();
    }

    m_MemoryBlock[MemoryBlockIndex_ShaderKey].SetSizeBy<Bit32>(1, pRes->GetStaticKeyLength());
    m_MemoryBlock[MemoryBlockIndex_OptionKey].SetSizeBy<Bit32>(1, pRes->GetStaticKeyLength());
    m_MemoryBlock[MemoryBlockIndex_DirtyFlagSet].SetSize(FlagSet::CalcBufferSize(dirtyFlagCount, m_BufferingCount));
    m_MemoryBlock[MemoryBlockIndex_Buffer].SetSizeBy<nn::gfx::Buffer>(1, m_BufferingCount);

    m_WorkMemory.Initialize();
    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_WorkMemory.Append(&m_MemoryBlock[blockIndex]);
    }
}

//--------------------------------------------------------------------------------------------------

bool ShadingModelObj::Initialize(const InitializeArgument& arg, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pBuffer != NULL,                      NN_G3D_RES_GET_NAME(arg.GetResource(), GetName()));
    NN_G3D_REQUIRES(IsAligned(pBuffer, Alignment_Buffer), NN_G3D_RES_GET_NAME(arg.GetResource(), GetName()));

    if (arg.IsMemoryCalculated() == false)
    {
        return false;
    }
    if (arg.GetWorkMemorySize() > bufferSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    const ResShadingModel* pRes = arg.GetResource();

    // メンバの初期化。
    m_pRes              = pRes;
    m_Flag              = 0;
    m_BufferingCount    = static_cast<uint8_t>(arg.GetBufferingCount());
    m_pShaderKey        = arg.GetBuffer<Bit32>(pBuffer, InitializeArgument::MemoryBlockIndex_ShaderKey);
    m_pOptionKey        = arg.GetBuffer<Bit32>(pBuffer, InitializeArgument::MemoryBlockIndex_OptionKey);
    m_pOptionBlockArray = arg.GetBuffer<nn::gfx::Buffer>(pBuffer, InitializeArgument::MemoryBlockIndex_Buffer);
    m_pUserPtr          = NULL;
    m_pBufferPtr        = pBuffer;
    m_OptionBlockSize   = 0;
    m_pMemoryPool       = NULL;
    m_MemoryPoolOffset  = 0;

    // ダーティフラグの初期化
    int dirtyFlagCount = m_pRes->GetStaticOptionCount();
    m_DirtyFlagSet.Initialize(dirtyFlagCount, m_BufferingCount,
                              arg.GetBuffer(pBuffer, InitializeArgument::MemoryBlockIndex_DirtyFlagSet),
                              arg.GetBufferSize(InitializeArgument::MemoryBlockIndex_DirtyFlagSet));

    // デフォルトキーで初期化
    m_pRes->WriteDefaultStaticKey(m_pShaderKey);
    std::copy(m_pShaderKey, m_pShaderKey + m_pRes->GetStaticKeyLength(), m_pOptionKey);
    // バリエーションの全範囲で初期化します。
    m_ShaderRange.pBegin =
        m_pRes->ToData().pKeyTable.Get() + m_pRes->GetStaticKeyLength();
    m_ShaderRange.pEnd =
        m_ShaderRange.pBegin + m_pRes->GetKeyLength() * m_pRes->GetShaderProgramCount();

    return true;
}

size_t ShadingModelObj::GetBlockBufferAlignment(nn::gfx::Device* pDevice) const NN_NOEXCEPT
{
    int idxBlock = m_pRes->GetSystemBlockIndex(ResUniformBlockVar::Type_Option);
    size_t size;
    if (idxBlock < 0)
    {
        size = 0;
    }
    else
    {
        const ResUniformBlockVar* pBlock = m_pRes->GetUniformBlock(idxBlock);
        size = pBlock->GetSize();
    }
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize(size);
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_ConstantBuffer );

    return nn::gfx::Buffer::GetBufferAlignment(pDevice, info);
}

size_t ShadingModelObj::CalculateBlockBufferSize(nn::gfx::Device* pDevice)
{
    int idxBlock = m_pRes->GetSystemBlockIndex(ResUniformBlockVar::Type_Option);
    if (idxBlock < 0)
    {
        return 0;
    }
    else
    {
        const ResUniformBlockVar* pBlock = m_pRes->GetUniformBlock(idxBlock);
        return nn::util::align_up(pBlock->GetSize(), GetBlockBufferAlignment(pDevice)) * m_BufferingCount;
    }
}

void ShadingModelObj::SetupBlockBufferImpl(nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t offset, size_t memoryPoolSize) NN_NOEXCEPT
{
    NN_UNUSED(memoryPoolSize);

    NN_G3D_ASSERT(m_BufferingCount, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    size_t size = CalculateBlockBufferSize(pDevice) / m_BufferingCount;

    m_DirtyFlagSet.ClearMainFlag();
    m_DirtyFlagSet.ClearAllSubFlag();

    nn::gfx::Buffer::InfoType bufferInfo;
    bufferInfo.SetDefault();
    bufferInfo.SetSize(size);
    bufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
    ptrdiff_t memoryPoolOffset = offset;
    for (int bufferIndex = 0; bufferIndex < m_BufferingCount; ++bufferIndex)
    {
        nn::gfx::Buffer* pOptionBlock = new(&m_pOptionBlockArray[bufferIndex]) nn::gfx::Buffer;
        pOptionBlock->Initialize(pDevice, bufferInfo, pMemoryPool, memoryPoolOffset, size);
        nn::gfx::util::SetBufferDebugLabel(pOptionBlock, "g3d_OptionUniformBlock");
        memoryPoolOffset += nn::util::align_up(size, GetBlockBufferAlignment(pDevice));
    }
    m_OptionBlockSize = size;

    // 初期化
    int idxBlock = m_pRes->GetSystemBlockIndex(ResUniformBlockVar::Type_Option);
    if (idxBlock >= 0)
    {
        const ResUniformBlockVar* pBlock = m_pRes->GetUniformBlock(idxBlock);
        if (pBlock->GetSize() > 0)
        {
            for (int bufferIndex = 0; bufferIndex < m_BufferingCount; ++bufferIndex)
            {
                nn::gfx::Buffer* pOptionBlock = &m_pOptionBlockArray[bufferIndex];
                void* pBuffer = pOptionBlock->Map<void>();
                memset(pBuffer, 0, pBlock->GetSize());
                pOptionBlock->FlushMappedRange(0, m_OptionBlockSize);
                pOptionBlock->Unmap();
            }
        }
    }

    m_Flag |= Flag_BlockBufferValid;

    return;
}

bool ShadingModelObj::SetupBlockBuffer(nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t offset, size_t memoryPoolSize) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pMemoryPool != NULL || memoryPoolSize == 0, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(IsBlockBufferValid() == false,              NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    size_t size = CalculateBlockBufferSize(pDevice);
    if (size > memoryPoolSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }
    // sizeが0の場合、nn::gfx::bufferを作成する必要がないので、リターンする
    if (size == 0)
    {
        return true;
    }

    m_pMemoryPool = pMemoryPool;
    m_MemoryPoolOffset = offset;

    SetupBlockBufferImpl(pDevice, pMemoryPool, offset, memoryPoolSize);

    return true;
}

void ShadingModelObj::CleanupBlockBuffer(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsBlockBufferValid() == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    for (int bufferIndex = 0; bufferIndex < m_BufferingCount; ++bufferIndex)
    {
        m_pOptionBlockArray[bufferIndex].Finalize(pDevice);
    }

    m_Flag ^= Flag_BlockBufferValid;

    m_pMemoryPool = NULL;
    m_MemoryPoolOffset = 0;
}

void ShadingModelObj::CalculateOptionBlock(int bufferIndex) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(m_pRes != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (m_OptionBlockSize == 0)
    {
        return;
    }

    m_DirtyFlagSet.Caclulate();

    if (!m_DirtyFlagSet.IsSubSetDirty(bufferIndex))
    {
        return;
    }

    uint32_t* pOptionBuffer = m_pOptionBlockArray[bufferIndex].Map< uint32_t >();

    int dirtyFlagCount = GetStaticOptionCount();
    for (int idxFlag32 = 0, flag32Count = (dirtyFlagCount + 31) >> 5;
        idxFlag32 < flag32Count; ++idxFlag32)
    {
        Bit32* pFlag = m_DirtyFlagSet.GetFlagSet(bufferIndex);
        Bit32 flag32 = pFlag[idxFlag32];
        while (flag32)
        {
            int bitIndex = 31 - CountLeadingZeros(flag32);
            int optionIndex = (idxFlag32 << 5) + bitIndex;
            const ResShaderOption* pOption = GetStaticOption(optionIndex);
            if (pOption->GetBranchOffset() != ResShaderOption::InvalidOffset)
            {
                // シェーダーコンパイラがデッドストリップを行う場合があるので判定します。
                // int32_t であっても uint32_t として扱う。
                const uint32_t* pChoiceValues = pOption->ToData().pChoiceValues.Get();
                NN_G3D_ASSERT(pChoiceValues != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
                int idxChoice = pOption->ReadStaticKey(m_pOptionKey);
                uint32_t* pBranch = AddOffset<uint32_t>(pOptionBuffer, pOption->GetBranchOffset());
                *pBranch = pChoiceValues[idxChoice];
#if defined(NN_BUILD_CONFIG_OS_COS)
                nn::util::SwapEndian(pBranch);
#endif
            }
            flag32 ^= 0x1 << bitIndex;
        }
    }

    m_DirtyFlagSet.ClearSubFlag(bufferIndex);

    m_pOptionBlockArray[bufferIndex].FlushMappedRange(0, m_OptionBlockSize);
    m_pOptionBlockArray[bufferIndex].Unmap();
}

void ShadingModelObj::ClearStaticKey()
{
    NN_G3D_REQUIRES(m_pRes != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    for (int idxOption = 0, optionCount = GetStaticOptionCount(); idxOption < optionCount; ++idxOption)
    {
        const ResShaderOption* pOption = GetStaticOption(idxOption);
        WriteStaticKey(idxOption, pOption->GetDefaultIndex());
    }
}

void ShadingModelObj::WriteStaticKey(int optionIndex, int choiceIndex) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(m_pRes != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES_RANGE(optionIndex, 0, GetStaticOptionCount(), NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    const ResShaderOption* pOption = GetStaticOption(optionIndex);
    NN_G3D_REQUIRES_RANGE(choiceIndex, 0, pOption->GetChoiceCount(), NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    if (!pOption->IsBranch())
    {
        pOption->WriteStaticKey(m_pShaderKey, choiceIndex);
    }
    else
    {
        // branch が true のオプションは m_pShaderKey をデフォルト値のままにします。
        // UniformBlock に書き込むための値を m_pOptionKey に記録します。
        pOption->WriteStaticKey(m_pOptionKey, choiceIndex);
        if (pOption->GetBranchOffset() != ResShaderOption::InvalidOffset)
        {
            SetDirtyFlag(optionIndex);
        }
    }
}

int ShadingModelObj::ReadStaticKey(int optionIndex) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(m_pRes != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES_RANGE(optionIndex, 0, GetStaticOptionCount(), NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    const ResShaderOption* pOption = GetStaticOption(optionIndex);
    if (!pOption->IsBranch())
    {
        return pOption->ReadStaticKey(m_pShaderKey);
    }
    else
    {
        // branch が true のオプションは m_pOptionKey に記録されています。
        return pOption->ReadStaticKey(m_pOptionKey);
    }
}

int ShadingModelObj::PrintRawKeyTo(char* pStr, int strLength) const NN_NOEXCEPT
{
    return ResShadingModel::PrintKeyTo(pStr, strLength, m_pShaderKey, m_pRes->GetStaticKeyLength());
}

int ShadingModelObj::PrintKeyTo(char* pStr, int strLength) const NN_NOEXCEPT
{
    uint32_t currentKey[NN_G3D_CONFIG_MAX_SHADER_KEY] = { 0 };
    for (int idxOption = 0, optionCount = GetStaticOptionCount(); idxOption < optionCount; ++idxOption)
    {
        const ResShaderOption* pOption = GetStaticOption(idxOption);
        int choice = pOption->ReadStaticKey(pOption->IsBranch() ? m_pOptionKey : m_pShaderKey);
        pOption->WriteStaticKey(currentKey, choice);
    }
    return ResShadingModel::PrintKeyTo(pStr, strLength, currentKey, m_pRes->GetStaticKeyLength());
}

int ShadingModelObj::PrintRawOptionTo(char* pStr, int strLength) const NN_NOEXCEPT
{
    return m_pRes->PrintStaticOptionTo(pStr, strLength, m_pShaderKey);
}

int ShadingModelObj::PrintOptionTo(char* pStr, int strLength) const NN_NOEXCEPT
{
    uint32_t currentKey[NN_G3D_CONFIG_MAX_SHADER_KEY] = { 0 };
    for (int idxOption = 0, optionCount = GetStaticOptionCount(); idxOption < optionCount; ++idxOption)
    {
        const ResShaderOption* pOption = GetStaticOption(idxOption);
        int choice = pOption->ReadStaticKey(pOption->IsBranch() ? m_pOptionKey : m_pShaderKey);
        pOption->WriteStaticKey(currentKey, choice);
    }
    return m_pRes->PrintStaticOptionTo(pStr, strLength, currentKey);
}

//--------------------------------------------------------------------------------------------------

void ShaderSelector::InitializeArgument::CalculateMemorySize() NN_NOEXCEPT
{
    const ResShadingModel* pRes = GetShadingModel()->GetResource();

    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_MemoryBlock[blockIndex].Initialize();
    }

    m_MemoryBlock[MemoryBlockIndex_ShaderKey].SetSizeBy<Bit32>(1, pRes->GetDynamicKeyLength());
    m_MemoryBlock[MemoryBlockIndex_OptionKey].SetSizeBy<Bit32>(1, pRes->GetDynamicKeyLength());
    m_MemoryBlock[MemoryBlockIndex_LastShaderKey].SetSizeBy<Bit32>(1, pRes->GetDynamicKeyLength());

    m_WorkMemory.Initialize();
    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_WorkMemory.Append(&m_MemoryBlock[blockIndex]);
    }
}

bool ShaderSelector::Initialize(const InitializeArgument& arg, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pBuffer != NULL,                      NN_G3D_RES_GET_NAME(arg.GetShadingModel()->GetResource(), GetName()));
    NN_G3D_REQUIRES(IsAligned(pBuffer, Alignment_Buffer), NN_G3D_RES_GET_NAME(arg.GetShadingModel()->GetResource(), GetName()));

    if (arg.IsMemoryCalculated() == false)
    {
        return false;
    }
    if (arg.GetWorkMemorySize() > bufferSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    ShadingModelObj* pShadingModel = arg.GetShadingModel();
    const ResShadingModel* pRes = pShadingModel->GetResource();

    // メンバの初期化。
    m_pShadingModel = pShadingModel;
    m_pShaderKey = arg.GetBuffer<Bit32>(pBuffer, InitializeArgument::MemoryBlockIndex_ShaderKey);
    m_pOptionKey = arg.GetBuffer<Bit32>(pBuffer, InitializeArgument::MemoryBlockIndex_OptionKey);
    m_pLastShaderKey = arg.GetBuffer<Bit32>(pBuffer, InitializeArgument::MemoryBlockIndex_LastShaderKey);
    m_pBufferPtr = pBuffer;

    int keyLength = m_pShadingModel->GetResource()->GetDynamicKeyLength();
    if (keyLength > 0)
    {
        pRes->WriteDefaultDynamicKey(m_pShaderKey);        // デフォルトキーで初期化。
        std::copy(m_pShaderKey, m_pShaderKey + keyLength, m_pOptionKey);
        pRes->WriteInvalidDynamicKey(m_pLastShaderKey);    // 無効なキーで初期化。
        m_pProgram = NULL;
    }
    else
    {
        // DynamicKey が存在しない場合は StaticKey で一意に決まっているはずです。
        const ShaderRange& range = m_pShadingModel->GetShaderRange();
        int idxProgram = pRes->FindProgramIndex(range, m_pShaderKey);
        NN_G3D_ASSERT(idxProgram != ShaderLocationProgramNone, NN_G3D_RES_GET_NAME(arg.GetShadingModel()->GetResource(), GetName()));
        m_pProgram = pRes->GetShaderProgram(idxProgram);
    }

    return true;
}

bool ShaderSelector::UpdateVariation(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(m_pShadingModel != NULL, NN_G3D_RES_GET_NAME(m_pShadingModel->GetResource(), GetName()));

    // シェーダーキーの dynamic 部分のみを使用します。
    // static 部分は ShadingModelObj で決定します。
    const ResShadingModel* pRes = m_pShadingModel->GetResource();
    int keyLength = pRes->GetDynamicKeyLength();
    const Bit32* pShaderKey = m_pShaderKey;
    Bit32* pLastShaderKey = m_pLastShaderKey;

    if (!std::equal(pShaderKey, pShaderKey + keyLength, pLastShaderKey))
    {
        // 前回のシェーダーキーの dynamic 部分に変更があった場合のみ更新を試みます。
        const ShaderRange& range = m_pShadingModel->GetShaderRange();
        int idxProgram = pRes->FindProgramIndex(range, pShaderKey);
        if (idxProgram == ShaderLocationProgramNone)
        {
            // 見つからなかった場合は最後のキーが無効であるとみなします。
            pRes->WriteInvalidDynamicKey(m_pLastShaderKey);
            m_pProgram = NULL;
            return false;
        }
        else
        {
            // 見つかった場合は最後のキーとして記録します。
            std::copy(pShaderKey, pShaderKey + keyLength, pLastShaderKey);
            m_pProgram = pRes->GetShaderProgram(idxProgram);
        }

    }
    // 遅延コンパイルのため更新を試みます。
    // リアルタイム編集のために毎フレーム呼び出します。
    ResShaderProgram* resShaderProgram = const_cast<ResShaderProgram*>(m_pProgram);
    resShaderProgram->Update(pDevice);
    return true;
}

void ShaderSelector::ClearDynamicKey()
{
    NN_G3D_REQUIRES(m_pShadingModel != NULL, NN_G3D_RES_GET_NAME(m_pShadingModel->GetResource(), GetName()));
    for (int idxOption = 0, optionCount = GetDynamicOptionCount(); idxOption < optionCount; ++idxOption)
    {
        const ResShaderOption* pOption = GetDynamicOption(idxOption);
        WriteDynamicKey(idxOption, pOption->GetDefaultIndex());
    }
}

void ShaderSelector::WriteDynamicKey(int optionIndex, int choiceIndex) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(m_pShadingModel != NULL, NN_G3D_RES_GET_NAME(m_pShadingModel->GetResource(), GetName()));

    const ResShaderOption* pOption = GetDynamicOption(optionIndex);
    if (!pOption->IsBranch())
    {
        pOption->WriteDynamicKey(m_pShaderKey, choiceIndex);
    }
    else
    {
        // branch が true のオプションは m_pShaderKey をデフォルト値のままにします。
        // ここではオプションの情報を記録しますが、シェーダーには情報を送りません。
        // 別途情報を送信し、オプションがデフォルトの場合にそれを参照するように
        // シェーダーを記述する必要があります。
        pOption->WriteDynamicKey(m_pOptionKey, choiceIndex);
    }
}

int ShaderSelector::ReadDynamicKey(int optionIndex) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(m_pShadingModel != NULL, NN_G3D_RES_GET_NAME(m_pShadingModel->GetResource(), GetName()));

    const ResShaderOption* pOption = GetDynamicOption(optionIndex);
    if (!pOption->IsBranch())
    {
        return pOption->ReadDynamicKey(m_pShaderKey);
    }
    else
    {
        // branch が true のオプションは m_pOptionKey に記録されています。
        return pOption->ReadDynamicKey(m_pOptionKey);
    }
}

int ShaderSelector::PrintRawKeyTo(char* pStr, int strLength) const NN_NOEXCEPT
{
    const ResShadingModel* pRes = m_pShadingModel->GetResource();
    return ResShadingModel::PrintKeyTo(pStr, strLength, m_pShaderKey, pRes->GetDynamicKeyLength());
}

int ShaderSelector::PrintKeyTo(char* pStr, int strLength) const NN_NOEXCEPT
{
    const ResShadingModel* pRes = m_pShadingModel->GetResource();
    uint32_t currentKey[NN_G3D_CONFIG_MAX_SHADER_KEY] = { 0 };
    for (int idxOption = 0, optionCount = GetDynamicOptionCount();
        idxOption < optionCount; ++idxOption)
    {
        const ResShaderOption* pOption = GetDynamicOption(idxOption);
        int choice = pOption->ReadDynamicKey(pOption->IsBranch() ? m_pOptionKey : m_pShaderKey);
        pOption->WriteDynamicKey(currentKey, choice);
    }
    return ResShadingModel::PrintKeyTo(pStr, strLength, currentKey, pRes->GetDynamicKeyLength());
}

int ShaderSelector::PrintRawOptionTo(char* pStr, int strLength) const NN_NOEXCEPT
{
    const ResShadingModel* pRes = m_pShadingModel->GetResource();
    return pRes->PrintDynamicOptionTo(pStr, strLength, m_pShaderKey);
}

int ShaderSelector::PrintOptionTo(char* pStr, int strLength) const NN_NOEXCEPT
{
    const ResShadingModel* pRes = m_pShadingModel->GetResource();
    uint32_t currentKey[NN_G3D_CONFIG_MAX_SHADER_KEY] = { 0 };
    for (int idxOption = 0, optionCount = GetDynamicOptionCount();
        idxOption < optionCount; ++idxOption)
    {
        const ResShaderOption* pOption = GetDynamicOption(idxOption);
        int choice = pOption->ReadDynamicKey(pOption->IsBranch() ? m_pOptionKey : m_pShaderKey);
        pOption->WriteDynamicKey(currentKey, choice);
    }
    return pRes->PrintDynamicOptionTo(pStr, strLength, currentKey);
}

}} // namespace nn::g3d

