﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_SceneAnimObj.h>
#include <nn/g3d/detail/g3d_Perf.h>

namespace nn { namespace g3d {

void CameraAnimObj::InitializeArgument::CalculateMemorySize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid() == true);

    int curveCount = GetMaxCurveCount();

    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_MemoryBlock[blockIndex].Initialize();
    }

    m_MemoryBlock[MemoryBlockIndex_ResultBuffer].SetSizeBy<CameraAnimResult>(1, 1);
    size_t size = IsContextEnabled() ? nn::util::align_up(sizeof(AnimFrameCache) * curveCount, Alignment_Default) : 0;
    m_MemoryBlock[MemoryBlockIndex_FrameCacheArray].SetSize(size);

    m_WorkMemory.Initialize();
    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_WorkMemory.Append(&m_MemoryBlock[blockIndex]);
    }
}

bool CameraAnimObj::Initialize(const InitializeArgument& arg, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pBuffer != NULL);
    NN_SDK_REQUIRES(IsAligned(pBuffer, Alignment_Buffer));

    if (!arg.IsMemoryCalculated() || !arg.IsValid())
    {
        return false;
    }
    if (arg.GetWorkMemorySize() > bufferSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    int curveCount = arg.GetMaxCurveCount();

    // メンバの初期化。
    SetBufferPtr(pBuffer);
    // フレーム関係はリソース設定時に初期化
    m_pRes = NULL;
    GetContext().Initialize(arg.GetBuffer<AnimFrameCache>(pBuffer, InitializeArgument::MemoryBlockIndex_FrameCacheArray), curveCount);
    SetResultBuffer(arg.GetBuffer(pBuffer, InitializeArgument::MemoryBlockIndex_ResultBuffer));

    return true;
}

void CameraAnimObj::SetResource(const ResCameraAnim* pRes) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pRes);

    m_pRes = pRes;

    bool loop = (pRes->ToData().flag & AnimFlag_PlayPolicyLoop) != 0;
    ResetFrameCtrl(pRes->GetFrameCount(), loop);
    GetContext().SetCurveCount(pRes->GetCurveCount());
    CameraAnimObj::ClearResult();
}

void CameraAnimObj::ClearResult() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);

    m_pRes->Initialize(GetResult());
}

void CameraAnimObj::Calculate() NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("CameraAnimObj::Calculate");
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);

    if (IsFrameChanged())
    {
        float frame = GetFrameCtrl().GetFrame();
        AnimContext& context = GetContext();
        if (context.IsFrameCacheValid())
        {
            m_pRes->Evaluate(GetResult(), frame, context.GetFrameCacheArray(0));
        }
        else
        {
            m_pRes->Evaluate(GetResult(), frame);
        }
        UpdateLastFrame();
    }
}

//--------------------------------------------------------------------------------------------------

void LightAnimObj::InitializeArgument::CalculateMemorySize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid() == true);

    int curveCount = GetMaxCurveCount();

    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_MemoryBlock[blockIndex].Initialize();
    }

    m_MemoryBlock[MemoryBlockIndex_ResultBuffer].SetSizeBy<LightAnimResult>(1, 1);
    size_t size = IsContextEnabled() ? nn::util::align_up(sizeof(AnimFrameCache) * curveCount, Alignment_Default) : 0;
    m_MemoryBlock[MemoryBlockIndex_FrameCacheArray].SetSize(size);

    m_WorkMemory.Initialize();
    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_WorkMemory.Append(&m_MemoryBlock[blockIndex]);
    }
}

bool LightAnimObj::Initialize(const InitializeArgument& arg, void* pBuffer, size_t bufferSize)
{
    NN_SDK_REQUIRES(pBuffer != NULL);
    NN_SDK_REQUIRES(IsAligned(pBuffer, Alignment_Buffer));

    if (!arg.IsMemoryCalculated() || !arg.IsValid())
    {
        return false;
    }
    if (arg.GetWorkMemorySize() > bufferSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    int curveCount = arg.GetMaxCurveCount();

    // メンバの初期化。
    SetBufferPtr(pBuffer);
    // フレーム関係はリソース設定時に初期化
    m_pRes = NULL;
    GetContext().Initialize(arg.GetBuffer<AnimFrameCache>(pBuffer, InitializeArgument::MemoryBlockIndex_FrameCacheArray), curveCount);
    SetResultBuffer(arg.GetBuffer(pBuffer, InitializeArgument::MemoryBlockIndex_ResultBuffer));

    return true;
}

void LightAnimObj::SetResource(const ResLightAnim* pRes) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pRes);

    m_pRes = pRes;

    bool loop = (pRes->ToData().flag & AnimFlag_PlayPolicyLoop) != 0;
    ResetFrameCtrl(pRes->GetFrameCount(), loop);
    GetContext().SetCurveCount(pRes->GetCurveCount());
    LightAnimObj::ClearResult();
}

void LightAnimObj::ClearResult() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);

    m_pRes->Initialize(GetResult());
}

void LightAnimObj::Calculate() NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("LightAnimObj::Calculate");
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);

    if (IsFrameChanged())
    {
        float frame = GetFrameCtrl().GetFrame();
        AnimContext& context = GetContext();
        if (context.IsFrameCacheValid())
        {
            m_pRes->Evaluate(GetResult(), frame, context.GetFrameCacheArray(0));
        }
        else
        {
            m_pRes->Evaluate(GetResult(), frame);
        }
        UpdateLastFrame();
    }
}

//--------------------------------------------------------------------------------------------------

void FogAnimObj::InitializeArgument::CalculateMemorySize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid() == true);

    int curveCount = GetMaxCurveCount();

    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_MemoryBlock[blockIndex].Initialize();
    }

    m_MemoryBlock[MemoryBlockIndex_ResultBuffer].SetSizeBy<FogAnimResult>(1, 1);
    size_t size = IsContextEnabled() ? nn::util::align_up(sizeof(AnimFrameCache) * curveCount, Alignment_Default) : 0;
    m_MemoryBlock[MemoryBlockIndex_FrameCacheArray].SetSize(size);

    m_WorkMemory.Initialize();
    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_WorkMemory.Append(&m_MemoryBlock[blockIndex]);
    }
}

bool FogAnimObj::Initialize(const InitializeArgument& arg, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pBuffer != NULL);
    NN_SDK_REQUIRES(IsAligned(pBuffer, Alignment_Buffer));

    if (!arg.IsMemoryCalculated() || !arg.IsValid())
    {
        return false;
    }
    if (arg.GetWorkMemorySize() > bufferSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    int curveCount = arg.GetMaxCurveCount();

    // メンバの初期化。
    SetBufferPtr(pBuffer);
    // フレーム関係はリソース設定時に初期化
    m_pRes = NULL;
    GetContext().Initialize(arg.GetBuffer<AnimFrameCache>(pBuffer, InitializeArgument::MemoryBlockIndex_FrameCacheArray), curveCount);
    SetResultBuffer(arg.GetBuffer(pBuffer, InitializeArgument::MemoryBlockIndex_ResultBuffer));

    return true;
}

void FogAnimObj::SetResource(const ResFogAnim* pRes) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pRes);

    m_pRes = pRes;

    bool loop = (pRes->ToData().flag & AnimFlag_PlayPolicyLoop) != 0;
    ResetFrameCtrl(pRes->GetFrameCount(), loop);
    GetContext().SetCurveCount(pRes->GetCurveCount());
    FogAnimObj::ClearResult();
}

void FogAnimObj::ClearResult() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);

    m_pRes->Initialize(GetResult());
}

void FogAnimObj::Calculate() NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("FogAnimObj::Calculate");
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);

    if (IsFrameChanged())
    {
        float frame = GetFrameCtrl().GetFrame();
        AnimContext& context = GetContext();
        if (context.IsFrameCacheValid())
        {
            m_pRes->Evaluate(GetResult(), frame, context.GetFrameCacheArray(0));
        }
        else
        {
            m_pRes->Evaluate(GetResult(), frame);
        }
        UpdateLastFrame();
    }
}

//--------------------------------------------------------------------------------------------------

void SceneAnimHelper::CalculateAimCameraMtx(nn::util::Matrix4x3fType* pMtx, const CameraAnimResult* pResult) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMtx);
    NN_SDK_REQUIRES_NOT_NULL(pResult);

    nn::util::Vector3fType pos;
    nn::util::Vector3fType aim;

    VectorLoad(&pos, pResult->view.pos);
    VectorLoad(&aim, pResult->view.aim);

    nn::util::Vector3fType back;
    VectorSubtract(&back, pos, aim);

    if (VectorGetX(back) == 0.0f && VectorGetZ(back) == 0.0f)
    {
        // カメラと注視点の XZ 座標が同じ場合ツイストは定義されない。

        // 真上を見る。
        if (VectorGetY(back) <= 0.0f)
        {
            nn::util::MatrixSet( pMtx,
                                 1.0f,             0.0f,             0.0f,
                                 0.0f,             0.0f,            -1.0f,
                                 0.0f,             1.0f,             0.0f,
                                -VectorGetX(pos), -VectorGetZ(pos),  VectorGetY(pos) );
        }
        // 真下を見る。
        else
        {
            nn::util::MatrixSet( pMtx,
                                 1.0f,              0.0f,             0.0f,
                                 0.0f,              0.0f,             1.0f,
                                 0.0f,             -1.0f,             0.0f,
                                -VectorGetX(pos),   VectorGetZ(pos), -VectorGetY(pos) );
        }
    }
    else
    {
        float twistSin = nn::util::SinTable(nn::util::RadianToAngleIndex(pResult->view.twist));
        float twistCos = nn::util::CosTable(nn::util::RadianToAngleIndex(pResult->view.twist));
        nn::util::Vector3fType right;
        nn::util::Vector3fType up;
        nn::util::Vector3fType rightTwist;
        nn::util::Vector3fType upTwist;

        VectorSet(&right, VectorGetZ(back), 0.0f, -VectorGetX(back));
        VectorNormalize(&right, right);
        VectorNormalize(&back, back);
        VectorCross(&up, back, right);

        //
        // r軸, u軸をru平面上でcameraTwistだけ半時計回りに回転させて rightTwist を求める。
        // right.y == 0であることに注意。
        //
        VectorSet(&rightTwist,
                  twistSin * VectorGetX(up) + twistCos * VectorGetX(right),
                  twistSin * VectorGetY(up),
                  twistSin * VectorGetZ(up) + twistCos * VectorGetZ(right));

        VectorSet(&upTwist,
                  twistCos * VectorGetX(up) - twistSin * VectorGetX(right),
                  twistCos * VectorGetY(up),
                  twistCos * VectorGetZ(up) - twistSin * VectorGetZ(right));

        nn::util::MatrixSet( pMtx,
                             VectorGetX(rightTwist),      VectorGetX(upTwist),      VectorGetX(back),
                             VectorGetY(rightTwist),      VectorGetY(upTwist),      VectorGetY(back),
                             VectorGetZ(rightTwist),      VectorGetZ(upTwist),      VectorGetZ(back),
                            -VectorDot(pos, rightTwist), -VectorDot(pos, upTwist), -VectorDot(pos, back) );
    }
}

void SceneAnimHelper::CalculateRotateCameraMtx(nn::util::Matrix4x3fType* pMtx, const CameraAnimResult* pResult) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMtx);
    NN_SDK_REQUIRES_NOT_NULL(pResult);

    nn::util::Vector3fType pos;
    nn::util::Vector3fType rotate;

    VectorLoad(&pos, pResult->view.pos);
    VectorLoad(&rotate, pResult->view.rotate);

    float sinX = nn::util::SinTable(nn::util::RadianToAngleIndex(pResult->view.rotate[0]));
    float sinY = nn::util::SinTable(nn::util::RadianToAngleIndex(pResult->view.rotate[1]));
    float sinZ = nn::util::SinTable(nn::util::RadianToAngleIndex(pResult->view.rotate[2]));
    float cosX = nn::util::CosTable(nn::util::RadianToAngleIndex(pResult->view.rotate[0]));
    float cosY = nn::util::CosTable(nn::util::RadianToAngleIndex(pResult->view.rotate[1]));
    float cosZ = nn::util::CosTable(nn::util::RadianToAngleIndex(pResult->view.rotate[2]));

    // Z軸, X軸, Y軸の順番で回転させている。
    //Vec3 right, up, back;
    nn::util::Vector3fType right;
    nn::util::Vector3fType up;
    nn::util::Vector3fType back;

    VectorSet(&right,
              sinX * sinY * sinZ + cosY * cosZ,
              cosX * sinZ,
              sinX * cosY * sinZ - sinY * cosZ);

    VectorSet(&up,
              sinX * sinY * cosZ - cosY * sinZ,
              cosX * cosZ,
              sinX * cosY * cosZ + sinY * sinZ);

    VectorSet(&back,
              cosX * sinY,
              -sinX,
              cosX * cosY);

    nn::util::MatrixSet( pMtx,
                         VectorGetX(right),      VectorGetX(up),      VectorGetX(back),
                         VectorGetY(right),      VectorGetY(up),      VectorGetY(back),
                         VectorGetZ(right),      VectorGetZ(up),      VectorGetZ(back),
                        -VectorDot(pos, right), -VectorDot(pos, up), -VectorDot(pos, back) );
}

void SceneAnimHelper::CalculateOrthoProjMtx(nn::util::Matrix4x4fType* pMtx, const CameraAnimResult* pResult) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMtx);
    NN_SDK_REQUIRES_NOT_NULL(pResult);

    float halfHeight = pResult->proj.height;
    float halfWidth = pResult->proj.aspect * halfHeight;
    float nearZ =  pResult->proj.nearZ;
    float farZ = pResult->proj.farZ;
    MatrixOrthographicOffCenterRightHanded(pMtx, -halfWidth, halfWidth, -halfHeight, halfHeight, nearZ, farZ);
}

void SceneAnimHelper::CalculatePerspProjMtx(nn::util::Matrix4x4fType* pMtx, const CameraAnimResult* pResult) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMtx);
    NN_SDK_REQUIRES_NOT_NULL(pResult);

    float fovy = pResult->proj.fovy;
    float aspect = pResult->proj.aspect;
    float nearZ =  pResult->proj.nearZ;
    float farZ = pResult->proj.farZ;
    MatrixPerspectiveFieldOfViewRightHanded(pMtx, fovy, aspect, nearZ, farZ);
}

void SceneAnimHelper::CalculateOrthoProjTexMtx(nn::util::Matrix4x3fType* pMtx, const CameraAnimResult* pResult) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMtx);
    NN_SDK_REQUIRES_NOT_NULL(pResult);

    float right = pResult->proj.aspect * pResult->proj.height;
    float left = -right;
    float top = pResult->proj.height;
    float bottom = -top;

    NN_SDK_ASSERT(top != bottom);
    NN_SDK_ASSERT(left != right);

    float scaleS = 0.5f;
    float scaleT = -0.5f;
    float translateS = 0.5f;
    float translateT = 0.5f;
    float reverseWidth = 1.0f / (right - left);
    float reverseHeight = 1.0f / (top - bottom);

    MatrixSet(pMtx,
              // row0
              2.0f * reverseWidth * scaleS,
              0.0f,
              0.0f,
              // row1
              0.0f,
              (2.0f * reverseHeight) * scaleT,
              0.0f,
              // row2
              0.0f,
              0.0f,
              0.0f,
              // row3
              ((-(right + left) * reverseWidth) * scaleS) + translateS,
              ((-(top + bottom) * reverseHeight) * scaleT) + translateT,
              1.0f);
}

void SceneAnimHelper::CalculatePerspProjTexMtx(nn::util::Matrix4x3fType* pMtx, const CameraAnimResult* pResult) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMtx);
    NN_SDK_REQUIRES_NOT_NULL(pResult);

    float fovy = pResult->proj.fovy;
    float aspect = pResult->proj.aspect;

    NN_SDK_ASSERT((fovy > 0.0f) && (fovy < nn::util::FloatPi));
    NN_SDK_ASSERT((aspect != 0.0f));

    // クリップ座標では上方向が正になっており、
    // テクスチャー座標は下方向が正になっているため、
    // SCALE_T でプロジェクションの上下方向を反転しています。
    float scaleS = 0.5f;
    float scaleT = -0.5f;
    float translateS = 0.5f;
    float translateT = 0.5f;

    float angle = fovy * 0.5f;
    float cot = 1.0f / std::tan(angle);

    MatrixSet(pMtx,
              // row0
              (cot / aspect) * scaleS,
              0.0f,
              0.0f,
              // row1
              0.0f,
              cot * scaleT,
              0.0f,
              // row2
              -translateS,
              -translateT,
              -1.0f,
              // row3
              0.0f,
              0.0f,
              0.0f);
}

}} // namespace nn::g3d
