﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_ResSkeletalAnim.h>
#include <algorithm>

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_SHADOW

namespace nn { namespace g3d {

void ResBoneAnim::Initialize(BoneAnimResult* pResult, const ResBone* pBone) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pResult != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    Bit32 flag = ToData().flag;
    const float* pBaseValue = ToData().pBaseValueArray.Get();
    static const float identityRT[] = { 0.0f, 0.0f, 0.0f, 1.0f };

    if (flag & Flag_BaseScale)
    {
        std::memcpy(&pResult->scale, ToData().pBaseValueArray.Get(), sizeof(pResult->scale));
        pBaseValue += sizeof(pResult->scale) / sizeof(float);
    }
    else
    {
        static const float identityS[] = { 1.0f, 1.0f, 1.0f };
        const float* pScale = pBone != NULL ? pBone->ToData().scale.v : identityS;
        std::memcpy(&pResult->scale, pScale, sizeof(pResult->scale));
    }

    if (flag & Flag_BaseRotate)
    {
        std::memcpy(&pResult->quat, pBaseValue, sizeof(pResult->quat));
        pBaseValue += sizeof(pResult->quat) / sizeof(float);
    }
    else
    {
        const float *pRotate = pBone != NULL ? pBone->ToData().rotate.quat.v : identityRT;
        std::memcpy(&pResult->quat, pRotate, sizeof(pResult->quat));
    }

    if (flag & Flag_BaseTranslate)
    {
        std::memcpy(&pResult->translate, pBaseValue, sizeof(pResult->translate));
    }
    else
    {
        const float *pTranslate = pBone != NULL ? pBone->ToData().translate.v : identityRT;
        std::memcpy(&pResult->translate, pTranslate, sizeof(pResult->translate));
    }

    // Mask_TransForm bit は LocalMtx の flag に引き継がれる。
    // RotateMode は BlendResult() のみで使用され、LocalMtx には引き継がれない。
    pResult->flag = (flag & Mask_TransForm);
    if (pBone != NULL)
    {
        pResult->flag |= pBone->GetRotateMode();
    }
}

void ResBoneAnim::Evaluate(BoneAnimResult* pResult, float frame) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pResult != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    for (int idxCurve = 0, curveCount = GetCurveCount(); idxCurve < curveCount; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ToData().targetOffset);
        *pTarget = pCurve->EvaluateFloat(frame);
    }
}

void ResBoneAnim::Evaluate(BoneAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pResult != NULL, NN_G3D_RES_GET_NAME(this, GetName()));
    NN_G3D_REQUIRES(pFrameCache != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    for (int idxCurve = 0, curveCount = GetCurveCount(); idxCurve < curveCount; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ToData().targetOffset);
        *pTarget = pCurve->EvaluateFloat(frame, &pFrameCache[idxCurve]);
    }

}

//--------------------------------------------------------------------------------------------------

BindResult ResSkeletalAnim::PreBind(const ResSkeleton* pSkeleton) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pSkeleton != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    ToData().pBindSkeleton.Set(const_cast<ResSkeleton*>(pSkeleton));
    uint16_t* pBindIndexArray = ToData().pBindIndexArray.Get();

    BindResult result;
    for (int idxAnim = 0, animCount = GetBoneAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        const ResBoneAnim* pBoneAnim = GetBoneAnim(idxAnim);

        const char* pName = pBoneAnim->ToData().pName.Get()->GetData();
        int idxTarget = pSkeleton->FindBoneIndex(pName);
        if (idxTarget >= 0)
        {
            pBindIndexArray[idxAnim] = static_cast<uint16_t>(idxTarget);
            result.SetSuccessBit();
        }
        else
        {
            pBindIndexArray[idxAnim] = AnimFlag_NotBound;
            result.SetFailureBit();
        }
    }
    return result;
}

BindResult ResSkeletalAnim::BindCheck(const ResSkeleton* pSkeleton) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pSkeleton != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;
    for (int idxAnim = 0, animCount = GetBoneAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        const ResBoneAnim* pBoneAnim = GetBoneAnim(idxAnim);

        const char* pName = pBoneAnim->ToData().pName.Get()->GetData();
        int idxTarget = pSkeleton->FindBoneIndex(pName);
        if (idxTarget >= 0)
        {
            result.SetSuccessBit();
        }
        else
        {
            result.SetFailureBit();
        }
    }
    return result;
}

bool ResSkeletalAnim::BakeCurve(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    if (bufferSize == 0)
    {
        return true;
    }
    if (pBuffer == NULL ||
        bufferSize < GetBakedSize())
    {
        return false;
    }
    for (int idxAnim = 0, animCount = GetBoneAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        ResBoneAnim* pBoneAnim = GetBoneAnim(idxAnim);
        for (int idxCurve = 0, curveCount = pBoneAnim->GetCurveCount(); idxCurve < curveCount; ++idxCurve)
        {
            ResAnimCurve* curve = pBoneAnim->GetCurve(idxCurve);
            size_t size = curve->CalculateBakedFloatSize();
            curve->BakeFloat(pBuffer, size);
            pBuffer = AddOffset(pBuffer, size);
        }
    }
    ToData().flag |= Flag_CurveBaked;

    return true;
}

void* ResSkeletalAnim::ResetCurve() NN_NOEXCEPT
{
    void* pBuffer = NULL;
    if (IsCurveBaked())
    {
        bool foundCurve = false;
        for (int idxAnim = 0, animCount = GetBoneAnimCount(); idxAnim < animCount; ++idxAnim)
        {
            ResBoneAnim* pBoneAnim = GetBoneAnim(idxAnim);
            for (int idxCurve = 0, curveCount = pBoneAnim->GetCurveCount(); idxCurve < curveCount; ++idxCurve)
            {
                ResAnimCurve* curve = pBoneAnim->GetCurve(idxCurve);
                Bit32 type = curve->ToData().flag & ResAnimCurve::Mask_Curve;
                if (!foundCurve && type == ResAnimCurve::CurveType_BakedFloat)
                {
                    pBuffer = curve->ToData().pKeyArray.Get();
                    foundCurve = true;
                }
                curve->Reset();
            }
        }
        ToData().flag ^= Flag_CurveBaked;
    }

    return pBuffer;
}

void ResSkeletalAnim::Reset() NN_NOEXCEPT
{
    ToData().pBindSkeleton.Set(NULL);
    uint16_t* pBindIndexArray = ToData().pBindIndexArray.Get();

    for (int idxAnim = 0, animCount = GetBoneAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        pBindIndexArray[idxAnim] = AnimFlag_NotBound;
    }

    ResetCurve();
}

}} // namespace nn::g3d

NN_PRAGMA_POP_WARNINGS
