﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_ResShape.h>
#include <limits>
#include <nn/g3d/g3d_ResFile.h>
#include <nn/gfx/util/gfx_ObjectDebugLabel.h>

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_SHADOW

namespace nn { namespace g3d {

namespace {

// DynamicVertexBufferMask のサイズが適切かチェック
NN_STATIC_ASSERT((std::is_same<uint8_t, decltype(ResVertexData::vertexBufferCount)>::value));

// NOTE: キーシェイプリソースに新規に頂点属性が増えた場合には、変更を加える必要があります。
const int targetAttributeIndexCount =
    ResKeyShape::KeyAttr_PositionCount +
    ResKeyShape::KeyAttr_NormalCount +
    ResKeyShape::KeyAttr_TangentCount +
    ResKeyShape::KeyAttr_BinormalCount +
    ResKeyShape::KeyAttr_ColorCount;
const int attributeIndexPadding = 2;
NN_STATIC_ASSERT((targetAttributeIndexCount + attributeIndexPadding) == NN_ARRAY_SIZE(ResKeyShapeData::targetAttribIndices));

}

//--------------------------------------------------------------------------------------------------

void ResVertex::Setup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    Setup(pDevice, this->pMemoryPool.Get(), 0);
}

void ResVertex::Setup(nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset) NN_NOEXCEPT
{
    // 頂点バッファーの初期化
    int bufferCount = GetVertexBufferCount();
    ptrdiff_t offset = this->memoryPoolOffset + memoryPoolOffset;
    for (int idxBuffer = 0; idxBuffer < bufferCount; ++idxBuffer)
    {
        nn::gfx::Buffer** pBufferArray = this->pVertexBufferPtrArray.Get();
        pBufferArray[idxBuffer] = &this->pVertexBufferArray.Get()[idxBuffer];
        nn::gfx::Buffer::InfoType* pBufferInfo = GetVertexBufferInfo(idxBuffer);
        nn::gfx::Buffer*     pBuffer     = new (GetVertexBuffer(idxBuffer)) nn::gfx::Buffer;

        pBufferInfo->SetGpuAccessFlags(nn::gfx::GpuAccess_VertexBuffer);
        pBuffer->Initialize(pDevice, *pBufferInfo, pMemoryPool, offset, pBufferInfo->GetSize());
        nn::gfx::util::SetBufferDebugLabel(pBuffer, "g3d_vertex");
        offset += nn::util::align_up(pBufferInfo->GetSize(), Alignment_VertexBuffer);
    }
}

void ResVertex::Cleanup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    int bufferCount = GetVertexBufferCount();
    for (int idxBuffer = 0; idxBuffer < bufferCount; ++idxBuffer)
    {
        nn::gfx::Buffer** pBufferArray = this->pVertexBufferPtrArray.Get();
        pBufferArray[idxBuffer] = &this->pVertexBufferArray.Get()[idxBuffer];
        nn::gfx::Buffer* pBuffer = GetVertexBuffer(idxBuffer);
        if (nn::gfx::IsInitialized(*pBuffer) == true)
        {
            pBuffer->Finalize(pDevice);
            pBuffer->nn::gfx::Buffer::~TBuffer();
        }
    }
}

void ResVertex::Reset() NN_NOEXCEPT
{
    Reset(ResetGuardFlag_None);
}

void ResVertex::Reset(Bit32 resetGuardFlag) NN_NOEXCEPT
{
    ResVertexData& resVertexData = this->ToData();
    int bufferCount = this->GetVertexBufferCount();
    for (int idxBuffer = 0; idxBuffer < bufferCount; ++idxBuffer)
    {
        nn::gfx::Buffer** pBufferArray = resVertexData.pVertexBufferPtrArray.Get();
        pBufferArray[idxBuffer] = &resVertexData.pVertexBufferArray.Get()[idxBuffer];
    }

    if ((resetGuardFlag & ResetGuardFlag_UserPtr) != ResetGuardFlag_UserPtr)
    {
        ToData().pUserPtr.Set(NULL);
    }

    if ((resetGuardFlag & ResetGuardFlag_DynamicVertexBuffer) != ResetGuardFlag_DynamicVertexBuffer)
    {
        for (int vertexAttrIndex = 0, vertexAttrCount = GetVertexAttrCount(); vertexAttrIndex < vertexAttrCount; ++vertexAttrIndex)
        {
            ResVertexAttr* pResVertexAttr = GetVertexAttr(vertexAttrIndex);
            pResVertexAttr->SetDynamicVertexAttrDisabled();
        }
    }
}

void ResVertex::CalculateDynamicVertexBufferIndex(DynamicVertexBufferMask* pOutDynamicVertexBufferMask) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDynamicVertexBufferMask);

    pOutDynamicVertexBufferMask->Reset();
    for (int vertexAttrIndex = 0, vertexAttrCount = GetVertexAttrCount(); vertexAttrIndex < vertexAttrCount; ++vertexAttrIndex)
    {
        const ResVertexAttr* pResVertexAttr = GetVertexAttr(vertexAttrIndex);
        if (!pResVertexAttr->IsDynamicVertexAttrEnabled())
        {
            continue;
        }
        pOutDynamicVertexBufferMask->Set(pResVertexAttr->GetBufferIndex());
    }
}

//--------------------------------------------------------------------------------------------------

void ResShape::Setup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    for (int idxMesh = 0, meshCount = GetMeshCount(); idxMesh < meshCount; ++idxMesh)
    {
        GetMesh(idxMesh)->Setup(pDevice);
    }
}

void ResShape::Setup(nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset) NN_NOEXCEPT
{
    for (int idxMesh = 0, meshCount = GetMeshCount(); idxMesh < meshCount; ++idxMesh)
    {
        GetMesh(idxMesh)->Setup(pDevice, pMemoryPool, memoryPoolOffset);
    }
}

void ResShape::Cleanup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    for (int idxMesh = 0, meshCount = GetMeshCount(); idxMesh < meshCount; ++idxMesh)
    {
        GetMesh(idxMesh)->Cleanup(pDevice);
    }
}

void ResShape::Reset() NN_NOEXCEPT
{
    Reset(ResetGuardFlag_None);
}

void ResShape::Reset(Bit32 resetGuardFlag) NN_NOEXCEPT
{
    for( int meshIndex = 0; meshIndex < this->GetMeshCount(); ++meshIndex )
    {
        ResMesh* pResMesh = this->GetMesh( meshIndex );
        ResMeshData& resMeshData = pResMesh->ToData();
        void* pGfxBuffer = &resMeshData.pSubMeshArray.Get()[pResMesh->GetSubMeshCount()];    // バイナリーのメモリレイアウトより
        resMeshData.pIndexBuffer.Set( reinterpret_cast<nn::gfx::Buffer*>( pGfxBuffer ) );
    }

    if ((resetGuardFlag & ResetGuardFlag_UserPtr) != ResetGuardFlag_UserPtr)
    {
        ToData().pUserPtr.Set(NULL);
    }
}

void ResShape::ActivateDynamicVertexAttrForShapeAnim() NN_NOEXCEPT
{
    if (GetKeyShapeCount() < 1)
    {
        return;
    }

    ResVertex* pResVertex = GetVertex();

    // ベース形状のキーシェイプを取得
    const ResKeyShape* pBaseKeyShape = GetKeyShape(0);
    for (int targetAttributeIndex = 0; targetAttributeIndex < targetAttributeIndexCount; ++targetAttributeIndex)
    {
        int attributeIndex = pBaseKeyShape->ToData().targetAttribIndices[targetAttributeIndex] - 1;
        if (attributeIndex == ResKeyShape::InvalidIndex)
        {
            continue;
        }

        ResVertexAttr* pResVertexAttribute = pResVertex->GetVertexAttr(attributeIndex);
        pResVertexAttribute->SetDynamicVertexAttrEnabled();
    }
}

void ResMesh::Setup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    Setup(pDevice, this->pMemoryPool.Get(), 0);
}

void ResMesh::Setup(nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset) NN_NOEXCEPT
{
    // インデックスバッファー作成
    nn::gfx::Buffer::InfoType* pBufferInfo = GetIndexBufferInfo();
    nn::gfx::Buffer*           pBuffer     = new (GetIndexBuffer()) nn::gfx::Buffer;

    pBufferInfo->SetGpuAccessFlags(nn::gfx::GpuAccess_IndexBuffer);
    pBuffer->Initialize(pDevice, *pBufferInfo, pMemoryPool, this->memoryPoolOffset + memoryPoolOffset, pBufferInfo->GetSize());
    nn::gfx::util::SetBufferDebugLabel(pBuffer, "g3d_index");
}

void ResMesh::Cleanup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    nn::gfx::Buffer* pBuffer = GetIndexBuffer();
    if (nn::gfx::IsInitialized(*pBuffer) == true)
    {
        pBuffer->Finalize(pDevice);
        pBuffer->nn::gfx::Buffer::~TBuffer();
    }
}

void ResMesh::DrawSubMesh(nn::gfx::CommandBuffer* pCommandBuffer, int submeshIndex, int submeshCount, int instances) const NN_NOEXCEPT
{
    DrawSubMesh(pCommandBuffer, submeshIndex, submeshCount, instances, 0);
}

void ResMesh::DrawSubMesh(nn::gfx::CommandBuffer* pCommandBuffer, int submeshIndex, int submeshCount, int instances, int baseInstance) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(submeshCount > 0);
    NN_SDK_REQUIRES(instances > 0);
    NN_SDK_REQUIRES(baseInstance >= 0);

    int count = GetIndexCount(submeshIndex, submeshCount);
    const ResSubMesh* pFirstSubMesh = GetSubMesh(submeshIndex);
    ptrdiff_t offset = pFirstSubMesh->GetOffset();

    nn::gfx::GpuAddress gpuAddress;
    GetIndexBuffer()->GetGpuAddress(&gpuAddress);
    gpuAddress.Offset(offset);
    pCommandBuffer->DrawIndexed(GetPrimitiveType(),
        GetIndexFormat(),
        gpuAddress,
        count, ToData().offset, instances, baseInstance);
}

}} // namespace nn::g3d

NN_PRAGMA_POP_WARNINGS
