﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_ResSceneAnim.h>

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_SHADOW

namespace nn { namespace g3d {

void ResCameraAnim::Initialize(CameraAnimResult* pResult) const NN_NOEXCEPT
{
    *pResult = *ToData().pBaseValueArray.Get();
}

void ResCameraAnim::Evaluate(CameraAnimResult* pResult, float frame) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pResult != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ToData().targetOffset);
        *pTarget = pCurve->EvaluateFloat(frame);
    }
}

void ResCameraAnim::Evaluate(CameraAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pResult != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ToData().targetOffset);
        *pTarget = pCurve->EvaluateFloat(frame, &pFrameCache[idxCurve]);
    }
}

bool ResCameraAnim::BakeCurve(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    if (bufferSize == 0)
    {
        return true;
    }
    if (pBuffer == NULL ||
        bufferSize < GetBakedSize())
    {
        return false;
    }
    for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        ResAnimCurve* curve = GetCurve(idxCurve);
        size_t size = curve->CalculateBakedFloatSize();
        curve->BakeFloat(pBuffer, size);
        pBuffer = AddOffset(pBuffer, size);
    }
    ToData().flag |= Flag_CurveBaked;

    return true;
}

void* ResCameraAnim::ResetCurve() NN_NOEXCEPT
{
    void* pBuffer = NULL;
    if (IsCurveBaked())
    {
        bool foundCurve = false;
        for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
        {
            ResAnimCurve* curve = GetCurve(idxCurve);
            Bit32 type = curve->ToData().flag & ResAnimCurve::Mask_Curve;
            if (!foundCurve && type == ResAnimCurve::CurveType_BakedFloat)
            {
                pBuffer = curve->ToData().pKeyArray.Get();
                foundCurve = true;
            }
            curve->Reset();
        }
        ToData().flag ^= Flag_CurveBaked;
    }

    return pBuffer;
}

void ResCameraAnim::Reset() NN_NOEXCEPT
{
    ResetCurve();
}

//--------------------------------------------------------------------------------------------------


void ResLightAnim::Initialize(LightAnimResult* pResult) const NN_NOEXCEPT
{
    static const int tblDimension[] = { 1, 3, 3, 2, 2, 3, 3 };
    static const int NUM_PARAM = sizeof(tblDimension) / sizeof(int);

    const Bit32* pBaseValue = ToData().pBaseValueArray.Get();
    Bit32* pTarget = reinterpret_cast<Bit32*>(pResult);
    for (int idxParam = 0; idxParam < NUM_PARAM; ++idxParam)
    {
        int dim = tblDimension[idxParam];
        if (ToData().flag & (Flag_BaseEnable << idxParam)) // BaseValue を持っている部分のみ埋める。
        {
            memcpy(pTarget, pBaseValue, sizeof(Bit32) * dim);
            pBaseValue += dim;
        }
        pTarget += dim;
    }
}

BindResult ResLightAnim::Bind(const BindFuncTable& funcTable) NN_NOEXCEPT
{
    BindResult result;

    if (-1 == GetLightTypeIndex())
    {
        const BindFuncTable::Name* pNameArray = funcTable.GetLightTypeArray();
        for (int idxLightType = 0, numLightType = funcTable.GetLightTypeCount();
            idxLightType < numLightType; ++idxLightType)
        {
            const BindFuncTable::Name& name = pNameArray[idxLightType];
            if (std::strncmp(ToData().pLightType.Get()->GetData(), name.str, name.len))
            {
                ToData().lightTypeIndex = static_cast<int8_t>(idxLightType);
                break;
            }
        }
        if (-1 == GetLightTypeIndex())
        {
            result.SetFailureBit();
        }
        else
        {
            result.SetSuccessBit();
        }
    }

    if (-1 == GetDistAttnFuncIndex())
    {
        const BindFuncTable::Name* pNameArray = funcTable.GetLightDistAttnArray();
        for (int idxDistAttn = 0, numDistAttn = funcTable.GetLightDistAttnCount();
            idxDistAttn < numDistAttn; ++idxDistAttn)
        {
            const BindFuncTable::Name& name = pNameArray[idxDistAttn];
            if (std::strncmp(ToData().pDistAttnFunc.Get()->GetData(), name.str, name.len))
            {
                ToData().distAttnFuncIndex = static_cast<int8_t>(idxDistAttn);
                break;
            }
        }
        if (-1 == GetDistAttnFuncIndex())
        {
            result.SetFailureBit();
        }
        else
        {
            result.SetSuccessBit();
        }
    }

    if (-1 == GetAngleAttnFuncIndex())
    {
        const BindFuncTable::Name* pNameArray = funcTable.GetLightAngleAttnArray();
        for (int idxAngleAttn = 0, numAngleAttn = funcTable.GetLightAngleAttnCount();
            idxAngleAttn < numAngleAttn; ++idxAngleAttn)
        {
            const BindFuncTable::Name& name = pNameArray[idxAngleAttn];
            if (std::strncmp(ToData().pAngleAttnFunc.Get()->GetData(), name.str, name.len))
            {
                ToData().angleAttnFuncIndex = static_cast<int8_t>(idxAngleAttn);
                break;
            }
        }
        if (-1 == GetAngleAttnFuncIndex())
        {
            result.SetFailureBit();
        }
        else
        {
            result.SetSuccessBit();
        }
    }

    return result;
}

void ResLightAnim::Release() NN_NOEXCEPT
{
    ToData().lightTypeIndex = -1;
    ToData().distAttnFuncIndex = -1;
    ToData().angleAttnFuncIndex = -1;
}

void ResLightAnim::Evaluate(LightAnimResult* pResult, float frame) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pResult != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    int idxCurve = 0;
    if (ToData().flag & Flag_CurveEnable)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve++);
        pResult->enable = pCurve->EvaluateInt(frame);
    }

    for (int numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ToData().targetOffset);
        *pTarget = pCurve->EvaluateFloat(frame);
    }
}

void ResLightAnim::Evaluate(LightAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pResult != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    int idxCurve = 0;
    if (ToData().flag & Flag_CurveEnable)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve++);
        pResult->enable = pCurve->EvaluateInt(frame, pFrameCache);
    }

    for (int numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ToData().targetOffset);
        *pTarget = pCurve->EvaluateFloat(frame, &pFrameCache[idxCurve]);
    }
}

bool ResLightAnim::BakeCurve(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    if (bufferSize == 0)
    {
        return true;
    }
    if (pBuffer == NULL ||
        bufferSize < GetBakedSize())
    {
        return false;
    }
    for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        ResAnimCurve* curve = GetCurve(idxCurve);
        size_t size = curve->CalculateBakedFloatSize();
        curve->BakeFloat(pBuffer, size);
        pBuffer = AddOffset(pBuffer, size);
    }
    ToData().flag |= Flag_CurveBaked;

    return true;
}

void* ResLightAnim::ResetCurve() NN_NOEXCEPT
{
    void* pBuffer = NULL;
    if (IsCurveBaked())
    {
        bool foundCurve = false;
        for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
        {
            ResAnimCurve* curve = GetCurve(idxCurve);
            Bit32 type = curve->ToData().flag & ResAnimCurve::Mask_Curve;
            if (!foundCurve && (type == ResAnimCurve::CurveType_BakedFloat
                || type == ResAnimCurve::CurveType_BakedBool))
            {
                pBuffer = curve->ToData().pKeyArray.Get();
                foundCurve = true;
            }
            curve->Reset();
        }
        ToData().flag ^= Flag_CurveBaked;
    }

    return pBuffer;
}

void ResLightAnim::Reset() NN_NOEXCEPT
{
    ResetCurve();
}

//--------------------------------------------------------------------------------------------------


void ResFogAnim::Initialize(FogAnimResult* pResult) const NN_NOEXCEPT
{
    *pResult = *ToData().pBaseValueArray.Get();
}

BindResult ResFogAnim::Bind(const BindFuncTable& funcTable) NN_NOEXCEPT
{
    BindResult result;

    if (-1 == GetDistAttnFuncIndex())
    {
        const BindFuncTable::Name* pNameArray = funcTable.GetFogDistAttnArray();
        for (int idxDistAttn = 0, distAttnCount = funcTable.GetFogDistAttnCount();
            idxDistAttn < distAttnCount; ++idxDistAttn)
        {
            const BindFuncTable::Name& name = pNameArray[idxDistAttn];
            if (std::strncmp(ToData().pDistAttnFunc.Get()->GetData(), name.str, name.len))
            {
                ToData().distAttnFuncIndex = static_cast<int8_t>(idxDistAttn);
                break;
            }
        }

        if (-1 == GetDistAttnFuncIndex())
        {
            result.SetFailureBit();
        }
        else
        {
            result.SetSuccessBit();
        }
    }

    return result;
}

void ResFogAnim::Release() NN_NOEXCEPT
{
    ToData().distAttnFuncIndex = -1;
}

void ResFogAnim::Evaluate(FogAnimResult* pResult, float frame) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pResult != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    for (int idxCurve = 0, curveCount = GetCurveCount(); idxCurve < curveCount; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ToData().targetOffset);
        *pTarget = pCurve->EvaluateFloat(frame);
    }
}

void ResFogAnim::Evaluate(FogAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pResult != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    for (int idxCurve = 0, curveCount = GetCurveCount(); idxCurve < curveCount; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ToData().targetOffset);
        *pTarget = pCurve->EvaluateFloat(frame, &pFrameCache[idxCurve]);
    }
}

bool ResFogAnim::BakeCurve(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    if (bufferSize == 0)
    {
        return true;
    }
    if (pBuffer == NULL ||
        bufferSize < GetBakedSize())
    {
        return false;
    }
    for (int idxCurve = 0, curveCount = GetCurveCount(); idxCurve < curveCount; ++idxCurve)
    {
        ResAnimCurve* curve = GetCurve(idxCurve);
        size_t size = curve->CalculateBakedFloatSize();
        curve->BakeFloat(pBuffer, size);
        pBuffer = AddOffset(pBuffer, size);
    }
    ToData().flag |= Flag_CurveBaked;

    return true;
}

void* ResFogAnim::ResetCurve() NN_NOEXCEPT
{
    void* pBuffer = NULL;
    if (IsCurveBaked())
    {
        bool foundCurve = false;
        for (int idxCurve = 0, curveCount = GetCurveCount(); idxCurve < curveCount; ++idxCurve)
        {
            ResAnimCurve* curve = GetCurve(idxCurve);
            Bit32 type = curve->ToData().flag & ResAnimCurve::Mask_Curve;
            if (!foundCurve && type == ResAnimCurve::CurveType_BakedFloat)
            {
                pBuffer = curve->ToData().pKeyArray.Get();
                foundCurve = true;
            }
            curve->Reset();
        }
        ToData().flag ^= Flag_CurveBaked;
    }

    return pBuffer;
}

void ResFogAnim::Reset() NN_NOEXCEPT
{
    ResetCurve();
}

//--------------------------------------------------------------------------------------------------


BindResult ResSceneAnim::Bind(const BindFuncTable& funcTable) NN_NOEXCEPT
{
    BindResult result;

    for (int idxLightAnim = 0, lightAnimCount = GetLightAnimCount();
        idxLightAnim < lightAnimCount; ++idxLightAnim)
    {
        ResLightAnim* pLightAnim = GetLightAnim(idxLightAnim);
        result |= pLightAnim->Bind(funcTable);
    }

    for (int idxFogAnim = 0, fogAnimCount = GetFogAnimCount();
        idxFogAnim < fogAnimCount; ++idxFogAnim)
    {
        ResFogAnim* pFogAnim = GetFogAnim(idxFogAnim);
        result |= pFogAnim->Bind(funcTable);
    }

    return result;
}

void ResSceneAnim::Release() NN_NOEXCEPT
{
    for (int idxLightAnim = 0, lightAnimCount = GetLightAnimCount();
        idxLightAnim < lightAnimCount; ++idxLightAnim)
    {
        ResLightAnim* pLightAnim = GetLightAnim(idxLightAnim);
        pLightAnim->Release();
    }

    for (int idxFogAnim = 0, fogAnimCount = GetFogAnimCount();
        idxFogAnim < fogAnimCount; ++idxFogAnim)
    {
        ResFogAnim* pFogAnim = GetFogAnim(idxFogAnim);
        pFogAnim->Release();
    }
}

void ResSceneAnim::Reset() NN_NOEXCEPT
{
    for (int idxCameraAnim = 0, cameraAnimCount = GetCameraAnimCount();
        idxCameraAnim < cameraAnimCount; ++idxCameraAnim)
    {
        ResCameraAnim* pCameraAnim = GetCameraAnim(idxCameraAnim);
        pCameraAnim->Reset();
    }

    for (int idxLightAnim = 0, lightAnimCount = GetLightAnimCount();
        idxLightAnim < lightAnimCount; ++idxLightAnim)
    {
        ResLightAnim* pLightAnim = GetLightAnim(idxLightAnim);
        pLightAnim->Reset();
    }

    for (int idxFogAnim = 0, fogAnimCount = GetFogAnimCount();
        idxFogAnim < fogAnimCount; ++idxFogAnim)
    {
        ResFogAnim* pFogAnim = GetFogAnim(idxFogAnim);
        pFogAnim->Reset();
    }
}

}} // namespace nn::g3d

NN_PRAGMA_POP_WARNINGS
