﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_ResModel.h>
#include <nn/g3d/g3d_ResFile.h>

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_SHADOW

namespace nn { namespace g3d {

BindResult ResModel::BindTexture(TextureBindCallback pCallback, void* pUserData) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pCallback != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;
    for (int idxMat = 0, materalCount = GetMaterialCount(); idxMat < materalCount; ++idxMat)
    {
        ResMaterial* pMaterial = GetMaterial(idxMat);
        result |= pMaterial->BindTexture(pCallback, pUserData);
    }

    return result;
}

bool ResModel::ForceBindTexture(const TextureRef& textureRef, const char* name) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(name != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    bool result = false;
    for (int idxMat = 0, materalCount = GetMaterialCount(); idxMat < materalCount; ++idxMat)
    {
        ResMaterial* pMaterial = GetMaterial(idxMat);
        if (pMaterial->ForceBindTexture(textureRef, name) == true)
        {
            result = true;
        }
    }
    return result;
}

void ResModel::ReleaseTexture() NN_NOEXCEPT
{
    for (int idxMat = 0, materalCount = GetMaterialCount(); idxMat < materalCount; ++idxMat)
    {
        ResMaterial* pMaterial = GetMaterial(idxMat);
        pMaterial->ReleaseTexture();
    }
}

void ResModel::ActivateDynamicVertexAttrForShapeAnim() NN_NOEXCEPT
{
    for (int shapeIndex = 0, shapeCount = GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        this->GetShape(shapeIndex)->ActivateDynamicVertexAttrForShapeAnim();
    }
}

void ResModel::Setup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    int materalCount = this->GetMaterialCount();
    for (int materialIndex = 0; materialIndex < materalCount; ++materialIndex)
    {
        this->GetMaterial(materialIndex)->Setup(pDevice);
    }

    int shapeCount = this->GetShapeCount();
    for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
    {
        this->GetShape(shapeIndex)->Setup(pDevice);
    }

    int vertexCount = this->GetVertexCount();
    for (int vertexIndex = 0; vertexIndex < vertexCount; ++vertexIndex)
    {
        this->GetVertex(vertexIndex)->Setup(pDevice);
    }
}

void ResModel::Setup(nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset) NN_NOEXCEPT
{
    int materalCount = this->GetMaterialCount();
    for (int materialIndex = 0; materialIndex < materalCount; ++materialIndex)
    {
        this->GetMaterial(materialIndex)->Setup(pDevice);
    }

    int shapeCount = this->GetShapeCount();
    for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
    {
        this->GetShape(shapeIndex)->Setup(pDevice, pMemoryPool, memoryPoolOffset);
    }

    int vertexCount = this->GetVertexCount();
    for (int vertexIndex = 0; vertexIndex < vertexCount; ++vertexIndex)
    {
        this->GetVertex(vertexIndex)->Setup(pDevice, pMemoryPool, memoryPoolOffset);
    }
}

void ResModel::Cleanup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    int materalCount = this->GetMaterialCount();
    for (int materialIndex = 0; materialIndex < materalCount; ++materialIndex)
    {
        this->GetMaterial(materialIndex)->Cleanup(pDevice);
    }

    int shapeCount = this->GetShapeCount();
    for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
    {
        this->GetShape(shapeIndex)->Cleanup(pDevice);
    }

    int vertexCount = this->GetVertexCount();
    for (int vertexIndex = 0; vertexIndex < vertexCount; ++vertexIndex)
    {
        this->GetVertex(vertexIndex)->Cleanup(pDevice);
    }
}

void ResModel::Reset() NN_NOEXCEPT
{
    Reset(ResetGuardFlag_None);
}

void ResModel::Reset(Bit32 resetGuardFlag) NN_NOEXCEPT
{
    int materalCount = this->GetMaterialCount();
    for (int materialIndex = 0; materialIndex < materalCount; ++materialIndex)
    {
        this->GetMaterial(materialIndex)->Reset(resetGuardFlag);
    }

    int shapeCount = this->GetShapeCount();
    for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
    {
        this->GetShape(shapeIndex)->Reset(resetGuardFlag);
    }

    int vertexCount = this->GetVertexCount();
    for (int vertexIndex = 0; vertexIndex < vertexCount; ++vertexIndex)
    {
        this->GetVertex(vertexIndex)->Reset(resetGuardFlag);
    }

    this->GetSkeleton()->Reset(resetGuardFlag);

    if ((resetGuardFlag & ResetGuardFlag_UserPtr) != ResetGuardFlag_UserPtr)
    {
        ToData().pUserPtr.Set(NULL);
    }
}

}} // namespace nn::g3d

NN_PRAGMA_POP_WARNINGS
