﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_ResMaterialAnim.h>
#include <algorithm>

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_SHADOW

namespace nn { namespace g3d {

template <bool useFrameCache>
void ResPerMaterialAnim::EvaluateShaderParamAnim(
     void* pResult, float frame, const uint16_t* pSubBindIndex, AnimFrameCache* pFrameCache) const NN_NOEXCEPT
{
    const ResShaderParamAnimInfo* pShaderParamAnimInfo = ToData().pShaderParamAnimInfoArray.Get();
    const ResAnimCurve* const pCurves = ToData().pCurveArray.Get();

    for (int idxParamAnim = 0, numParamAnim = GetShaderParamAnimCount();
        idxParamAnim < numParamAnim; ++idxParamAnim, ++pShaderParamAnimInfo)
    {
        int idxCurve   = pShaderParamAnimInfo->beginCurve;
        int intCount   = pShaderParamAnimInfo->intCurveCount;
        int floatCount = pShaderParamAnimInfo->floatCurveCount;
        if (pSubBindIndex[idxParamAnim] == AnimFlag_NotBound)
        {
            continue;
        }

        // texsrt の mode をカーブすると、int カーブと float カーブが混ざる
        // 以下の処理では、int カーブが先に詰まっているとしている
        const ResAnimCurve* pCurve = pCurves + idxCurve;
        int* pIntResult = AddOffset<int>(pResult, sizeof(int) * idxCurve);
        idxCurve  += intCount;
        for (int idxInt = 0; idxInt < intCount; ++idxInt, ++pIntResult, ++pCurve, ++pFrameCache)
        {
            const ResAnimCurve* pIntCurve = pCurve;
            if (NN_STATIC_CONDITION(useFrameCache))
            {
                *pIntResult = pIntCurve->EvaluateInt(frame, pFrameCache);
            }
            else
            {
                *pIntResult = pIntCurve->EvaluateInt(frame);
            }
        }

        float* pFloatResult = AddOffset<float>(pResult, sizeof(float) * idxCurve);
        for (int idxFloat = 0; idxFloat < floatCount;
            ++idxFloat, ++pFloatResult, ++pCurve, ++pFrameCache)
        {
            const ResAnimCurve* pFloatCurve = pCurve;
            if (NN_STATIC_CONDITION(useFrameCache))
            {
                *pFloatResult = pFloatCurve->EvaluateFloat(frame, pFrameCache);
            }
            else
            {
                *pFloatResult = pFloatCurve->EvaluateFloat(frame);
            }
        }
    }

    return;
}

template void ResPerMaterialAnim::EvaluateShaderParamAnim<true>(void*, float, const uint16_t*, AnimFrameCache*) const;
template void ResPerMaterialAnim::EvaluateShaderParamAnim<false>(void*, float, const uint16_t*, AnimFrameCache*) const;

template <bool useFrameCache>
void ResPerMaterialAnim::EvaluateTexturePatternAnim(
     int32_t* pResult, float frame, const uint16_t* pSubBindIndex, AnimFrameCache* pFrameCache) const NN_NOEXCEPT
{
    const ResTexturePatternAnimInfo* pTexturePatternAnimInfo = ToData().pTexturePatternAnimInfoArray.Get();

    int resultIndex = 0;
    for (int idxPatAnim = 0, patternAnimCount = GetTexturePatternAnimCount();
        idxPatAnim < patternAnimCount; ++idxPatAnim, ++pTexturePatternAnimInfo)
    {
        if (pSubBindIndex[idxPatAnim] == AnimFlag_NotBound ||
            pTexturePatternAnimInfo->curveIndex == AnimFlag_NotBound)
        {
            continue;
        }

        const ResAnimCurve* pCurve = GetCurve(pTexturePatternAnimInfo->curveIndex);
        if (NN_STATIC_CONDITION(useFrameCache))
        {
            pResult[resultIndex] = pCurve->EvaluateInt(frame, &pFrameCache[resultIndex]);
        }
        else
        {
            pResult[resultIndex] = pCurve->EvaluateInt(frame);
        }
        ++resultIndex;
    }

    return;
}

template void ResPerMaterialAnim::EvaluateTexturePatternAnim<true>(int32_t*, float, const uint16_t*, AnimFrameCache*) const;
template void ResPerMaterialAnim::EvaluateTexturePatternAnim<false>(int32_t*, float, const uint16_t*, AnimFrameCache*) const;

template <bool useFrameCache>
void ResPerMaterialAnim::EvaluateVisibilityAnim(
     int32_t* pResult, float frame, AnimFrameCache* pFrameCache) const NN_NOEXCEPT
{
    // カーブがなければ、なにもしない
    if (visibilityCurveIndex == AnimFlag_NotBound)
    {
        return;
    }

    const ResAnimCurve* pCurve = GetCurve(visibilityCurveIndex);
    if (NN_STATIC_CONDITION(useFrameCache))
    {
        *pResult = pCurve->EvaluateInt(frame, pFrameCache);
    }
    else
    {
        *pResult = pCurve->EvaluateInt(frame);
    }

    return;
}

template void ResPerMaterialAnim::EvaluateVisibilityAnim<true>(int32_t*, float, AnimFrameCache*) const;
template void ResPerMaterialAnim::EvaluateVisibilityAnim<false>(int32_t*, float, AnimFrameCache*) const;

BindResult ResPerMaterialAnim::PreBind(const ResMaterial* pTarget) NN_NOEXCEPT
{
    BindResult result;
    // シェーダーパラメータアニメーションの処理
    ResShaderParamAnimInfo* pShaderParamAnimInfo = ToData().pShaderParamAnimInfoArray.Get();
    for (int idxParamAnim = 0, paramAnimCount = GetShaderParamAnimCount();
        idxParamAnim < paramAnimCount; ++idxParamAnim, ++pShaderParamAnimInfo)
    {
        const char* pName = pShaderParamAnimInfo->pName.Get()->GetData();
        int idxTarget = pTarget->FindShaderParamIndex(pName);
        pShaderParamAnimInfo->subbindIndex = static_cast<uint16_t>(idxTarget); // -1 -> 0xFFFF
        if (idxTarget == nn::util::ResDic::Npos)
        {
            result.SetFailureBit();
        }
        else
        {
            result.SetSuccessBit();
        }
    }

    // テクスチャーパターンアニメーションの処理
    ResTexturePatternAnimInfo* pTexturePatternAnimInfo = ToData().pTexturePatternAnimInfoArray.Get();
    for (int idxPatAnim = 0, patternAnimCount = GetTexturePatternAnimCount();
        idxPatAnim < patternAnimCount; ++idxPatAnim, ++pTexturePatternAnimInfo)
    {
        const char* pName = pTexturePatternAnimInfo->pName.Get()->GetData();
        int idxTarget = pTarget->FindSamplerIndex(pName);
        pTexturePatternAnimInfo->subbindIndex = static_cast<int8_t>(idxTarget);
        if (idxTarget == nn::util::ResDic::Npos)
        {
            result.SetFailureBit();
        }
        else
        {
            result.SetSuccessBit();
        }
    }

    return result;
}

BindResult ResPerMaterialAnim::BindCheck(const ResMaterial* pTarget) const NN_NOEXCEPT
{
    BindResult result;
    // シェーダーパラメータアニメーションの処理
    const ResShaderParamAnimInfo* pShaderParamAnimInfo = ToData().pShaderParamAnimInfoArray.Get();
    for (int idxParamAnim = 0, paramAnimCount = GetShaderParamAnimCount();
        idxParamAnim < paramAnimCount; ++idxParamAnim, ++pShaderParamAnimInfo)
    {
        const char* pName = pShaderParamAnimInfo->pName.Get()->GetData();
        int idxTarget = pTarget->FindShaderParamIndex(pName);
        if (idxTarget == nn::util::ResDic::Npos)
        {
            result.SetFailureBit();
        }
        else
        {
            result.SetSuccessBit();
        }
    }

    // テクスチャーパターンアニメーションの処理
    const ResTexturePatternAnimInfo* pTexturePatternAnimInfo = ToData().pTexturePatternAnimInfoArray.Get();
    for (int idxPatAnim = 0, patternAnimCount = GetTexturePatternAnimCount();
        idxPatAnim < patternAnimCount; ++idxPatAnim, ++pTexturePatternAnimInfo)
    {
        const char* pName = pTexturePatternAnimInfo->pName.Get()->GetData();
        int idxTarget = pTarget->FindSamplerIndex(pName);
        if (idxTarget == nn::util::ResDic::Npos)
        {
            result.SetFailureBit();
        }
        else
        {
            result.SetSuccessBit();
        }
    }

    return result;
}

//--------------------------------------------------------------------------------------------------

BindResult ResMaterialAnim::PreBind(const ResModel* pModel) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModel != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    ToData().pBindModel.Set(const_cast<ResModel*>(pModel));
    uint16_t* pBindIndexArray = ToData().pBindIndexArray.Get();

    BindResult result;
    for (int idxMatAnim = 0, materialAnimCount = GetPerMaterialAnimCount(); idxMatAnim < materialAnimCount; ++idxMatAnim)
    {
        ResPerMaterialAnim* pMatAnim = GetPerMaterialAnim(idxMatAnim);

        const char* pName = pMatAnim->ToData().pName.Get()->GetData();
        const ResMaterial* pMaterial = pModel->FindMaterial(pName);
        if (pMaterial != NULL)
        {
            pBindIndexArray[idxMatAnim] = static_cast<uint16_t>(pMaterial->GetIndex());
            // ビジビリティアニメを含んでいるなら、この時点でビジビリティの関連付けは
            // 出来たと判定できるので、成功ビットを立てておく
            if (pMatAnim->IsVisibilityAnimEnabled())
            {
                result.SetSuccessBit();
            }
            BindResult subBindResult = pMatAnim->PreBind(pMaterial);
            result |= subBindResult;
        }
        else
        {
            pBindIndexArray[idxMatAnim] = AnimFlag_NotBound;
            result.SetFailureBit();
        }
    }
    return result;
}

BindResult ResMaterialAnim::BindCheck(const ResModel* pModel) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModel != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;
    for (int idxMatAnim = 0, materialAnimCount = GetPerMaterialAnimCount(); idxMatAnim < materialAnimCount; ++idxMatAnim)
    {
        const ResPerMaterialAnim* pMatAnim = GetPerMaterialAnim(idxMatAnim);

        const char* pName = pMatAnim->ToData().pName.Get()->GetData();
        const ResMaterial* pMaterial = pModel->FindMaterial(pName);
        if (pMaterial != NULL)
        {
            // ビジビリティアニメを含んでいるなら、この時点でビジビリティの関連付けは
            // 出来たと判定できるので、成功ビットを立てておく
            if (pMatAnim->IsVisibilityAnimEnabled())
            {
                result.SetSuccessBit();
            }
            BindResult subBindResult = pMatAnim->BindCheck(pMaterial);
            result |= subBindResult;
        }
        else
        {
            result.SetFailureBit();
        }
    }
    return result;
}

BindResult ResMaterialAnim::BindTexture(TextureBindCallback pCallback, void* pUserData) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pCallback != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;

    for (int idxTex = 0, textureCount = GetTextureCount(); idxTex < textureCount; ++idxTex)
    {
        if (!IsTextureBound(idxTex))
        {
            TextureRef textureRef = pCallback(GetTextureName(idxTex), pUserData);
            SetTexture(idxTex, textureRef.GetTextureView());
            SetTextureDescriptorSlot(idxTex, textureRef.GetDescriptorSlot());
            if (textureRef.IsValid())
            {
                result.SetSuccessBit();
            }
            else
            {
                result.SetFailureBit();
            }
        }
    }

    return result;
}

bool ResMaterialAnim::ForceBindTexture(const TextureRef& textureRef, const char* name) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(name != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    bool success = false;
    for (int idxTex = 0, textureCount = GetTextureCount(); idxTex < textureCount; ++idxTex)
    {
        if (0 == strcmp(GetTextureName(idxTex), name))
        {
            SetTexture(idxTex, textureRef.GetTextureView());
            SetTextureDescriptorSlot(idxTex, textureRef.GetDescriptorSlot());
            success = true;
        }
    }
    return success;
}

void ResMaterialAnim::ReleaseTexture() NN_NOEXCEPT
{
    for (int idxTex = 0, textureCount = GetTextureCount(); idxTex < textureCount; ++idxTex)
    {
        SetTexture(idxTex, NULL);
        SetTextureDescriptorSlot(idxTex, nn::gfx::DescriptorSlot());
    }
}

bool ResMaterialAnim::BakeCurve(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    if (bufferSize == 0)
    {
        return true;
    }
    if (pBuffer == NULL ||
        bufferSize < GetBakedSize())
    {
        return false;
    }
    for (int idxMatAnim = 0, materialAnimCount = GetPerMaterialAnimCount(); idxMatAnim < materialAnimCount; ++idxMatAnim)
    {
        ResPerMaterialAnim* pMatAnim = GetPerMaterialAnim(idxMatAnim);
        for (int idxCurve = 0, curveCount = pMatAnim->GetCurveCount();
            idxCurve < curveCount; ++idxCurve)
        {
            ResAnimCurve* curve = pMatAnim->GetCurve(idxCurve);
            size_t size = curve->CalculateBakedSize();
            curve->Bake(pBuffer, size);
            pBuffer = AddOffset(pBuffer, size);
        }
    }
    ToData().flag |= Flag_CurveBaked;

    return true;
}

void* ResMaterialAnim::ResetCurve() NN_NOEXCEPT
{
    void* pBuffer = NULL;
    if (IsCurveBaked())
    {
        bool foundCurve = false;
        for (int idxMatAnim = 0, materialAnimCount = GetPerMaterialAnimCount(); idxMatAnim < materialAnimCount; ++idxMatAnim)
        {
            ResPerMaterialAnim* pMatAnim = GetPerMaterialAnim(idxMatAnim);
            for (int idxCurve = 0, curveCount = pMatAnim->GetCurveCount();
                idxCurve < curveCount; ++idxCurve)
            {
                ResAnimCurve* curve = pMatAnim->GetCurve(idxCurve);
                Bit32 type = curve->ToData().flag & ResAnimCurve::Mask_Curve;
                if (!foundCurve && (type == ResAnimCurve::CurveType_BakedFloat ||
                    type == ResAnimCurve::CurveType_BakedInt || type == ResAnimCurve::CurveType_BakedBool))
                {
                    pBuffer = curve->ToData().pKeyArray.Get();
                    foundCurve = true;
                }
                curve->Reset();
            }
        }
        ToData().flag ^= Flag_CurveBaked;
    }

    return pBuffer;
}

void ResMaterialAnim::Reset() NN_NOEXCEPT
{
    ToData().pBindModel.Set(NULL);
    uint16_t* pBindIndexArray = ToData().pBindIndexArray.Get();
    for (int idxMatAnim = 0, materialAnimCount = GetPerMaterialAnimCount(); idxMatAnim < materialAnimCount; ++idxMatAnim)
    {
        ResPerMaterialAnim* pMatAnim = GetPerMaterialAnim(idxMatAnim);

        // シェーダーパラメータの処理
        ResShaderParamAnimInfo* pShaderParamInfo = pMatAnim->ToData().pShaderParamAnimInfoArray.Get();
        for (int idxParamAnim = 0, paramAnimCount = pMatAnim->GetShaderParamAnimCount();
            idxParamAnim < paramAnimCount; ++idxParamAnim, ++pShaderParamInfo)
        {
            pShaderParamInfo->subbindIndex = AnimFlag_NotBound;
        }

        // テクスチャーパターンの処理
        ResTexturePatternAnimInfo* pTexturePatternInfo = pMatAnim->ToData().pTexturePatternAnimInfoArray.Get();
        for (int idxPatAnim = 0, patternAnimCount = pMatAnim->GetTexturePatternAnimCount();
            idxPatAnim < patternAnimCount; ++idxPatAnim, ++pTexturePatternInfo)
        {
            pTexturePatternInfo->subbindIndex = -1;
        }

        pBindIndexArray[idxMatAnim] = AnimFlag_NotBound;
    }
    ResetCurve();
}

}} // namespace nn::g3d

NN_PRAGMA_POP_WARNINGS
