﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_ResMaterial.h>
#include <nn/g3d/g3d_ResFile.h>
#include <nn/gfx/util/gfx_ObjectDebugLabel.h>
#include <limits>
#include <algorithm>

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_SHADOW

namespace nn { namespace g3d {

namespace {

struct TexMtx
{
    float m[3][2]; // column-major
};

enum
{
    NUM_COMP_MTX14 = 4,
    NUM_COMP_MTX24 = 8
};

size_t ConvertTexSrtMaya(void* pDst, const void* pSrc)
{
    const TexSrt& srt = *static_cast<const TexSrt*>(pSrc);
    float sinR, cosR;
    int index = nn::util::RadianToAngleIndex(srt.r);
    nn::util::SinCosTable(&sinR, &cosR, index);
    const float sinPart = 0.5f * sinR - 0.5f;
    const float cosPart = -0.5f * cosR;

    TexMtx& mtx = *static_cast<TexMtx*>(pDst);
    mtx.m[0][0] =  srt.sx * cosR;
    mtx.m[0][1] = -srt.sy * sinR;
    mtx.m[1][0] =  srt.sx * sinR;
    mtx.m[1][1] =  srt.sy * cosR;
    mtx.m[2][0] =  srt.sx * (cosPart - sinPart - srt.tx);
    mtx.m[2][1] =  srt.sy * (cosPart + sinPart + srt.ty) + 1.0f;

    return sizeof(TexMtx);
}

size_t ConvertTexSrt3dsmax(void* pDst, const void* pSrc)
{
    const TexSrt& srt = *static_cast<const TexSrt*>(pSrc);
    float sinR, cosR;
    int index = nn::util::RadianToAngleIndex(srt.r);
    nn::util::SinCosTable(&sinR, &cosR, index);
    const float sxcr = srt.sx * cosR;
    const float sxsr = srt.sx * sinR;
    const float sycr = srt.sy * cosR;
    const float sysr = srt.sy * sinR;

    TexMtx& mtx = *static_cast<TexMtx*>(pDst);
    mtx.m[0][0] =  sxcr;
    mtx.m[0][1] = -sysr;
    mtx.m[1][0] =  sxsr;
    mtx.m[1][1] =  sycr;
    mtx.m[2][0] = -sxcr * (srt.tx + 0.5f) + sxsr * (srt.ty - 0.5f) + 0.5f;
    mtx.m[2][1] =  sysr * (srt.tx + 0.5f) + sycr * (srt.ty - 0.5f) + 0.5f;

    return sizeof(TexMtx);
}

size_t ConvertTexSrtSoftimage(void* pDst, const void* pSrc)
{
    const TexSrt& srt = *static_cast<const TexSrt*>(pSrc);
    float sinR, cosR;
    int index = nn::util::RadianToAngleIndex(srt.r);
    nn::util::SinCosTable(&sinR, &cosR, index);
    const float sxcr = srt.sx * cosR;
    const float sxsr = srt.sx * sinR;
    const float sycr = srt.sy * cosR;
    const float sysr = srt.sy * sinR;

    TexMtx& mtx = *static_cast<TexMtx*>(pDst);
    mtx.m[0][0] =  sxcr;
    mtx.m[0][1] =  sysr;
    mtx.m[1][0] = -sxsr;
    mtx.m[1][1] =  sycr;
    mtx.m[2][0] =  sxsr - sxcr * srt.tx - sxsr * srt.ty;
    mtx.m[2][1] = -sycr - sysr * srt.tx + sycr * srt.ty + 1.0f;

    return sizeof(TexMtx);
}

size_t ConvertTexSrtExMaya(void* pDst, const void* pSrc)
{
    const TexSrtEx& srtEx = *static_cast<const TexSrtEx*>(pSrc);
    const TexSrt& srt = srtEx.srt;
    float sinR, cosR;
    int index = nn::util::RadianToAngleIndex(srt.r);
    nn::util::SinCosTable(&sinR, &cosR, index);
    const float sxcr = srt.sx * cosR;
    const float sxsr = srt.sx * sinR;
    const float sycr = srt.sy * cosR;
    const float sysr = srt.sy * sinR;
    const float tx = -0.5f * (sxcr + sxsr - srt.sx) - srt.tx * srt.sx;
    const float ty = -0.5f * (sycr - sysr + srt.sy) + srt.ty * srt.sy + 1.0f;

    nn::util::FloatColumnMajor4x3& mtx = *static_cast<nn::util::FloatColumnMajor4x3*>(pDst);
    // ToDo ポインター
    const nn::util::FloatColumnMajor4x3* pEffectMtx = srtEx.pEffectMtx;
    if (pEffectMtx)
    {
        mtx.m[0][0] =  sxcr * pEffectMtx->m[0][0] + sxsr * pEffectMtx->m[1][0] + tx * pEffectMtx->m[2][0];
        mtx.m[1][0] = -sysr * pEffectMtx->m[0][0] + sycr * pEffectMtx->m[1][0] + ty * pEffectMtx->m[2][0];
        mtx.m[0][1] =  sxcr * pEffectMtx->m[0][1] + sxsr * pEffectMtx->m[1][1] + tx * pEffectMtx->m[2][1];
        mtx.m[1][1] = -sysr * pEffectMtx->m[0][1] + sycr * pEffectMtx->m[1][1] + ty * pEffectMtx->m[2][1];
        mtx.m[0][2] =  sxcr * pEffectMtx->m[0][2] + sxsr * pEffectMtx->m[1][2] + tx * pEffectMtx->m[2][2];
        mtx.m[1][2] = -sysr * pEffectMtx->m[0][2] + sycr * pEffectMtx->m[1][2] + ty * pEffectMtx->m[2][2];
        mtx.m[0][3] =  sxcr * pEffectMtx->m[0][3] + sxsr * pEffectMtx->m[1][3] + tx * pEffectMtx->m[2][3];
        mtx.m[1][3] = -sysr * pEffectMtx->m[0][3] + sycr * pEffectMtx->m[1][3] + ty * pEffectMtx->m[2][3];

        memcpy(mtx.m[2], pEffectMtx->m[2], NUM_COMP_MTX14 << 2);
    }
    else
    {
        mtx.m[0][0] =  sxcr;
        mtx.m[1][0] = -sysr;
        mtx.m[0][1] =  sxsr;
        mtx.m[1][1] =  sycr;
        mtx.m[0][2] =  tx;
        mtx.m[1][2] =  ty;
        mtx.m[0][3] = mtx.m[1][3] = mtx.m[2][0] = mtx.m[2][1] = mtx.m[2][3] = 0.0f;
        mtx.m[2][2] = 1.0f;
    }

    return sizeof(nn::util::FloatColumnMajor4x3);
}

size_t ConvertTexSrtEx3dsmax(void* pDst, const void* pSrc)
{
    const TexSrtEx& srtEx = *static_cast<const TexSrtEx*>(pSrc);
    const TexSrt& srt = srtEx.srt;
    float sinR, cosR;
    int index = nn::util::RadianToAngleIndex(srt.r);
    nn::util::SinCosTable(&sinR, &cosR, index);
    const float sxcr = srt.sx * cosR;
    const float sxsr = srt.sx * sinR;
    const float sycr = srt.sy * cosR;
    const float sysr = srt.sy * sinR;
    const float tx = -sxcr * (srt.tx + 0.5f) + sxsr * (srt.ty - 0.5f) + 0.5f;
    const float ty =  sysr * (srt.tx + 0.5f) + sycr * (srt.ty - 0.5f) + 0.5f;

    nn::util::FloatColumnMajor4x3& mtx = *static_cast<nn::util::FloatColumnMajor4x3*>(pDst);
    // ToDo ポインター
    const nn::util::FloatColumnMajor4x3* pEffectMtx = srtEx.pEffectMtx;
    if (pEffectMtx)
    {
        mtx.m[0][0] =  sxcr * pEffectMtx->m[0][0] + sxsr * pEffectMtx->m[1][0] + tx * pEffectMtx->m[2][0];
        mtx.m[1][0] = -sysr * pEffectMtx->m[0][0] + sycr * pEffectMtx->m[1][0] + ty * pEffectMtx->m[2][0];
        mtx.m[0][1] =  sxcr * pEffectMtx->m[0][1] + sxsr * pEffectMtx->m[1][1] + tx * pEffectMtx->m[2][1];
        mtx.m[1][1] = -sysr * pEffectMtx->m[0][1] + sycr * pEffectMtx->m[1][1] + ty * pEffectMtx->m[2][1];
        mtx.m[0][2] =  sxcr * pEffectMtx->m[0][2] + sxsr * pEffectMtx->m[1][2] + tx * pEffectMtx->m[2][2];
        mtx.m[1][2] = -sysr * pEffectMtx->m[0][2] + sycr * pEffectMtx->m[1][2] + ty * pEffectMtx->m[2][2];
        mtx.m[0][3] =  sxcr * pEffectMtx->m[0][3] + sxsr * pEffectMtx->m[1][3] + tx * pEffectMtx->m[2][3];
        mtx.m[1][3] = -sysr * pEffectMtx->m[0][3] + sycr * pEffectMtx->m[1][3] + ty * pEffectMtx->m[2][3];

        memcpy(mtx.m[2], pEffectMtx->m[2], NUM_COMP_MTX14 << 2);
    }
    else
    {
        mtx.m[0][0] =  sxcr;
        mtx.m[1][0] = -sysr;
        mtx.m[0][1] =  sxsr;
        mtx.m[1][1] =  sycr;
        mtx.m[0][2] =  tx;
        mtx.m[1][2] =  ty;
        mtx.m[0][3] = mtx.m[1][3] = mtx.m[2][0] = mtx.m[2][1] = mtx.m[2][3] = 0.0f;
        mtx.m[2][2] = 1.0f;
    }

    return sizeof(nn::util::FloatColumnMajor4x3);
}

size_t ConvertTexSrtExSoftimage(void* pDst, const void* pSrc)
{
    const TexSrtEx& srtEx = *static_cast<const TexSrtEx*>(pSrc);
    const TexSrt& srt = srtEx.srt;
    float sinR, cosR;
    int index = nn::util::RadianToAngleIndex(srt.r);
    nn::util::SinCosTable(&sinR, &cosR, index);
    const float sxcr = srt.sx * cosR;
    const float sxsr = srt.sx * sinR;
    const float sycr = srt.sy * cosR;
    const float sysr = srt.sy * sinR;
    const float tx =  sxsr - sxcr * srt.tx - sxsr * srt.ty;
    const float ty = -sycr - sysr * srt.tx + sycr * srt.ty + 1.0f;

    nn::util::FloatColumnMajor4x3& mtx = *static_cast<nn::util::FloatColumnMajor4x3*>(pDst);
    // ToDo ポインター
    const nn::util::FloatColumnMajor4x3* pEffectMtx = srtEx.pEffectMtx;
    if (pEffectMtx)
    {
        mtx.m[0][0] = sxcr * pEffectMtx->m[0][0] - sxsr * pEffectMtx->m[1][0] + tx * pEffectMtx->m[2][0];
        mtx.m[1][0] = sysr * pEffectMtx->m[0][0] + sycr * pEffectMtx->m[1][0] + ty * pEffectMtx->m[2][0];
        mtx.m[0][1] = sxcr * pEffectMtx->m[0][1] - sxsr * pEffectMtx->m[1][1] + tx * pEffectMtx->m[2][1];
        mtx.m[1][1] = sysr * pEffectMtx->m[0][1] + sycr * pEffectMtx->m[1][1] + ty * pEffectMtx->m[2][1];
        mtx.m[0][2] = sxcr * pEffectMtx->m[0][2] - sxsr * pEffectMtx->m[1][2] + tx * pEffectMtx->m[2][2];
        mtx.m[1][2] = sysr * pEffectMtx->m[0][2] + sycr * pEffectMtx->m[1][2] + ty * pEffectMtx->m[2][2];
        mtx.m[0][3] = sxcr * pEffectMtx->m[0][3] - sxsr * pEffectMtx->m[1][3] + tx * pEffectMtx->m[2][3];
        mtx.m[1][3] = sysr * pEffectMtx->m[0][3] + sycr * pEffectMtx->m[1][3] + ty * pEffectMtx->m[2][3];

        memcpy(mtx.m[2], pEffectMtx->m[2], NUM_COMP_MTX14 << 2);
    }
    else
    {
        mtx.m[0][0] =  sxcr;
        mtx.m[1][0] =  sysr;
        mtx.m[0][1] = -sxsr;
        mtx.m[1][1] =  sycr;
        mtx.m[0][2] =  tx;
        mtx.m[1][2] =  ty;
        mtx.m[0][3] = mtx.m[1][3] = mtx.m[2][0] = mtx.m[2][1] = mtx.m[2][3] = 0.0f;
        mtx.m[2][2] = 1.0f;
    }

    return sizeof(nn::util::FloatColumnMajor4x3);
}

size_t (* const s_pFuncConvertTexSrt[])(void*, const void*) =
{
    &ConvertTexSrtMaya,
    &ConvertTexSrt3dsmax,
    &ConvertTexSrtSoftimage
};

size_t (* const s_pFuncConvertTexSrtEx[])(void*, const void*) =
{
    &ConvertTexSrtExMaya,
    &ConvertTexSrtEx3dsmax,
    &ConvertTexSrtExSoftimage
};

size_t (* const s_pSystemParamCallback[])(void*, const void*, const ResShaderParam*, const void*) =
{
    &ResShaderParam::ConvertSrt2dCallback,
    &ResShaderParam::ConvertSrt3dCallback,
    &ResShaderParam::ConvertTexSrtCallback,
    &ResShaderParam::ConvertTexSrtExCallback
};

} // anonymous namespace

//--------------------------------------------------------------------------------------------------

size_t ResShaderParam::GetSize(Type type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(type, Type_Bool, Type_Num);

    if (type <= Type_Float4)
    {
        return sizeof(float) * ((type & 0x3) + 1);
    }
    else if (type <= Type_Float4x4)
    {
        return sizeof(float) * 4 * (((type - Type_Reserved2) >> 2) + 2);
    }

    static const size_t s_tblSize[] =
    {
        sizeof(TexMtx), sizeof(nn::util::FloatColumnMajor4x3), sizeof(TexMtx)
    };
    return s_tblSize[type - Type_Srt2d];
}

size_t ResShaderParam::GetSrcSize(Type type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(type, Type_Bool, Type_Num);

    if (type <= Type_Float4)
    {
        int compCount = (type & 0x3) + 1;
        return sizeof(float) * compCount;
    }
    else if( type <= Type_Float4x4)
    {
        // colCount, rowCount は CPU で扱う row-major matrix 換算の値。
        const int colCount = (type & 0x3) + 1;
        const int rowCount = ((type - Type_Reserved2) >> 2) + 2;
        return sizeof(float) * colCount * rowCount;
    }

    static const size_t s_tblSrcSize[] =
    {
        sizeof(Srt2d), sizeof(Srt3d), sizeof(TexSrt), sizeof(TexSrtEx)
    };
    return s_tblSrcSize[type - Type_Srt2d];
}

bool ResShaderParam::SetDependPointer(void* pSrcShaderParam, const void* pDependParam) const NN_NOEXCEPT
{
    // depend の領域が確保されている + 依存先と依存元が一致している = シェーダー内で依存先に自身を指定している
    if (this->GetSrcSize() > GetSrcSize(this->GetType()) )
    {
        if (this->GetDependedIndex() == this->GetDependIndex())
        {
            nn::util::BytePtr bytePtr(pSrcShaderParam);
            bytePtr.Advance(this->GetType());

            //リソースのアラインメント対応後に有効化
            //bytePtr.AlignUp(sizeof(nn::Bit64));

            void** pDependPtr = bytePtr.Get<void*>();
            *pDependPtr = const_cast<void*>(pDependParam);
            return true;
        }
    }
    // 依存先に自身以外のシェーダーパラメーターを設定している場合は上書き設定しない。
    return false;
}

bool ResShaderParam::GetDependPointer(void** pOutParamPtr, const void* pSrcShaderParam) const NN_NOEXCEPT
{
    // depend の領域が確保されているかどうかを判断
    if (this->GetSrcSize() > GetSrcSize(this->GetType()))
    {
        nn::util::BytePtr bytePtr(const_cast<void*>(pSrcShaderParam));
        bytePtr.Advance(this->GetType());

        //リソースのアラインメント対応後に有効化
        //bytePtr.AlignUp(sizeof(nn::Bit64));

        void** pDependPtr = bytePtr.Get<void*>();
        *pOutParamPtr = *pDependPtr;
        return true;
    }
    *pOutParamPtr = NULL;
    return false;
}

template <bool swap>
void ResShaderParam::Convert(void* pDst, const void* pSrc) const NN_NOEXCEPT
{
    ResShaderParam::Type type = GetType();
    NN_G3D_REQUIRES_RANGE(type, Type_Bool, Type_Num, NN_G3D_RES_GET_NAME(this, GetId()));

    if (type <= Type_Float4)
    {
        int compCount = (type & 0x3) + 1;
        Copy32<swap>(pDst, pSrc, compCount);
    }
    else if(type <= Type_Float4x4)
    {
        // colCount, rowCount は CPU で扱う row-major matrix 換算の値。
        const int colCount = (type & 0x3) + 1;
        const int rowCount = ((type - Type_Reserved2) >> 2) + 2;
        const int compCount = static_cast<int>(colCount);
        const size_t size = sizeof(float) * colCount;
        const size_t stride = sizeof(float) * 4;
        for (int idxRow = 0; idxRow < rowCount; ++idxRow)
        {
            Copy32<swap>(pDst, pSrc, compCount);
            pDst = AddOffset(pDst, stride);
            pSrc = AddOffset(pSrc, size);
        }
    }
}

// SRT -> Mtx23 (column-major)
size_t ResShaderParam::ConvertSrt2dCallback(void* pDst, const void* pSrc, const ResShaderParam*, const void*) NN_NOEXCEPT
{
    const Srt2d& srt = *static_cast<const Srt2d*>(pSrc);

    float sinR, cosR;
    int index = nn::util::RadianToAngleIndex(srt.r);
    nn::util::SinCosTable(&sinR, &cosR, index);

    TexMtx& mtx = *static_cast<TexMtx*>(pDst);
    mtx.m[0][0] =  srt.sx * cosR;
    mtx.m[0][1] =  srt.sx * sinR;
    mtx.m[1][0] = -srt.sy * sinR;
    mtx.m[1][1] =  srt.sy * cosR;
    mtx.m[2][0] =  srt.tx;
    mtx.m[2][1] =  srt.ty;

    return sizeof(TexMtx);
}

// SRT -> Mtx34
size_t ResShaderParam::ConvertSrt3dCallback(void* pDst, const void* pSrc, const ResShaderParam*, const void*) NN_NOEXCEPT
{
    const Srt3d& srt = *static_cast<const Srt3d*>(pSrc);

    nn::util::FloatColumnMajor4x3& mtx = *static_cast<nn::util::FloatColumnMajor4x3*>(pDst);
    nn::util::Vector3fType scale =
    {
        {srt.scale.v[0], srt.scale.v[1], srt.scale.v[2]}
    };
    nn::util::Vector3fType rotate =
    {
        {srt.rotate.v[0], srt.rotate.v[1], srt.rotate.v[2]}
    };
    nn::util::Vector3fType translate =
    {
        {srt.translate.v[0], srt.translate.v[1], srt.translate.v[2]}
    };
    nn::util::Matrix4x3fType calculateMtx;
    MatrixSetScaleRotateXyz(&calculateMtx, scale, rotate);
    MatrixSetTranslate(&calculateMtx, translate);
    MatrixStore(&mtx, calculateMtx);

    //mtx.SetSR(srt.scale, srt.rotate);
    //mtx.SetT(srt.translate);

    return sizeof(nn::util::FloatColumnMajor4x3);
}

// SRT * Mtx34 -> Mtx34
size_t ResShaderParam::ConvertSrt2dExCallback(void* pDst, const void* pSrc, const ResShaderParam*, const void*) NN_NOEXCEPT
{
    const TexSrtEx& srtEx = *static_cast<const TexSrtEx*>(pSrc);
    const TexSrt& srt = srtEx.srt;

    float sinR, cosR;
    int index = nn::util::RadianToAngleIndex(srt.r);
    nn::util::SinCosTable(&sinR, &cosR, index);
    const float sxcr = srt.sx * cosR;
    const float sxsr = srt.sx * sinR;
    const float sycr = srt.sy * cosR;
    const float sysr = srt.sy * sinR;
    const float tx = srt.sx;
    const float ty = srt.sy;

    nn::util::FloatColumnMajor4x3& mtx = *static_cast<nn::util::FloatColumnMajor4x3*>(pDst);
    // ToDo ポインター
    const nn::util::FloatColumnMajor4x3* pEffectMtx = srtEx.pEffectMtx;
    if (pEffectMtx)
    {
        mtx.m[0][0] =  sxcr * pEffectMtx->m[0][0] - sxsr * pEffectMtx->m[1][0] + tx * pEffectMtx->m[2][0];
        mtx.m[1][0] =  sysr * pEffectMtx->m[0][0] + sycr * pEffectMtx->m[1][0] + ty * pEffectMtx->m[2][0];
        mtx.m[0][1] =  sxcr * pEffectMtx->m[0][1] - sxsr * pEffectMtx->m[1][1] + tx * pEffectMtx->m[2][1];
        mtx.m[1][1] =  sysr * pEffectMtx->m[0][1] + sycr * pEffectMtx->m[1][1] + ty * pEffectMtx->m[2][1];
        mtx.m[0][2] =  sxcr * pEffectMtx->m[0][2] - sxsr * pEffectMtx->m[1][2] + tx * pEffectMtx->m[2][2];
        mtx.m[1][2] =  sysr * pEffectMtx->m[0][2] + sycr * pEffectMtx->m[1][2] + ty * pEffectMtx->m[2][2];
        mtx.m[0][3] =  sxcr * pEffectMtx->m[0][3] - sxsr * pEffectMtx->m[1][3] + tx * pEffectMtx->m[2][3];
        mtx.m[1][3] =  sysr * pEffectMtx->m[0][3] + sycr * pEffectMtx->m[1][3] + ty * pEffectMtx->m[2][3];

        memcpy(mtx.m[2], pEffectMtx->m[2], NUM_COMP_MTX14 << 2);
    }
    else
    {
        mtx.m[0][0] =  sxcr;
        mtx.m[1][0] =  sysr;
        mtx.m[0][1] = -sxsr;
        mtx.m[1][1] =  sycr;
        mtx.m[0][2] =  tx;
        mtx.m[1][2] =  ty;
        mtx.m[0][3] = mtx.m[1][3] = mtx.m[2][0] = mtx.m[2][1] = mtx.m[2][3] = 0.0f;
        mtx.m[2][2] = 1.0f;
    }

    return sizeof(nn::util::FloatColumnMajor4x3);
}

size_t ResShaderParam::ConvertTexSrtCallback(void* pDst, const void* pSrc, const ResShaderParam*, const void*) NN_NOEXCEPT
{
    TexSrt::Mode mode = *static_cast<const TexSrt::Mode*>(pSrc);
    NN_SDK_ASSERT_RANGE(mode, TexSrt::Mode_Maya, TexSrt::Mode_Num);
    return s_pFuncConvertTexSrt[mode](pDst, pSrc);
}

size_t ResShaderParam::ConvertTexSrtExCallback(void* pDst, const void* pSrc, const ResShaderParam*, const void*) NN_NOEXCEPT
{
    TexSrt::Mode mode = *static_cast<const TexSrt::Mode*>(pSrc);
    NN_SDK_ASSERT_RANGE(mode, TexSrt::Mode_Maya, TexSrt::Mode_Num);
    return s_pFuncConvertTexSrtEx[mode](pDst, pSrc);
}

//--------------------------------------------------------------------------------------------------

BindResult ResMaterial::BindTexture(TextureBindCallback pCallback, void* pUserData) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pCallback != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;
    for (int idxTex = 0, textureCount = GetSamplerCount(); idxTex < textureCount; ++idxTex)
    {
        if (!IsTextureBound(idxTex))
        {
            TextureRef textureRef = pCallback(GetTextureName(idxTex), pUserData);
            SetTexture(idxTex, textureRef.GetTextureView());
            SetTextureDescriptorSlot(idxTex, textureRef.GetDescriptorSlot());
            if (textureRef.IsValid())
            {
                result.SetSuccessBit();
            }
            else
            {
                result.SetFailureBit();
            }
        }
    }

    return result;
}

bool ResMaterial::ForceBindTexture(const TextureRef& textureRef, const char* name) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(name != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    bool success = false;
    for (int idxTex = 0, textureCount = GetSamplerCount(); idxTex < textureCount; ++idxTex)
    {
        if (0 == strcmp(GetTextureName(idxTex), name))
        {
            SetTexture(idxTex, textureRef.GetTextureView());
            SetTextureDescriptorSlot(idxTex, textureRef.GetDescriptorSlot());
            success = true;
        }
    }
    return success;

}

void ResMaterial::ReleaseTexture() NN_NOEXCEPT
{
    for (int idxTex = 0, textureCount = GetSamplerCount(); idxTex < textureCount; ++idxTex)
    {
        SetTexture(idxTex, NULL);
        SetTextureDescriptorSlot(idxTex, nn::gfx::DescriptorSlot());
    }
}

void ResMaterial::Setup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    int samplerCount = this->GetSamplerCount();
    for (int samperIndex = 0; samperIndex < samplerCount; ++samperIndex)
    {
        nn::gfx::SamplerInfo* pSamplerInfo = GetSamplerInfo(samperIndex);
        nn::gfx::Sampler*     pSampler     = new (GetSampler(samperIndex)) nn::gfx::Sampler;
        pSampler->Initialize(pDevice, *pSamplerInfo);
        nn::gfx::util::SetSamplerDebugLabel(pSampler, GetSamplerName(samperIndex));
    }

    for (int idxShaderParam = 0, shaderParamCount = GetShaderParamCount();
        idxShaderParam < shaderParamCount; ++idxShaderParam)
    {
        ResShaderParam* pShaderParam = GetShaderParam(idxShaderParam);
        ResShaderParam::Type type = pShaderParam->GetType();
        if (type >= ResShaderParam::Type_Srt2d && pShaderParam->GetConvertShaderParamCallback() == NULL)
        {
            pShaderParam->SetConvertShaderParamCallback(
                s_pSystemParamCallback[type - ResShaderParam::Type_Srt2d]);
        }
    }
}

void ResMaterial::Cleanup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    int samplerCount = this->GetSamplerCount();
    for (int samperIndex = 0; samperIndex < samplerCount; ++samperIndex)
    {
        nn::gfx::Sampler* pSampler = GetSampler(samperIndex);
        if (nn::gfx::IsInitialized(*pSampler) == true)
        {
            pSampler->Finalize(pDevice);
            pSampler->nn::gfx::Sampler::~TSampler();
        }
    }
}

void ResMaterial::Reset() NN_NOEXCEPT
{
    Reset(ResetGuardFlag_None);
}

void ResMaterial::Reset(Bit32 resetGuardFlag) NN_NOEXCEPT
{
    int shaderParamCount = this->GetShaderParamCount();
    for (int shaderParamIndex = 0; shaderParamIndex < shaderParamCount; ++shaderParamIndex)
    {
        ResShaderParam* pShaderParam = this->GetShaderParam(shaderParamIndex);
        pShaderParam->SetOffset(-1);
        pShaderParam->SetConvertShaderParamCallback(NULL);
    }

    this->SetTextureCount(this->GetSamplerCount());
    this->SetRawParamSize(0);

    if ((resetGuardFlag & ResetGuardFlag_UserPtr) != ResetGuardFlag_UserPtr)
    {
        ToData().pUserPtr.Set(NULL);
    }

    ToData().shaderParamVolatileCount = 0;
    memset(pVolatileFlag.Get(), 0, shaderParamCount >> 5);
}

template void ResShaderParam::Convert<true>(void*, const void*) const;
template void ResShaderParam::Convert<false>(void*, const void*) const;

}} // namespace nn::g3d

NN_PRAGMA_POP_WARNINGS
