﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_ResBoneVisibilityAnim.h>
#include <algorithm>

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_SHADOW

namespace nn { namespace g3d {

BindResult ResBoneVisibilityAnim::PreBind(const ResModel* pModel) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModel != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    ToData().pBindModel.Set(const_cast<ResModel*>(pModel));
    uint16_t* pBindIndexArray = ToData().pBindIndexArray.Get();
    const nn::util::BinPtrToString* pNameArray = ToData().pNameArray.Get();
    const nn::util::ResDic* pDic = pModel->GetSkeleton()->ToData().pBoneDic.Get();

    BindResult result;
    for (int idxAnim = 0, animCount = GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        const char* pName = pNameArray[idxAnim].Get()->GetData();
        int idxTarget = pDic->FindIndex(pName);
        if (idxTarget >= 0)
        {
            pBindIndexArray[idxAnim] = static_cast<uint16_t>(idxTarget);
            result.SetSuccessBit();
        }
        else
        {
            pBindIndexArray[idxAnim] = AnimFlag_NotBound;
            result.SetFailureBit();
        }
    }
    return result;
}

BindResult ResBoneVisibilityAnim::BindCheck(const ResModel* pModel) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModel != NULL, NN_G3D_RES_GET_NAME(this, GetName()));

    const nn::util::BinPtrToString* pNameArray = ToData().pNameArray.Get();
    const nn::util::ResDic* pDic = pModel->GetSkeleton()->ToData().pBoneDic.Get();

    BindResult result;
    for (int idxAnim = 0, animCount = GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        const char* pName = pNameArray[idxAnim].Get()->GetData();
        int idxTarget = pDic->FindIndex(pName);
        if (idxTarget >= 0)
        {
            result.SetSuccessBit();
        }
        else
        {
            result.SetFailureBit();
        }
    }
    return result;
}

bool ResBoneVisibilityAnim::BakeCurve(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    if (bufferSize == 0)
    {
        return true;
    }
    if (pBuffer == NULL ||
        bufferSize < GetBakedSize())
    {
        return false;
    }
    for (int idxCurve = 0, curveCount = GetCurveCount(); idxCurve < curveCount; ++idxCurve)
    {
        ResAnimCurve* curve = GetCurve(idxCurve);
        size_t size = curve->CalculateBakedIntSize();
        curve->BakeInt(pBuffer, size);
        pBuffer = AddOffset(pBuffer, size);
    }
    ToData().flag |= Flag_CurveBaked;

    return true;
}

void* ResBoneVisibilityAnim::ResetCurve() NN_NOEXCEPT
{
    void* pBuffer = NULL;
    if (IsCurveBaked())
    {
        bool foundCurve = false;
        for (int idxCurve = 0, curveCount = GetCurveCount(); idxCurve < curveCount; ++idxCurve)
        {
            ResAnimCurve* curve = GetCurve(idxCurve);
            Bit32 type = curve->ToData().flag & ResAnimCurve::Mask_Curve;
            if (!foundCurve && type == ResAnimCurve::CurveType_BakedBool)
            {
                pBuffer = curve->ToData().pKeyArray.Get();
                foundCurve = true;
            }
            curve->Reset();
        }
        ToData().flag ^= Flag_CurveBaked;
    }

    return pBuffer;
}

void ResBoneVisibilityAnim::Reset() NN_NOEXCEPT
{
    ToData().pBindModel.Set(NULL);
    uint16_t* pBindIndexArray = ToData().pBindIndexArray.Get();
    for (int idxAnim = 0, animCount = GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        pBindIndexArray[idxAnim] = AnimFlag_NotBound;
    }

    ResetCurve();
}

}} // namespace nn::g3d

NN_PRAGMA_POP_WARNINGS
