﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_MaterialObj.h>
#include <nn/gfx/util/gfx_ObjectDebugLabel.h>
#include <algorithm>

namespace nn { namespace g3d {

//---------------------------------------------------------------------------------------------------------------------

void MaterialObj::InitializeArgument::CalculateMemorySize() NN_NOEXCEPT
{
    NN_G3D_REQUIRES(GetBufferingCount() > 0, NN_G3D_RES_GET_NAME(GetResource(), GetName()));
    const ResMaterial* pRes = GetResource();
    int dirtyFlagCount = pRes->GetShaderParamCount();
    int bufferingCount = GetBufferingCount();

    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_MemoryBlock[blockIndex].Initialize();
    }

    // memoryblockのアライメントのデフォルトは8
    m_MemoryBlock[MemoryBlockIndex_DirtyFlagSet].SetSize(FlagSet::CalcBufferSize(dirtyFlagCount, bufferingCount));
    m_MemoryBlock[MemoryBlockIndex_SrcParamArray].SetSize(nn::util::align_up(pRes->GetSrcParamSize(), Alignment_Default));
    m_MemoryBlock[MemoryBlockIndex_TextureArray].SetSizeBy<nn::gfx::TextureView*>(1, pRes->GetTextureCount());
    m_MemoryBlock[MemoryBlockIndex_TextureDescriptorSlotArray].SetSizeBy<nn::gfx::DescriptorSlot>(1, pRes->GetTextureCount());
    m_MemoryBlock[MemoryBlockIndex_MatBlockArray].SetSizeBy<nn::gfx::Buffer>(1,  bufferingCount);

    m_WorkMemory.Initialize();
    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_WorkMemory.Append(&m_MemoryBlock[blockIndex]);
    }
}

//--------------------------------------------------------------------------------------------------

bool MaterialObj::Initialize(const InitializeArgument& arg, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pBuffer != NULL,                      NN_G3D_RES_GET_NAME(arg.GetResource(), GetName()));
    NN_G3D_REQUIRES(IsAligned(pBuffer, Alignment_Buffer), NN_G3D_RES_GET_NAME(arg.GetResource(), GetName()));

    if (arg.IsMemoryCalculated() == false)
    {
        return false;
    }
    if (arg.GetWorkMemorySize() > bufferSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    const ResMaterial* pRes = arg.GetResource();
    int textureCount    = pRes->GetTextureCount();
    int dirtyFlagCount  = pRes->GetShaderParamCount();

    // メンバの初期化。
    m_pRes                        = pRes;
    m_Flag                        = 0;
    m_BufferingCount              = static_cast<uint8_t>(arg.GetBufferingCount());
    m_pSrcParam                   = arg.GetBuffer(pBuffer,  InitializeArgument::MemoryBlockIndex_SrcParamArray);
    memcpy(m_pSrcParam, pRes->ToData().pSrcParam.Get(), pRes->GetSrcParamSize());
    m_ppTextureArray              = arg.GetBuffer<const nn::gfx::TextureView*>(pBuffer,  InitializeArgument::MemoryBlockIndex_TextureArray);
    m_pTextureDescriptorSlotArray = arg.GetBuffer<nn::gfx::DescriptorSlot>(pBuffer, InitializeArgument::MemoryBlockIndex_TextureDescriptorSlotArray);
    m_pMaterialBlockArray         = arg.GetBuffer<nn::gfx::Buffer>(pBuffer, InitializeArgument::MemoryBlockIndex_MatBlockArray);
    m_pUserPtr                    = NULL;
    m_pBufferPtr                  = pBuffer;
    m_SizePerMaterialBlock        = 0;
    m_pMemoryPool                 = NULL;
    m_MemoryPoolOffset            = 0;

    m_DirtyFlagSet.Initialize(dirtyFlagCount, m_BufferingCount,
                              arg.GetBuffer(pBuffer, InitializeArgument::MemoryBlockIndex_DirtyFlagSet),
                              arg.GetBufferSize(InitializeArgument::MemoryBlockIndex_DirtyFlagSet));

    for (int idxTexture = 0; idxTexture < textureCount; ++idxTexture)
    {
        m_ppTextureArray[idxTexture] = pRes->GetTexture(idxTexture);
        m_pTextureDescriptorSlotArray[idxTexture] = pRes->GetTextureDescriptorSlot(idxTexture);
    }

    InitializeDependPointer();

    return true;
}

size_t MaterialObj::GetBlockBufferAlignment(nn::gfx::Device* pDevice) const NN_NOEXCEPT
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize(m_pRes->GetRawParamSize());
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_ConstantBuffer );

    return nn::gfx::Buffer::GetBufferAlignment(pDevice, info);
}

size_t MaterialObj::CalculateBlockBufferSize(nn::gfx::Device* pDevice) const NN_NOEXCEPT
{
    return nn::util::align_up(m_pRes->GetRawParamSize(), GetBlockBufferAlignment(pDevice)) * m_BufferingCount;
}

void MaterialObj::SetupBlockBufferImpl(nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t offset, size_t memoryPoolSize) NN_NOEXCEPT
{
    NN_UNUSED(memoryPoolSize);

    // アプリケーションがデフォルト値でユニフォームブロックを埋められるよう、
    // ここではシェーダーパラメータを変換せず、最初の CalculateMaterial で変換する。
    ResetDirtyFlags();
    // 上記の処理でメインは初期化されているので、サブをクリアしておく
    m_DirtyFlagSet.ClearAllSubFlag();

    // 定数バッファー作成。バッファー数分、バッファーオブジェクトおよびビューオブジェクトを作る
    m_SizePerMaterialBlock = m_pRes->GetRawParamSize();
    ptrdiff_t memoryPoolOffset = offset;
    for (int bufferIndex = 0; bufferIndex < m_BufferingCount; bufferIndex++)
    {
        nn::gfx::Buffer::InfoType bufferInfo;
        bufferInfo.SetDefault();
        bufferInfo.SetSize(m_SizePerMaterialBlock);
        bufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
        nn::gfx::Buffer* pMatBlock = new(&m_pMaterialBlockArray[bufferIndex]) nn::gfx::Buffer;
        pMatBlock->Initialize(pDevice, bufferInfo, pMemoryPool, memoryPoolOffset, m_SizePerMaterialBlock);
        nn::gfx::util::SetBufferDebugLabel(pMatBlock, "g3d_MaterialUniformBlock");
        memoryPoolOffset += nn::util::align_up(m_SizePerMaterialBlock, GetBlockBufferAlignment(pDevice));
    }

    m_Flag |= Flag_BlockBufferValid;

    return;
}

bool MaterialObj::SetupBlockBuffer(nn::gfx::Device* pDevice, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t offset, size_t memoryPoolSize) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pMemoryPool != NULL || memoryPoolSize == 0, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(IsBlockBufferValid() == false,              NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    size_t size = CalculateBlockBufferSize(pDevice);

    if (size > memoryPoolSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    // sizeが0の場合、nn::gfx::bufferを作成する必要がないので、リターンする
    if (size == 0)
    {
        return true;
    }

    m_pMemoryPool = pMemoryPool;
    m_MemoryPoolOffset = offset;

    SetupBlockBufferImpl(pDevice, pMemoryPool, offset, memoryPoolSize);

    return true;
}

void MaterialObj::CleanupBlockBuffer(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsBlockBufferValid() == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    for (int blockIndex = 0; blockIndex < m_BufferingCount; blockIndex++)
    {
        nn::gfx::Buffer& materialBlock = m_pMaterialBlockArray[blockIndex];
        materialBlock.Finalize(pDevice);
        materialBlock.nn::gfx::Buffer::~TBuffer();
    }

    m_Flag &= ~Flag_BlockBufferValid;
    m_SizePerMaterialBlock = 0;

    m_pMemoryPool = NULL;
    m_MemoryPoolOffset = 0;
}

void MaterialObj::CalculateMaterial(int bufferIndex) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(bufferIndex < GetBufferingCount(), NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (m_SizePerMaterialBlock == 0)
    {
        return; // 更新不要。
    }

    m_DirtyFlagSet.Caclulate();

    if (!m_DirtyFlagSet.IsSubSetDirty(bufferIndex) && !m_pRes->HasVolatile())
    {
        return;
    }

    // bufferIndex のバッファーとダーティフラグで更新します。
    void* pBuffer = GetMaterialBlock(bufferIndex)->Map();
    if (IsBlockSwapEnabled())
    {
        ConvertDirtyParams<true>(pBuffer, m_DirtyFlagSet.GetFlagSet(bufferIndex));
    }
    else
    {
        ConvertDirtyParams<false>(pBuffer, m_DirtyFlagSet.GetFlagSet(bufferIndex));
    }
    m_DirtyFlagSet.ClearSubFlag(bufferIndex);

    GetMaterialBlock(bufferIndex)->FlushMappedRange(0, m_SizePerMaterialBlock);
    GetMaterialBlock(bufferIndex)->Unmap();
}

template <bool swap>
void MaterialObj::ConvertParams(void* pBuffer) NN_NOEXCEPT
{
    if (m_pRes->GetRawParamSize() <= 0)
    {
        return;
    }

    int paramCount = m_pRes->GetShaderParamCount();
    for (int idxParam = 0; idxParam < paramCount; ++idxParam)
    {
        const ResShaderParam* pParam = m_pRes->GetShaderParam(idxParam);
        ptrdiff_t offset = pParam->GetOffset();
        if (offset >= 0)
        {
            void* pDst = AddOffset(pBuffer, offset);
            const void* pSrc = AddOffset(m_pSrcParam, pParam->GetSrcOffset());
            pParam->Convert<swap>(pDst, pSrc, m_pUserPtr);
        }
    }
}

template <bool swap>
void MaterialObj::ConvertDirtyParams(void* pBuffer, Bit32* pDirtyFlagArray) NN_NOEXCEPT
{
    int dirtyFlagCount = m_pRes->GetShaderParamCount();
    const Bit32* pVolatileFlagArray = m_pRes->ToData().pVolatileFlag.Get();

    for (int idxFlag32 = 0, flag32Count = (dirtyFlagCount + 31) >> 5;
        idxFlag32 < flag32Count; ++idxFlag32)
    {
        Bit32 flag32 = pDirtyFlagArray[idxFlag32];
        Bit32 flagVolatile = pVolatileFlagArray[idxFlag32];
        flag32 |= flagVolatile;
        while (flag32)
        {
            int bitIndex = 31 - CountLeadingZeros(flag32);
            int paramIndex = (idxFlag32 << 5) + bitIndex;
            const ResShaderParam* pParam = GetResShaderParam(paramIndex);
            NN_G3D_ASSERT(pParam->GetOffset() >= 0, NN_G3D_RES_GET_NAME(m_pRes, GetName())); // オフセットが負ならフラグは立たない。
            void* pDst = AddOffset(pBuffer, pParam->GetOffset());
            const void* pSrc = AddOffset(m_pSrcParam, pParam->GetSrcOffset());
            pParam->Convert<swap>(pDst, pSrc, m_pUserPtr);
            flag32 ^= 0x1 << bitIndex;
        }
    }
}

template <bool swap>
void MaterialObj::ConvertVolatileParams(void* pBuffer) NN_NOEXCEPT
{
    int dirtyFlagCount = m_pRes->GetShaderParamCount();

    // 外部データを参照するパラメータはダーティフラグでは更新を判定できないので、
    // 常に計算しなおす。
    for (int idxParam = dirtyFlagCount, end = m_pRes->GetShaderParamCount();
        idxParam < end; ++idxParam)
    {
        const ResShaderParam* pParam = GetResShaderParam(idxParam);
        ptrdiff_t offset = pParam->GetOffset();
        if (offset >= 0)
        {
            void* pDst = AddOffset(pBuffer, offset);
            const void* pSrc = AddOffset(m_pSrcParam, pParam->GetSrcOffset());
            pParam->Convert<swap>(pDst, pSrc, m_pUserPtr);
        }
    }
}

void MaterialObj::ResetDirtyFlags() NN_NOEXCEPT
{
    int shaderParamCount = m_pRes->GetShaderParamCount();
    m_DirtyFlagSet.ClearMainFlag();

    for (int idxShaderParam = 0; idxShaderParam < shaderParamCount; ++idxShaderParam)
    {
        const ResShaderParam* pParam = GetResShaderParam(idxShaderParam);
        // ユニフォームブロック内のオフセットが解決されていないパラメータは変換できないので、
        // 解決されているパラメータのみダーティフラグを立てる。
        if (pParam->GetOffset() >= 0)
        {
            m_DirtyFlagSet.Set(idxShaderParam);
        }
    }
}

void MaterialObj::InitializeDependPointer() NN_NOEXCEPT
{
    for (int idxShaderParam = 0, shaderParamCount = m_pRes->GetShaderParamCount();
        idxShaderParam < shaderParamCount; ++idxShaderParam)
    {
        const ResShaderParam *pResShaderParam = m_pRes->GetShaderParam(idxShaderParam);
        int dependIndex = pResShaderParam->GetDependIndex();
        if (dependIndex != idxShaderParam)
        {
            void* pSrcShaderParam = AddOffset(m_pSrcParam, pResShaderParam->GetSrcOffset());
            void* pDependShaderParam = AddOffset(m_pSrcParam, m_pRes->GetShaderParam(dependIndex)->GetSrcOffset());

            nn::util::BytePtr bytePtr(pSrcShaderParam);
            bytePtr.Advance(pResShaderParam->GetType());
            //リソースのアラインメント対応後に有効化
            //bytePtr.AlignUp(sizeof(nn::Bit64));
            void** pDependPtr = bytePtr.Get<void*>();
            *pDependPtr = pDependShaderParam;
        }
    }
}

}} // namespace nn::g3d
