﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_BoneVisibilityAnimObj.h>
#include <algorithm>
#include <nn/g3d/detail/g3d_Flag.h>
#include <nn/g3d/detail/g3d_Perf.h>
#include <nn/g3d/g3d_ModelObj.h>

namespace nn { namespace g3d {

void BoneVisibilityAnimObj::InitializeArgument::CalculateMemorySize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid() == true);

    int targetCount = 0;
    int animCount = 0;
    if (GetMaxBoneAnimCount() > 0)
    {
        targetCount = GetMaxBoneCount();
        animCount = GetMaxBoneAnimCount();
    }
    int bindCount = std::max(targetCount, animCount);
    int curveCount = GetMaxCurveCount();

    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_MemoryBlock[blockIndex].Initialize();
    }

    m_MemoryBlock[MemoryBlockIndex_ResultBuffer].SetSize(nn::util::align_up(animCount, 32) >> 3);
    m_MemoryBlock[MemoryBlockIndex_BindTable].SetSizeBy<Bit32>(1,  bindCount);
    size_t size = IsContextEnabled() ? nn::util::align_up(sizeof(AnimFrameCache) * curveCount, Alignment_Default) : 0;
    m_MemoryBlock[MemoryBlockIndex_FrameCacheArray].SetSize(size);

    m_WorkMemory.Initialize();
    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_WorkMemory.Append(&m_MemoryBlock[blockIndex]);
    }
}

bool BoneVisibilityAnimObj::Initialize(const InitializeArgument& arg, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pBuffer != NULL || bufferSize == 0);
    NN_SDK_REQUIRES(IsAligned(pBuffer, Alignment_Buffer));

    if (!arg.IsMemoryCalculated() || !arg.IsValid())
    {
        return false;
    }
    if (arg.GetWorkMemorySize() > bufferSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    int targetCount = 0;
    int animCount = 0;
    if (arg.GetMaxBoneAnimCount() > 0)
    {
        targetCount = arg.GetMaxBoneCount();
        animCount = arg.GetMaxBoneAnimCount();
    }
    int bindCount = std::max(targetCount, animCount);
    int curveCount = arg.GetMaxCurveCount();

    // メンバの初期化。
    SetBufferPtr(pBuffer);
    // フレーム関係はリソース設定時に初期化
    m_pRes = NULL;
    GetBindTable().Initialize(arg.GetBuffer<Bit32>(pBuffer, InitializeArgument::MemoryBlockIndex_BindTable), bindCount);
    GetContext().Initialize(arg.GetBuffer<AnimFrameCache>(pBuffer, InitializeArgument::MemoryBlockIndex_FrameCacheArray), curveCount);
    SetResultBuffer(arg.GetBuffer(pBuffer, InitializeArgument::MemoryBlockIndex_ResultBuffer));
    m_MaxBoneAnim = static_cast<uint16_t>(arg.GetMaxBoneAnimCount());

    return true;
}

void BoneVisibilityAnimObj::SetResource(const ResBoneVisibilityAnim* pRes) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsAcceptable(pRes) == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    m_pRes = pRes;
    m_pCurveArray = pRes->ToData().pCurveArray.Get();
    m_CurveCount = pRes->GetCurveCount();

    SetTargetUnbound();

    bool loop = (pRes->ToData().flag & AnimFlag_PlayPolicyLoop) != 0;
    ResetFrameCtrl(pRes->GetFrameCount(), loop);
    GetBindTable().SetAnimCount(pRes->GetAnimCount());
    GetContext().SetCurveCount(pRes->GetCurveCount());
}

BindResult BoneVisibilityAnimObj::Bind(const ResModel* pModel) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsAcceptable(pModel) == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    const nn::util::ResDic* pDic = NULL;
    int count = 0;
    const ResSkeleton* pSkeleton = pModel->GetSkeleton();
    pDic = pSkeleton->ToData().pBoneDic.Get();
    count = pSkeleton->GetBoneCount();

    NN_G3D_ASSERT(count <= GetBindTable().GetSize(), NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    BindResult result;
    AnimBindTable& bindTable = GetBindTable();
    bindTable.ClearAll(count);
    const nn::util::BinPtrToString* pNameArray = m_pRes->ToData().pNameArray.Get();
    for (int idxAnim = 0, animCount = bindTable.GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        const char* pName = pNameArray[idxAnim].Get()->GetData();
        int idxTarget = pDic->FindIndex(pName);
        if (idxTarget >= 0)
        {
            bindTable.Bind(idxAnim, idxTarget);
            result.SetSuccessBit();
        }
        else
        {
            result.SetFailureBit();
        }
    }

    SetTargetBound();
    BoneVisibilityAnimObj::ClearResult();
    return result;
}

BindResult BoneVisibilityAnimObj::Bind(const ModelObj* pModelObj) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModelObj != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    return Bind(pModelObj->GetResource());
}

void BoneVisibilityAnimObj::BindFast(const ResModel* pModel) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsAcceptable(pModel) == true,     NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pModel == m_pRes->GetBindModel(), NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    int count = pModel->GetSkeleton()->GetBoneCount();

    AnimBindTable& bindTable = GetBindTable();
    bindTable.ClearAll(count);
    bindTable.BindAll(m_pRes->ToData().pBindIndexArray.Get());

    SetTargetBound();
    BoneVisibilityAnimObj::ClearResult();
}

void BoneVisibilityAnimObj::ClearResult() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);

    int sizeResult = nn::util::align_up(GetAnimCount(), 32) >> 3; // 1ビジビリティ1ビットでのサイズ。
    memcpy(GetResultBuffer(), m_pRes->ToData().pBaseValueArray.Get(), sizeResult);
}

void BoneVisibilityAnimObj::Calculate() NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("BoneVisibilityAnimObj::Calculate");
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);
    NN_G3D_REQUIRES(IsTargetBound() == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (IsFrameChanged())
    {
        float frame = GetFrameCtrl().GetFrame();
        const AnimBindTable& bindTable = GetBindTable();
        AnimContext& context = GetContext();
        Bit32* pResultBuffer = static_cast<Bit32*>(GetResultBuffer());
        if (context.IsFrameCacheValid())
        {
            for (int idxCurve = 0; idxCurve < m_CurveCount; ++idxCurve)
            {
                const ResAnimCurve* pCurve = GetCurve(idxCurve);
                int idxAnim = static_cast<int>(pCurve->ToData().targetOffset);

                if (!bindTable.IsCalculateEnabled(idxAnim))
                {
                    continue;
                }

                int visible = pCurve->EvaluateInt(frame, context.GetFrameCacheArray(idxCurve));
                // relative_repeat で範囲外を指定した場合の処理
                if (visible != 0)
                {
                    visible = 1;
                }
                SetBit(pResultBuffer, idxAnim, static_cast<Bit32>(visible));
            }
        }
        else
        {
            for (int idxCurve = 0; idxCurve < m_CurveCount; ++idxCurve)
            {
                const ResAnimCurve* pCurve = GetCurve(idxCurve);
                int idxAnim = static_cast<int>(pCurve->ToData().targetOffset);

                if (!bindTable.IsCalculateEnabled(idxAnim))
                {
                    continue;
                }

                int visible = pCurve->EvaluateInt(frame);
                // relative_repeat で範囲外を指定した場合の処理
                if (visible != 0)
                {
                    visible = 1;
                }
                SetBit(pResultBuffer, idxAnim, static_cast<Bit32>(visible));
            }
        }
        UpdateLastFrame();
    }
}

void BoneVisibilityAnimObj::ApplyTo(ModelObj* pModelObj) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(m_pRes != NULL,          NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(IsTargetBound() == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pModelObj != NULL,       NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    const Bit32* pResultBuffer = static_cast<const Bit32*>(GetResultBuffer());
    const AnimBindTable& bindTable = GetBindTable();
    for (int idxAnim = 0, animCount = bindTable.GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        if (!bindTable.IsApplyEnabled(idxAnim))
        {
            continue;
        }

        int  idxTarget = bindTable.GetTargetIndex(idxAnim);
        bool visible   = IsBitOn<bool>(pResultBuffer, idxAnim);
        pModelObj->SetBoneVisible(idxTarget, visible);
    }
}

void BoneVisibilityAnimObj::RevertTo(ModelObj* pModelObj) const NN_NOEXCEPT
{
    NN_G3D_REQUIRES(m_pRes != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(IsTargetBound(), NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pModelObj != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    SkeletonObj* pSkeletonObj = pModelObj->GetSkeleton();

    const AnimBindTable& bindTable = GetBindTable();
    for (int idxAnim = 0, animCount = bindTable.GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        if (!bindTable.IsApplyEnabled(idxAnim))
        {
            continue;
        }

        int  idxTarget = bindTable.GetTargetIndex(idxAnim);
        const ResBone* pResBone = pSkeletonObj->GetResBone(idxTarget);
        bool visible = pResBone->IsVisible();

        pModelObj->SetBoneVisible(idxTarget, visible);
    }
}

}} // namespace nn::g3d
