﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/font/font_PairFont.h>

namespace nn {
namespace font {

/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

PairFont::PairFont()
:   m_pFirst(NULL),
    m_pSecond(NULL),
    m_AlternateWithFirst(true)
{
}

PairFont::PairFont(Font* pFirst, Font* pSecond)
:   m_pFirst(NULL),
    m_pSecond(NULL),
    m_AlternateWithFirst(true)
{
    SetFont(pFirst, pSecond);
}


PairFont::~PairFont()
{
}


/* ------------------------------------------------------------------------
        フォントアクセサ
   ------------------------------------------------------------------------ */

void
PairFont::SetFont(Font* pFirst, Font* pSecond)
{
    NN_SDK_ASSERT_NOT_NULL(pFirst);
    NN_SDK_ASSERT_NOT_NULL(pSecond);
    NN_SDK_ASSERT(pFirst->GetCharacterCode() == pSecond->GetCharacterCode());
    m_pFirst = pFirst;
    m_pSecond = pSecond;
}


/* ------------------------------------------------------------------------
        フォント全体情報アクセサ
   ------------------------------------------------------------------------ */

int
PairFont::GetWidth() const
{
    return std::max(m_pFirst->GetWidth(), m_pSecond->GetWidth());
}

int
PairFont::GetHeight() const
{
    return std::max(m_pFirst->GetHeight(), m_pSecond->GetHeight());
}

int
PairFont::GetAscent() const
{
    return m_pFirst->GetHeight() >= m_pSecond->GetHeight() ?
        m_pFirst->GetAscent():
        m_pSecond->GetAscent();
}

int
PairFont::GetDescent() const
{
    return m_pFirst->GetHeight() >= m_pSecond->GetHeight() ?
        m_pFirst->GetDescent():
        m_pSecond->GetDescent();
}

int
PairFont::GetBaselinePos() const
{
    return m_pFirst->GetHeight() >= m_pSecond->GetHeight() ?
        m_pFirst->GetBaselinePos():
        m_pSecond->GetBaselinePos();
}

int
PairFont::GetCellHeight() const
{
    return m_pFirst->GetHeight() >= m_pSecond->GetHeight() ?
        m_pFirst->GetCellHeight():
        m_pSecond->GetCellHeight();
}

int
PairFont::GetCellWidth() const
{
    return m_pFirst->GetWidth() >= m_pSecond->GetWidth() ?
        m_pFirst->GetCellWidth():
        m_pSecond->GetCellWidth();
}

int
PairFont::GetMaxCharWidth() const
{
    return std::max(m_pFirst->GetMaxCharWidth(), m_pSecond->GetMaxCharWidth());
}

Font::Type
PairFont::GetType() const
{
    return Type_Pair;
}

TexFmt
PairFont::GetTextureFormat() const
{
    // テクスチャフォーマットは一致していなければならない
    NN_SDK_ASSERT(m_pFirst->GetTextureFormat() == m_pSecond->GetTextureFormat());

    return m_pFirst->GetTextureFormat();
}

int
PairFont::GetLineFeed() const
{
    return std::max(m_pFirst->GetLineFeed(), m_pSecond->GetLineFeed());
}

const CharWidths
PairFont::GetDefaultCharWidths() const
{
    return m_pFirst->GetWidth() >= m_pSecond->GetWidth() ?
        m_pFirst->GetDefaultCharWidths():
        m_pSecond->GetDefaultCharWidths();
}

void
PairFont::SetDefaultCharWidths(const CharWidths& widths)
{
    m_pFirst->SetDefaultCharWidths(widths);
    m_pSecond->SetDefaultCharWidths(widths);
}

bool
PairFont::SetAlternateChar(uint32_t c)
{
    bool bFirst = m_pFirst->SetAlternateChar(c);
    bool bSecond = m_pSecond->SetAlternateChar(c);
    if (bFirst)
    {
        m_AlternateWithFirst = true;
        return true;
    }
    else if (bSecond)
    {
        m_AlternateWithFirst = false;
        return true;
    }
    else
    {
        return false;
    }
}

void
PairFont::SetLineFeed(int linefeed)
{
    NN_SDK_ASSERT_RANGE(linefeed, 0, USHRT_MAX);

    return m_pFirst->GetHeight() >= m_pSecond->GetHeight() ?
        m_pFirst->SetLineFeed(linefeed):
        m_pSecond->SetLineFeed(linefeed);
}



/* ------------------------------------------------------------------------
        文字単体情報アクセサ
   ------------------------------------------------------------------------ */

int
PairFont::GetCharWidth(uint32_t c) const
{
    if (m_pFirst->HasGlyph(c))
    {
        return m_pFirst->GetCharWidth(c);
    }
    else if (m_pSecond->HasGlyph(c))
    {
        return m_pSecond->GetCharWidth(c);
    }
    else if (m_AlternateWithFirst)
    {
        return m_pFirst->GetCharWidth(c);
    }
    else
    {
        return m_pSecond->GetCharWidth(c);
    }
}

const CharWidths
PairFont::GetCharWidths(uint32_t c) const
{
    if (m_pFirst->HasGlyph(c))
    {
        return m_pFirst->GetCharWidths(c);
    }
    else if (m_pSecond->HasGlyph(c))
    {
        return m_pSecond->GetCharWidths(c);
    }
    else if (m_AlternateWithFirst)
    {
        return m_pFirst->GetCharWidths(c);
    }
    else
    {
        return m_pSecond->GetCharWidths(c);
    }
}

void
PairFont::GetGlyph(
     Glyph*     glyph,
     uint32_t   c
) const
{
    if (m_pFirst->HasGlyph(c))
    {
        m_pFirst->GetGlyph(glyph, c);
        glyph->baselineDifference += static_cast<int16_t>(m_pFirst->GetBaselinePos() - GetBaselinePos());
    }
    else if (m_pSecond->HasGlyph(c))
    {
        m_pSecond->GetGlyph(glyph, c);
        glyph->baselineDifference += static_cast<int16_t>(m_pSecond->GetBaselinePos() - GetBaselinePos());
    }
    else if (m_AlternateWithFirst)
    {
        m_pFirst->GetGlyph(glyph, c);
        glyph->baselineDifference += static_cast<int16_t>(m_pFirst->GetBaselinePos() - GetBaselinePos());
    }
    else
    {
        m_pSecond->GetGlyph(glyph, c);
        glyph->baselineDifference += static_cast<int16_t>(m_pSecond->GetBaselinePos() - GetBaselinePos());
    }
}

bool
PairFont::HasGlyph(uint32_t c) const
{
    return m_pFirst->HasGlyph(c) || m_pSecond->HasGlyph(c);
}

int
PairFont::GetKerning(uint32_t c0, uint32_t c1) const
{
    if (m_pFirst->HasGlyph(c0) && m_pFirst->HasGlyph(c1))
    {
        return m_pFirst->GetKerning(c0, c1);
    }

    if (m_pSecond->HasGlyph(c0) && m_pSecond->HasGlyph(c1))
    {
        return m_pSecond->GetKerning(c0, c1);
    }

    return 0;
}

/* ------------------------------------------------------------------------
        文字ストリーム
   ------------------------------------------------------------------------ */

CharacterCode
PairFont::GetCharacterCode() const
{
    // 文字列エンコーディング形式は一致していなければならない
    return m_pFirst->GetCharacterCode();
}

/* ------------------------------------------------------------------------
        テクスチャフィルタ
   ------------------------------------------------------------------------ */

void
PairFont::SetLinearFilterEnabled(
    bool    atSmall,
    bool    atLarge
)
{
    m_pFirst->SetLinearFilterEnabled(atSmall, atLarge);
    m_pSecond->SetLinearFilterEnabled(atSmall, atLarge);
}

bool
PairFont::IsLinearFilterEnabledAtSmall() const
{
    // 1 つ目のフォントの値を返します(値は一致していなければならない)。
    return m_pFirst->IsLinearFilterEnabledAtSmall();
}

bool
PairFont::IsLinearFilterEnabledAtLarge() const
{
    // 1 つ目のフォント値を返します(値は一致していなければならない)。
    return m_pFirst->IsLinearFilterEnabledAtLarge();
}

uint32_t
PairFont::GetTextureWrapFilterValue() const
{
    // 1 つ目のフォント値を返します(値は一致していなければならない)。
    return m_pFirst->GetTextureWrapFilterValue();
}

/* ------------------------------------------------------------------------
        表示設定
   ------------------------------------------------------------------------ */
bool
PairFont::IsColorBlackWhiteInterpolationEnabled() const
{
    // 1 つ目のフォント値を返します(値は一致していなければならない)。
    return m_pFirst->IsColorBlackWhiteInterpolationEnabled();
}

void
PairFont::SetColorBlackWhiteInterpolationEnabled(bool flag)
{
    m_pFirst->SetColorBlackWhiteInterpolationEnabled(flag);
    m_pSecond->SetColorBlackWhiteInterpolationEnabled(flag);
}

bool
PairFont::IsBorderEffectEnabled() const
{
    // 1 つ目のフォント値を返します(値は一致していなければならない)。
    return m_pFirst->IsBorderEffectEnabled();
}



}   // namespace font
}   // namespace nn
