﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>

#include <nn/nn_SdkLog.h>
#include <nn/font/font_GlyphTreeMap.h>
#include <nn/font/detail/font_Log.h>

namespace nn
{
namespace font
{

uint8_t GlyphNode::CalculateLineKind(uint16_t height)
{
    if (height > 1024)
    {
        NN_SDK_ASSERT(false, "The font height(%d) is too large.", height);
        return 0;
    }
    else if (height > 512)
    {
        return 10;
    }
    else if (height > 256)
    {
        return 9;
    }
    else if (height > 128)
    {
        return 8;
    }
    else if (height > 64)
    {
        return 7;
    }
    else if (height > 32)
    {
        return 6;
    }
    else if (height > 16)
    {
        return 5;
    }
    else
    {
        return 4;
    }
}

uint32_t GlyphNode::CalculateLineHeight(uint8_t kind)
{
    return 1 << kind;
}

GlyphTreeMap::GlyphTreeMap()
: m_pRootNode(NULL)
, m_NodeCountMax(0)
{
}

void GlyphTreeMap::Initialize(nn::AlignedAllocateFunctionWithUserData pAllocateFunction, void* pUserDataForAllocateFunction, uint32_t nodeNumMax)
{
    m_pRootNode = NULL;
    m_NodeCountMax = nodeNumMax;
    m_FreeList.Initialize(pAllocateFunction(sizeof(GlyphNode) * nodeNumMax, sizeof(void*), pUserDataForAllocateFunction), sizeof(GlyphNode), nodeNumMax);
}

void GlyphTreeMap::Finalize(nn::FreeFunctionWithUserData pFreeFunction, void* pUserDataForFreeFunction)
{
    pFreeFunction(m_FreeList.GetWork(), pUserDataForFreeFunction);
    m_FreeList.Cleanup();
}

GlyphNode* GlyphTreeMap::Find(uint32_t code, uint16_t size, uint16_t face) const
{
    GlyphNode::KeyType key;
    key.detail.code = code;
    key.detail.fontSize = size;
    key.detail.fontFace = face;
    return Find(m_pRootNode, key.raw);
}

GlyphNode* GlyphTreeMap::Insert(uint32_t code, uint16_t size, uint16_t face)
{
    if (m_FreeList.IsEmpty())
    {
        return NULL;
    }
    else
    {
        GlyphNode* pNode = new(m_FreeList.Get()) GlyphNode(code, size, face);
        m_pRootNode = Insert(m_pRootNode, pNode);
        m_pRootNode->m_Color = GlyphNode::Black;
        return pNode;
    }
}

void GlyphTreeMap::Erase(uint32_t code, uint16_t size, uint16_t face)
{
    GlyphNode::KeyType key;
    key.detail.code = code;
    key.detail.fontSize = size;
    key.detail.fontFace = face;
    m_pRootNode = Erase(m_pRootNode, key.raw);
    if (m_pRootNode)
    {
        m_pRootNode->m_Color = GlyphNode::Black;
    }
}

GlyphNode* GlyphTreeMap::Find(GlyphNode* pNode, uint64_t keyRaw) const
{
    while (pNode)
    {
        if (keyRaw < pNode->m_Key.raw) {
            pNode = pNode->m_pLeftNode;
        } else if (keyRaw > pNode->m_Key.raw) {
            pNode = pNode->m_pRightNode;
        } else {
            return pNode;
        }
    }
    return NULL;
}

GlyphNode* GlyphTreeMap::Insert(GlyphNode* pOrigin, GlyphNode* pNode)
{
    if (pOrigin == NULL) {
        pNode->m_Color = GlyphNode::Red;
        pNode->m_pLeftNode = NULL;
        pNode->m_pRightNode = NULL;
        return pNode;
    }
    {
        if (pNode->m_Key.raw < pOrigin->m_Key.raw) {
            pOrigin->m_pLeftNode = Insert(pOrigin->m_pLeftNode, pNode);
        } else if (pNode->m_Key.raw > pOrigin->m_Key.raw) {
            pOrigin->m_pRightNode = Insert(pOrigin->m_pRightNode, pNode);
        } else {
            // 置き換わり
            if (pOrigin != pNode) {
                // 全く同じオブジェクトの場合はこの処理は行う必要がない
                pNode->m_pRightNode = pOrigin->m_pRightNode;
                pNode->m_pLeftNode = pOrigin->m_pLeftNode;
                pNode->m_Color = pOrigin->m_Color;
                m_FreeList.Put(pOrigin);
            }
            pOrigin = pNode;
        }
        if (IsRed(pOrigin->m_pRightNode) && ! IsRed(pOrigin->m_pLeftNode)) pOrigin = RotateLeft(pOrigin);
        if (IsRed(pOrigin->m_pLeftNode) && IsRed(pOrigin->m_pLeftNode->m_pLeftNode)) pOrigin = RotateRight(pOrigin);
        if (IsRed(pOrigin->m_pLeftNode) && IsRed(pOrigin->m_pRightNode)) FlipColors(pOrigin);
    }
    return pOrigin;
}

GlyphNode* GlyphTreeMap::Erase(GlyphNode* pNode, uint64_t key_raw)
{
    if (key_raw < pNode->m_Key.raw) {
        if ( ! IsRed(pNode->m_pLeftNode) && ! IsRed(pNode->m_pLeftNode->m_pLeftNode)) {
            pNode = MoveRedLeft(pNode);
        }
        pNode->m_pLeftNode = Erase(pNode->m_pLeftNode, key_raw);
    } else {
        if (IsRed(pNode->m_pLeftNode)) {
            pNode = RotateRight(pNode);
        }
        if (key_raw == pNode->m_Key.raw && (pNode->m_pRightNode == NULL)) {
            m_FreeList.Put(pNode);
            return NULL;
        }
        if ( ! IsRed(pNode->m_pRightNode) && ! IsRed(pNode->m_pRightNode->m_pLeftNode)) {
            pNode = MoveRedRight(pNode);
        }
        if (key_raw == pNode->m_Key.raw) {
            GlyphNode* pNewNode = Find(pNode->m_pRightNode, GetMin(pNode->m_pRightNode)->m_Key.raw);
            pNewNode->m_pRightNode = EraseMin(pNode->m_pRightNode);
            pNewNode->m_pLeftNode = pNode->m_pLeftNode;
            pNewNode->m_Color = pNode->m_Color;
            m_FreeList.Put(pNode);
            pNode = pNewNode;
        } else {
            pNode->m_pRightNode = Erase(pNode->m_pRightNode, key_raw);
        }
    }
    return FixUp(pNode);
}

void GlyphTreeMap::Dump() const
{
#if !defined(NN_SDK_BUILD_RELEASE)
    uint32_t nodeNum = 0;
    if (m_pRootNode)
    {
        nodeNum = DumpRecursive(m_pRootNode, 0);
    }
    NN_DETAIL_FONT_INFO("dump %d nodes.\n", nodeNum);
#endif
}

void GlyphTreeMap::Reset()
{
    m_pRootNode = NULL;
    void* pWork = m_FreeList.GetWork();
    m_FreeList.Initialize(pWork, sizeof(GlyphNode), m_NodeCountMax);
}

void GlyphTreeMap::UpdateFlagsForCompleteTextureCacheRecursive(GlyphNode* pNode)
{
    uint32_t flag = pNode->m_Flag.storage;
    uint32_t isKeep = static_cast<uint32_t>((flag & (GlyphNode::FlagMask_Requested | GlyphNode::FlagMask_Used)) != 0);
    pNode->m_Flag.storage = static_cast<uint8_t>((isKeep << GlyphNode::FlagBit_Kept) | (flag & (GlyphNode::FlagMask_SystemReserved | GlyphNode::FlagMask_NotInFont)));
    if (pNode->m_pLeftNode)
    {
        UpdateFlagsForCompleteTextureCacheRecursive(pNode->m_pLeftNode);
    }
    if (pNode->m_pRightNode)
    {
        UpdateFlagsForCompleteTextureCacheRecursive(pNode->m_pRightNode);
    }
}

void GlyphTreeMap::ClearLockGroupRecursive(GlyphNode* pNode, uint32_t groupMask)
{
    pNode->GetLockGroup().SetAllBitOff(groupMask);
    if (pNode->m_pLeftNode)
    {
        ClearLockGroupRecursive(pNode->m_pLeftNode, groupMask);
    }
    if (pNode->m_pRightNode)
    {
        ClearLockGroupRecursive(pNode->m_pRightNode, groupMask);
    }
}

uint32_t GlyphTreeMap::DumpRecursive(GlyphNode* pNode, uint32_t level)
{
    uint32_t nodeNum = 1;
    for (uint32_t i = 0; i < level; i++)
    {
        NN_DETAIL_FONT_INFO("  ");
    }
    NN_DETAIL_FONT_INFO("%d 0x%x %d %s%s%s%s%s%s%s", pNode->GetFace(), pNode->GetCode(), pNode->GetSize(),
        pNode->IsRequested() ? "r" : "",
        pNode->IsPlotting() ? "p" : "",
        pNode->IsUsed() ? "u" : "",
        pNode->IsErased() ? "e" : "",
        pNode->IsKeeped() ? "k" : "",
        pNode->IsSystemReserved() ? "s" : "",
        pNode->IsNotInFont() ? "n" : ""
    );
    if (pNode->GetLockGroup().IsAnyBitOn(std::numeric_limits<uint32_t>::max()) != 0)
    {
        NN_DETAIL_FONT_INFO(" lock:0x%x\n", pNode->GetLockGroup().storage);
    }
    else
    {
        NN_DETAIL_FONT_INFO("\n");
    }
    if (pNode->m_pLeftNode)
    {
        nodeNum += DumpRecursive(pNode->m_pLeftNode, level + 1);
    }
    if (pNode->m_pRightNode)
    {
        nodeNum += DumpRecursive(pNode->m_pRightNode, level + 1);
    }
    return nodeNum;
}

} // namespace nn::font
} // namespace nn
