﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/font/font_ExtendedTagProcessorBase.h>
#include <nn/font/font_TextWriterBase.h>

namespace nn {
namespace font {

namespace detail {

uint32_t AcquireNextChar(const char** position)
{
    return ConvertCharUtf8ToUtf32AndIncrement(position);
}

uint32_t AcquireNextChar(const uint16_t** position)
{
    uint32_t code = **position;
    (*position)++;
    return code;
}

} // namespace nn::font::detail

template <typename CharType>
const CharType* ExtendedTagProcessorBase<CharType>::AcquireNextPrintableChar(bool* pIsPrintable, const CharType* position)
{
    const CharType* positionTop = position;
    uint32_t code = detail::AcquireNextChar(&position);
    if (code != ShiftIn && code != ShiftOut)
    {
        *pIsPrintable = true;
        return position;
    }
    const TagInfo* pTagInfo;
    AnalyzeTagHeader(&pTagInfo, positionTop);
    if (pTagInfo == NULL)
    {
        NN_SDK_ASSERT(false, "Failed to read extended tags.");
        return position;
    }
    switch (pTagInfo->tagId)
    {
        case TagIdRuby:
            // ルビタグの場合はルビ部分の位置から描画可能文字が始まる
            position = reinterpret_cast<const TagInfoRuby*>(positionTop)->rubyText;
            break;
        default:
            // 何もしない
            break;
    }
    *pIsPrintable = false;
    return position;
}

template <typename CharType>
typename TagProcessorBase<CharType>::Operation ExtendedTagProcessorBase<CharType>::ProcessTag(
    uint32_t code,
    typename TagProcessorBase<CharType>::ContextType* pContext,
    nn::font::Rectangle* pRect)
{
    if (code != ShiftIn && code != ShiftOut)
    {
        if (pRect == NULL)
        {
            return TagProcessorBase<CharType>::Process(code, pContext);
        }
        else
        {
            return TagProcessorBase<CharType>::CalculateRect(pRect, pContext, code);
        }
    }

    const TagInfo* pTagInfo;
    const CharType* pNextStr = AnalyzeTagHeader(&pTagInfo, pContext->str - 1);
    if (pTagInfo == NULL)
    {
        NN_SDK_ASSERT(false, "Failed to read extended tags.");
        return TagProcessorBase<CharType>::Operation_EndDraw;
    }
    else if (pNextStr > pContext->strEnd)
    {
        NN_SDK_ASSERT(false, "The string ended in the middle of the tags.");
        return TagProcessorBase<CharType>::Operation_EndDraw;
    }

    typename TagProcessorBase<CharType>::Operation operation = TagProcessorBase<CharType>::Operation_Default;
    if (pTagInfo->tagGroup == TagSystem)
    {
        switch (pTagInfo->tagId)
        {
            case TagIdRuby:
                operation = ProcessTagRuby(pTagInfo, pContext, pRect, pNextStr);
                break;
            default:
                // 何もしない
                break;
        }
    }
    return operation;
}

template <typename CharType>
const CharType* ExtendedTagProcessorBase<CharType>::AnalyzeTagHeader(
    const TagInfo** pTagInfo,
    const CharType* pText)
{
    if (*pText == ShiftIn)
    {
        *pTagInfo = reinterpret_cast<const TagInfo*>(pText);
        size_t tagSize = static_cast<size_t>(reinterpret_cast<const TagInfoShiftIn*>(*pTagInfo)->paramSize);
        return reinterpret_cast<const CharType*>(nn::util::ConstBytePtr(pText, sizeof(TagInfoShiftIn) + tagSize).Get());
    }
    else if (*pText == ShiftOut)
    {
        *pTagInfo = reinterpret_cast<const TagInfo*>(pText);
        return reinterpret_cast<const CharType*>(nn::util::ConstBytePtr(pText, sizeof(TagInfoShiftOut)).Get());
    }
    else
    {
        *pTagInfo = NULL;
        return pText;
    }
}

template <typename CharType>
typename TagProcessorBase<CharType>::Operation ExtendedTagProcessorBase<CharType>::ProcessTagRuby(
    const TagInfo* pTagInfo,
    typename TagProcessorBase<CharType>::ContextType* pContext,
    nn::font::Rectangle* pRect,
    const CharType* pNextStr)
{
    const float rubyScale = 0.4f;
    const float rubyCharSpace = 0.0f;
    const float rubyBaseLineOffset = 0.0f;
    const TagInfoRuby* pTagInfoRuby = reinterpret_cast<const TagInfoRuby*>(pTagInfo);
    pContext->str = pNextStr;

    // CalculateRect の場合はスキップ
    if (pRect != NULL)
    {
        return TagProcessorBase<CharType>::Operation_Default;
    }

    // タグの解析
    const int kanjiLength = pTagInfoRuby->kanjiSize / sizeof(CharType);
    const int rubyLength = pTagInfoRuby->rubySize / sizeof(CharType);
    const CharType* pRubyText = pTagInfoRuby->rubyText;
    const CharType* pKanjiText = pNextStr;

    // TextWriter の設定
    nn::font::TextWriterBase<CharType> rubyWriter(*pContext->writer);
    rubyWriter.SetDrawFlag(nn::font::TextWriterBase<CharType>::PositionFlag_VerticalOriginBaseLine);
    rubyWriter.SetLineSpace(0.0f);
    rubyWriter.SetCharSpace(rubyCharSpace);
    rubyWriter.SetItalicRatio(0.0f);
    rubyWriter.ResetTagProcessor();
    {
        // 幅を基準としてスケールを設定
        const float width = pContext->writer->GetFontWidth() * rubyScale;
        const float scale = width / rubyWriter.GetFont()->GetWidth();
        rubyWriter.SetScale(scale, scale);
    }

    // 漢字とルビの幅計算
    float kanjiWidth;
    float rubyWidth;
    {
        nn::font::TagProcessorBase<CharType>& originalTagProcessor = pContext->writer->GetTagProcessor();
        ExtendedTagProcessorBase<CharType> kanjiTagProcessor;
        pContext->writer->SetTagProcessor(&kanjiTagProcessor);
        kanjiWidth = pContext->writer->CalculateStringWidth(pKanjiText, kanjiLength);
        // TagProcessor を元に戻す
        pContext->writer->SetTagProcessor(&originalTagProcessor);
        rubyWidth = rubyWriter.CalculateStringWidth(pRubyText, rubyLength);
    }

    // 描画位置 X の設定
    const float widthDiff = kanjiWidth - rubyWidth;
    if (widthDiff > 0.0f)
    {
        const int charSpaceCount = rubyLength + 1;
        const float charSpace = widthDiff / charSpaceCount;
        rubyWriter.MoveCursorX(charSpace);
        rubyWriter.SetCharSpace(charSpace + rubyCharSpace);
    }
    else
    {
        rubyWriter.MoveCursorX((kanjiWidth - rubyWidth) / 2.0f);
    }
    // 斜体の分だけ X を調整
    if (pContext->writer->GetItalicRatio() != 0.0f)
    {
        rubyWriter.MoveCursorX(pContext->writer->GetItalicRatio() * pContext->writer->GetScaleX() * pContext->writer->GetFont()->GetWidth() / 2.0f);
    }

    // 描画位置 Y の設定
    rubyWriter.MoveCursorY(-pContext->writer->GetScaleY() * pContext->writer->GetFont()->GetBaselinePos() - rubyBaseLineOffset);

    // ルビの描画位置にカーニングおよび palt を反映させる
    const nn::font::Font* pBaseFont = pContext->writer->GetFont();
    if (pBaseFont != NULL)
    {
        const float kerning = static_cast<float>(pBaseFont->GetKerning(pContext->prevCode, *pKanjiText)) * pContext->writer->GetScaleX();
        if (kerning != 0.0f)
        {
            rubyWriter.MoveCursorX(kerning);
        }
    }

    // ルビ描画
    rubyWriter.Print(pRubyText, rubyLength);

    return TagProcessorBase<CharType>::Operation_Default;
}

/* ------------------------------------------------------------------------
        明示的実体化
   ------------------------------------------------------------------------ */

template class ExtendedTagProcessorBase<char>;
template class ExtendedTagProcessorBase<uint16_t>;

}   // namespace font
}   // namespace nn
