﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk2/detail/atk2_VirtualVoiceManager.h>

#include <nn/atk2/detail/atk2_RendererVoice.h>

namespace nn { namespace atk2 { namespace detail {

#if 0
NN_DEFINE_STATIC_CONSTANT( const uint32_t VirtualVoiceManager::InvalidVoiceId );
NN_DEFINE_STATIC_CONSTANT( const int VirtualVoiceManager::VirtualVoiceCount );
NN_DEFINE_STATIC_CONSTANT( const int VirtualVoiceManager::VirtualVoiceElementCount );
#endif

VirtualVoiceManager::VirtualVoiceManager() NN_NOEXCEPT
{
    // TODO: ボイス数が固定なので、変更できるようにした上でコンストラクタで初期化を呼ばないようにする
    //Initialize();
}

void VirtualVoiceManager::Initialize() NN_NOEXCEPT
{
    m_VoiceInfoTableRead = 0;
    m_VoiceInfoDirtyTable[0].Set();
    m_VoiceInfoDirtyTable[1].Set();

    for(uint32_t i = 0 ; i < VirtualVoiceElementCount; i++ )
    {
        m_VirtualVoiceAllocationTable[i] = 0xffffffff;
    }
    for(uint32_t i = 0 ; i < VirtualVoiceCount; i++ )
    {
        m_RendererVoiceTable[i] = NULL;

        m_VoiceInfoTable[0][i].voiceState = VoiceState_Stop;
        m_VoiceInfoTable[1][i].voiceState = VoiceState_Stop;
    }
}

uint32_t VirtualVoiceManager::AllocVirtualVoice() NN_NOEXCEPT
{
    uint32_t disableBit = 0u;
    uint32_t tableIndex = 0u;

    while( tableIndex < VirtualVoiceElementCount )
    {
        const uint32_t x = m_VirtualVoiceAllocationTable[tableIndex] ^ disableBit;
        if( x == 0u )
        {
            //  tableIndex からは確保できないため、次のテーブルに進みます。
            disableBit = 0u;
            tableIndex++;
            continue;
        }

        const uint32_t allocatableBit = nn::util::isols1b( x );
        const uint32_t shift = nn::util::popcount( allocatableBit - 1 );

        //  i * 32 + shift
        //  voiceId は Voice クラスで管理され、返却時に FreeVirtualVoice の引数に与えられます
        const uint32_t voiceId= ( tableIndex << 5 ) | shift;

        if( m_RendererVoiceTable[voiceId] != nullptr )
        {
            //  m_LowLevelVoiceTable[voiceId] はまだ解放されていないため、
            //  別のボイスを確保するようにします。
            disableBit |= allocatableBit;
            continue;
        }

        m_VirtualVoiceAllocationTable[tableIndex] ^= allocatableBit;
        return voiceId;
    }
    return InvalidVoiceId;
}

void VirtualVoiceManager::FreeVirtualVoice(uint32_t voiceId) NN_NOEXCEPT
{
    uint32_t i = (voiceId >> 5); // i = voiceId / 32;
    uint32_t shift = (voiceId & 0x1f);
    m_VirtualVoiceAllocationTable[i] |= (1<<shift);
}

void VirtualVoiceManager::SetRendererVoice( uint32_t voiceId, RendererVoice* rendererVoice ) NN_NOEXCEPT
{
    m_RendererVoiceTable[voiceId] = rendererVoice;
}

RendererVoice* VirtualVoiceManager::GetRendererVoice( uint32_t voiceId ) NN_NOEXCEPT
{
    return m_RendererVoiceTable[voiceId];
}

VoiceInfo* VirtualVoiceManager::GetVoiceInfo( uint32_t voiceId ) NN_NOEXCEPT
{
    return &m_VoiceInfoTable[m_VoiceInfoTableRead][voiceId];
}

const VoiceInfo* VirtualVoiceManager::GetVoiceInfo( uint32_t voiceId ) const NN_NOEXCEPT
{
    return &m_VoiceInfoTable[m_VoiceInfoTableRead][voiceId];
}

bool VirtualVoiceManager::Update() NN_NOEXCEPT
{
    UpdateVoiceInfo();
    return true;
}

void VirtualVoiceManager::UpdateVoiceInfo() NN_NOEXCEPT
{
    uint32_t writeIndex = m_VoiceInfoTableRead ? 0 : 1;

    for(uint32_t i = 0 ; i < VirtualVoiceCount; i++ )
    {
        VoiceInfo* pVoiceInfo = &m_VoiceInfoTable[writeIndex][i];

        if (m_RendererVoiceTable[i] != nullptr)
        {
            m_RendererVoiceTable[i]->GetVoiceInfo( pVoiceInfo );
            m_VoiceInfoDirtyTable[writeIndex].Set(i);
        }
        else
        {
            if (m_VoiceInfoDirtyTable[writeIndex].Test(i))
            {
                pVoiceInfo->voiceState = VoiceState_Stop;
                pVoiceInfo->waveBufferState = WaveBufferState_Free;
                pVoiceInfo->pVoice = nullptr;
                m_VoiceInfoDirtyTable[writeIndex].Reset(i);
            }
        }
    }

    m_VoiceInfoTableRead = writeIndex;
}

int VirtualVoiceManager::GetAllocatedVirtualVoiceCount() const NN_NOEXCEPT
{
    int allocatedCount = 0;

    for (int i = 0; i < VirtualVoiceElementCount; i++)
    {
        // AllocationTable は未確保ボイスのビットがオンになっているため反転する
        uint32_t allocatedFlag = ~m_VirtualVoiceAllocationTable[i];

        allocatedCount += nn::util::popcount( allocatedFlag );
    }

    return allocatedCount;
}

int VirtualVoiceManager::GetUnreleasedRendererVoiceCount() const
{
    int count = 0;
    for (int i = 0; i < VirtualVoiceCount; i++)
    {
        if (m_RendererVoiceTable[i] != nullptr)
        {
            count++;
        }
    }

    return count;
}

}}}
