﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk2/detail/atk2_RendererVoiceManager.h>

#include <utility>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_BytePtr.h>

#include <nn/audio/audio_Adpcm.h>
#include <nn/audio/audio_Common.h>
#include <nn/atk2/atk2_AudioEngine.h>
#include <nn/atk2/detail/atk2_RendererManager.h>

namespace nn { namespace atk2 { namespace detail {

    NN_DEFINE_STATIC_CONSTANT( const int RendererVoiceManager::UnassignedIndex );

    void RendererVoiceManager::GetDefaultInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        NN_UNUSED(arg);
        NN_UNUSED(platformArg);
    }

    size_t RendererVoiceManager::GetRequiredBufferSize(const InitArg& arg, const PlatformInitArg& platformArg) const NN_NOEXCEPT
    {
        NN_UNUSED(platformArg);
        NN_SDK_ASSERT_GREATER_EQUAL( arg.voiceCount, 0 );
        size_t result = 0;
        result += GetRequiredBufferSizeForVoicePtrTable(arg.voiceCount);
        result += GetRequiredBufferSizeForAssignedTableIndex(arg.voiceCount);
        result += GetRequiredBufferSizeForVoiceArray(arg.voiceCount);
        return result;
    }

    RendererVoiceManager::Result RendererVoiceManager::Initialize(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL( arg.voiceCount, 0 );
        NN_SDK_ASSERT_GREATER_EQUAL( arg.workBufferSize, GetRequiredPlatformBufferSize( arg, platformArg ) );
        NN_SDK_ASSERT_GREATER_EQUAL( platformArg.workBufferSizeForMemoryPool, GetRequiredBufferSizeForMemoryPool( arg, platformArg ) );

        if (arg.workBufferSize < GetRequiredBufferSize(arg, platformArg) ||
            platformArg.workBufferSizeForMemoryPool < GetRequiredBufferSizeForMemoryPool(arg, platformArg))
        {
            NN_SDK_LOG("Insufficient buffer size.\n" );
            return Result_InsufficientBufferSize;
        }

        // メンバのインスタンスの確保
        nn::util::BytePtr workBuffer( arg.workBuffer );
        m_ppVoiceTable = workBuffer.Get<RendererVoice*>();
        workBuffer.Advance( GetRequiredBufferSizeForVoicePtrTable(arg.voiceCount) );
        m_pAssignedTableIndex = workBuffer.Get<int>();
        workBuffer.Advance( GetRequiredBufferSizeForAssignedTableIndex(arg.voiceCount) );
        m_pVoiceArray = workBuffer.Get();

        nn::util::BytePtr workBufferForMemoryPool( platformArg.workBufferForMemoryPool );
        m_pAdpcmContextArray = workBufferForMemoryPool.Get();

        for (int i = 0; i < arg.voiceCount; ++i)
        {
            m_ppVoiceTable[i] = new(workBuffer.Get<RendererVoice>()) RendererVoice();
            m_ppVoiceTable[i]->SetAudioEngine(arg._pAudioEngine);
            workBuffer.Advance( sizeof(RendererVoice) );
            m_ppVoiceTable[i]->SetAdpcmParameterBuffer(workBufferForMemoryPool.Get());
            workBufferForMemoryPool.Advance(nn::util::align_up(sizeof(AdpcmContext), RendererManager::BufferAlignSize));
        }

        m_UsingVoiceCount = 0;
        m_VoiceCount = arg.voiceCount;
        for (int i = 0; i < m_VoiceCount; ++i)
        {
            m_pAssignedTableIndex[i] = UnassignedIndex;
        }
        m_pAudioEngine = arg._pAudioEngine;

        return Result_Success;
    }

    void RendererVoiceManager::Finalize() NN_NOEXCEPT
    {
    }

    bool RendererVoiceManager::Update() NN_NOEXCEPT
    {
        Lock();
        for (int i = 0; i < m_UsingVoiceCount; ++i)
        {
            m_ppVoiceTable[i]->Update();
        }

        // TODO: RendererVoice の更新とどちらが先？
        if (!m_pAudioEngine->GetVoiceCommandManager().ProcessCommand())
        {
            // false でもメッセージキューに積まれていないだけで、処理に失敗しているわけではない
            //return Result_FailedToProcessVoiceCommand;
        }
        if (!m_pAudioEngine->GetVoiceReplyCommandManager().ProcessCommand())
        {
            //return Result_FailedToProcessVoiceReplyCommand;
        }
        Unlock();

        return true;
    }

    void RendererVoiceManager::Lock() NN_NOEXCEPT
    {
        m_UpdateCriticalSection.Lock();
    }

    void RendererVoiceManager::Unlock() NN_NOEXCEPT
    {
        m_UpdateCriticalSection.Unlock();
    }

    RendererVoice* RendererVoiceManager::AllocVoice() NN_NOEXCEPT
    {
        if (m_UsingVoiceCount < m_VoiceCount)
        {
            const int assignTableIndex = m_UsingVoiceCount;
            ++m_UsingVoiceCount;

            //  voiceArrayIndex は pVoice が m_VoiceArray の何番目にあるかを表します。
            //  つまり、&m_VoiceArray[voiceArrayIndex] == pVoice を満たします。
            RendererVoice* pVoice = m_ppVoiceTable[assignTableIndex];
            const ptrdiff_t voiceArrayIndex = GetVoiceArrayIndex( pVoice );
            NN_SDK_ASSERT_RANGE(voiceArrayIndex, 0, m_VoiceCount);
            m_pAssignedTableIndex[voiceArrayIndex] = assignTableIndex;

            RendererVoice::Result result = pVoice->Initialize();
            if (result != RendererVoice::Result_Success)
            {
                NN_SDK_LOG("Failed to initialize voice.\n");
            }
            if (!pVoice->AcquireVoiceSlot())
            {
                NN_SDK_LOG("Failed to acquire voice slot.\n");
            }
            return pVoice;
        }

        return nullptr;
    }

    void RendererVoiceManager::FreeVoice(RendererVoice* pVoice) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pVoice);
        pVoice->ReleaseVoiceSlot();
        pVoice->Finalize();

        //  voiceArrayIndex は pVoice が m_VoiceArray の何番目にあるかを表します。
        const ptrdiff_t voiceArrayIndex = GetVoiceArrayIndex( pVoice );
        NN_SDK_ASSERT_RANGE(voiceArrayIndex, 0, m_VoiceCount);

        //  voiceTableIndex は pVoice が m_pVoiceTable の何番目に割り当てられているかを表します。
        const int voiceTableIndex = m_pAssignedTableIndex[voiceArrayIndex];
        NN_SDK_ASSERT_NOT_EQUAL(voiceTableIndex, UnassignedIndex);  //  未割り当ての確認

        //  使用中のボイスのうち、m_pVoiceTable の最後のボイスと pVoice を交換します。
        const int lastTableIndex = m_UsingVoiceCount - 1;
        --m_UsingVoiceCount;
        NN_SDK_ASSERT_GREATER_EQUAL(m_UsingVoiceCount, 0);

        const ptrdiff_t lastArrayIndex = GetVoiceArrayIndex( m_ppVoiceTable[lastTableIndex] );
        NN_SDK_ASSERT_RANGE(lastArrayIndex , 0, m_VoiceCount);
        m_ppVoiceTable[voiceTableIndex] = m_ppVoiceTable[lastTableIndex];
        m_ppVoiceTable[lastTableIndex] = pVoice;
        m_pAssignedTableIndex[lastArrayIndex] = voiceTableIndex;
        m_pAssignedTableIndex[voiceArrayIndex] = UnassignedIndex;
    }

    int RendererVoiceManager::GetVoiceCount() const NN_NOEXCEPT
    {
        return m_VoiceCount;
    }

    int RendererVoiceManager::GetUsingVoiceCount() const NN_NOEXCEPT
    {
        return m_UsingVoiceCount;
    }

    size_t RendererVoiceManager::GetRequiredPlatformBufferSize(const InitArg& arg, const PlatformInitArg& platformArg) const NN_NOEXCEPT
    {
        NN_UNUSED(arg);
        NN_UNUSED(platformArg);
        size_t result = 0;
        return result;
    }

    size_t RendererVoiceManager::GetRequiredBufferSizeForMemoryPool(const InitArg& arg, const PlatformInitArg& platformArg) const NN_NOEXCEPT
    {
        NN_UNUSED(platformArg);
        NN_SDK_ASSERT_GREATER_EQUAL( arg.voiceCount, 0 );
        size_t result = 0;
        result += GetRequiredBufferSizeForAdpcmContextArray(arg.voiceCount);
        return result;
    }

    // private

    size_t RendererVoiceManager::GetRequiredBufferSizeForVoicePtrTable(int voiceCount) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL( voiceCount, 0 );
        return sizeof(RendererVoice*) * voiceCount;
    }

    size_t RendererVoiceManager::GetRequiredBufferSizeForAssignedTableIndex(int voiceCount) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL( voiceCount, 0 );
        return sizeof(int) * voiceCount;
    }

    ptrdiff_t RendererVoiceManager::GetVoiceArrayIndex(RendererVoice* pVoice) NN_NOEXCEPT
    {
        return ( reinterpret_cast<char*>( pVoice ) - reinterpret_cast<char*>( m_pVoiceArray ) ) / sizeof(RendererVoice);
    }

    size_t RendererVoiceManager::GetRequiredBufferSizeForVoiceArray(int voiceCount) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL( voiceCount, 0 );
        return sizeof(RendererVoice) * voiceCount;
    }

    size_t RendererVoiceManager::GetRequiredBufferSizeForAdpcmContextArray(int voiceCount) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL( voiceCount, 0 );
        size_t alignedAdpcmContextSize = nn::util::align_up(sizeof(AdpcmContext), RendererManager::BufferAlignSize);
        return alignedAdpcmContextSize * voiceCount;
    }
}}}
