﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk2/detail/atk2_RendererVoice.h>

#include <nn/nn_SdkLog.h>
#include <nn/atk2/atk2_AudioEngine.h>
#include <nn/atk2/atk2_SubMix.h>

namespace {

    nn::audio::SampleFormat ConvertSampleFormatToVoiceSampleFormat(nn::atk2::detail::SampleFormat sampleFormat) NN_NOEXCEPT
    {
        switch (sampleFormat)
        {
        case nn::atk2::detail::SampleFormat_PcmS8:
            return nn::audio::SampleFormat_PcmInt8;
        case nn::atk2::detail::SampleFormat_PcmS16:
            return nn::audio::SampleFormat_PcmInt16;
        case nn::atk2::detail::SampleFormat_Adpcm:
            return nn::audio::SampleFormat_Adpcm;
        default:
            NN_SDK_LOG("Unsupported sample format.\n");
            return nn::audio::SampleFormat_PcmInt16;
        }
    }

    nn::audio::VoiceType::PlayState ConvertVoiceStateToVoicePlayState(nn::atk2::detail::VoiceState state) NN_NOEXCEPT
    {
        switch (state)
        {
        case nn::atk2::detail::VoiceState_Play:
            return nn::audio::VoiceType::PlayState_Play;
        case nn::atk2::detail::VoiceState_Stop:
            return nn::audio::VoiceType::PlayState_Stop;
        case nn::atk2::detail::VoiceState_Pause:
            return nn::audio::VoiceType::PlayState_Pause;
        default:
            NN_SDK_LOG("Unsupported sample format.\n");
            return nn::audio::VoiceType::PlayState_Play;
        }
    }

}

namespace nn { namespace atk2 { namespace detail {

    RendererVoice::RendererVoice() NN_NOEXCEPT
        : m_SampleRate(48000)
        , m_SampleFormat(SampleFormat_PcmS16)
        , m_pAdpcmParameter(nullptr)
        //, m_Volume(1.0f)
        , m_ChannelCount(1)
        , m_Priority(0)
        , m_VoiceType()
        //, m_NodeId()
        , m_VoiceState(VoiceState_Stop)
        , m_PlayPosition(0)
        , m_WaveBufferList()
        , m_LastAppendBuffer(nullptr)
        //, m_IsVoiceParamDirty(false)
        , m_pAudioEngine(nullptr)
        , m_pVoice(nullptr)
        , m_IsAvailable(false)
        , m_IsNeedUpdateRenderer(false)
    {
    }

    RendererVoice::Result RendererVoice::Initialize() NN_NOEXCEPT
    {
        m_IsAvailable = true;
        m_IsNeedUpdateRenderer = false;
        return Result_Success;
    }

    void RendererVoice::Finalize() NN_NOEXCEPT
    {
        ClearWaveBufferList();
        m_IsAvailable = false;
        m_pVoice = nullptr;
        m_IsNeedUpdateRenderer = false;
    }

    RendererVoice::Result RendererVoice::Update() NN_NOEXCEPT
    {
        Result result = Result_Success;
        bool playWaveBufferFlag = false;
        UpdateVoicePlayState(&playWaveBufferFlag);
        if (m_VoiceState != VoiceState_Stop)
        {
            UpdateWaveBuffer(playWaveBufferFlag);
        }
        return result;
    }

    RendererVoice::Result RendererVoice::AppendWaveBufferList(WaveBuffer* pWaveBuffer) NN_NOEXCEPT
    {
        m_WaveBufferList.PushBack(pWaveBuffer);

        return Result_Success;
    }

    RendererVoice::Result RendererVoice::ClearWaveBufferList() NN_NOEXCEPT
    {
        m_WaveBufferList.Clear();
        m_LastAppendBuffer = nullptr;
        return Result_Success;
    }

    void RendererVoice::SetAudioEngine(AudioEngine* pAudioEngine) NN_NOEXCEPT
    {
        m_pAudioEngine = pAudioEngine;
    }

    void RendererVoice::SetAdpcmParameterBuffer(void* buffer) NN_NOEXCEPT
    {
        m_pAdpcmParameter = reinterpret_cast<nn::audio::AdpcmParameter*>(buffer);
    }

    void RendererVoice::GetVoiceInfo(VoiceInfo* pOutVoiceInfo) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutVoiceInfo);
        pOutVoiceInfo->pVoice = m_pVoice;
        pOutVoiceInfo->voiceState = m_VoiceState;
        pOutVoiceInfo->playPosition = m_PlayPosition;
        volatile WaveBuffer* waveBufferListBegin = m_WaveBufferList.PeekFront();
        if (waveBufferListBegin != nullptr)
        {
            pOutVoiceInfo->waveBufferState = waveBufferListBegin->state;
            pOutVoiceInfo->waveBufferTag = waveBufferListBegin->userParam;
        }
        else
        {
            pOutVoiceInfo->waveBufferState = WaveBufferState_Free;
        }
    }

    void RendererVoice::SetVoice(Voice* pVoice) NN_NOEXCEPT
    {
        m_pVoice = pVoice;
    }

    void RendererVoice::SetSampleRate(int sampeRate) NN_NOEXCEPT
    {
        m_SampleRate = sampeRate;
    }

    void RendererVoice::SetSampleFormat(SampleFormat sampeFormat) NN_NOEXCEPT
    {
        m_SampleFormat = sampeFormat;
    }

    void RendererVoice::SetAdpcmParam(AdpcmParam* pAdpcmParam) NN_NOEXCEPT
    {
        NN_UNUSED(pAdpcmParam);
        //m_pAdpcmParameter = pAdpcmParam;
#if 0
        for (int i = 0; i < 8; i++)
        {
            for (int j = 0; j < 2; j++)
            {
                m_pAdpcmParameter->coefficients[i + j * 8] = pAdpcmParam->coef[i][j];
            }
        }
#endif
    }

    void RendererVoice::SetOutputReceiver(OutputReceiver* pOutputReceiver) NN_NOEXCEPT
    {
        m_pOutputReceiver = pOutputReceiver;
        SetVoiceDestination();
    }

    bool RendererVoice::AcquireVoiceSlot() NN_NOEXCEPT
    {
        nn::audio::SampleFormat sampleFormat = ConvertSampleFormatToVoiceSampleFormat(m_SampleFormat);
        size_t parameterSize = (sampleFormat == nn::audio::SampleFormat_Adpcm) ? sizeof(nn::audio::AdpcmParameter) : 0;
        void* pParameter = (sampleFormat == nn::audio::SampleFormat_Adpcm) ? m_pAdpcmParameter : nullptr;
        NN_SDK_LOG("[atk2]AcquireVoiceSlot 0x%08x, %08d, %02d, %02d, %02d, 0x%08x, %04zd\n"
            , &m_VoiceType, m_SampleRate, m_ChannelCount, sampleFormat, m_Priority, pParameter, parameterSize);
        m_IsNeedUpdateRenderer = true;
        if (nn::audio::AcquireVoiceSlot(&m_pAudioEngine->GetRendererManager().GetAudioRendererConfig(), &m_VoiceType, m_SampleRate, m_ChannelCount, sampleFormat, m_Priority, pParameter, parameterSize))
        {
            return false;
        }
        m_IsAvailable = true;
        return true;
    }

    void RendererVoice::ReleaseVoiceSlot() NN_NOEXCEPT
    {
        nn::audio::ReleaseVoiceSlot(&m_pAudioEngine->GetRendererManager().GetAudioRendererConfig(), &m_VoiceType);
        m_IsAvailable = false;
        NN_SDK_LOG("[atk2]ReleaseVoiceSlot 0x%08x\n", &m_VoiceType);
        m_IsNeedUpdateRenderer = true;
    }

    void RendererVoice::SetVoiceDestination() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( m_pOutputReceiver );
        switch( m_pOutputReceiver->GetReceiverType() )
        {
            case OutputReceiver::ReceiverType::ReceiverType_SubMix:
                nn::audio::SetVoiceDestination(&m_pAudioEngine->GetRendererManager().GetAudioRendererConfig(), &m_VoiceType, reinterpret_cast<SubMix*>(m_pOutputReceiver)->GetAudioSubMixInstance());
                NN_SDK_LOG("[atk2]SetVoiceDestination 0x%08x, 0x%08x\n", &m_VoiceType, reinterpret_cast<SubMix*>(m_pOutputReceiver)->GetAudioSubMixInstance());
                break;
            case OutputReceiver::ReceiverType::ReceiverType_FinalMix:
                nn::audio::SetVoiceDestination(&m_pAudioEngine->GetRendererManager().GetAudioRendererConfig(), &m_VoiceType, reinterpret_cast<FinalMix*>(m_pOutputReceiver)->GetAudioFinalMixInstance());
                NN_SDK_LOG("[atk2]SetVoiceDestination 0x%08x, 0x%08x\n", &m_VoiceType, reinterpret_cast<FinalMix*>(m_pOutputReceiver)->GetAudioFinalMixInstance());
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
        }
        m_IsNeedUpdateRenderer = true;
    }

    void RendererVoice::SetVoicePriority(int priority) NN_NOEXCEPT
    {
        nn::audio::SetVoicePriority(&m_VoiceType, priority);
        NN_SDK_LOG("[atk2]SetVoicePriority 0x%08x, %02d\n", &m_VoiceType, priority);
        m_IsNeedUpdateRenderer = true;
    }

    void RendererVoice::SetVoicePitch(float pitch) NN_NOEXCEPT
    {
        nn::audio::SetVoicePitch(&m_VoiceType, pitch);
        NN_SDK_LOG("[atk2]SetVoicePitch 0x%08x, %04f\n", &m_VoiceType, pitch);
        m_IsNeedUpdateRenderer = true;
    }

    void RendererVoice::SetVoiceVolume(float volume) NN_NOEXCEPT
    {
        nn::audio::SetVoiceVolume(&m_VoiceType, volume);
        NN_SDK_LOG("[atk2]SetVoiceVolume 0x%08x, %04f\n", &m_VoiceType, volume);
        m_IsNeedUpdateRenderer = true;
    }

    void RendererVoice::SetVoiceMixVolume(float volume, int sourceIndex, int destinationIndex) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( m_pOutputReceiver );
        switch( m_pOutputReceiver->GetReceiverType() )
        {
            case OutputReceiver::ReceiverType::ReceiverType_SubMix:
                nn::audio::SetVoiceMixVolume(&m_VoiceType, reinterpret_cast<SubMix*>(m_pOutputReceiver)->GetAudioSubMixInstance(), volume, sourceIndex, destinationIndex);
                break;
            case OutputReceiver::ReceiverType::ReceiverType_FinalMix:
                nn::audio::SetVoiceMixVolume(&m_VoiceType, reinterpret_cast<FinalMix*>(m_pOutputReceiver)->GetAudioFinalMixInstance(), volume, sourceIndex, destinationIndex);
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
        }
        NN_SDK_LOG("[atk2]SetVoiceMixVolume 0x%08x, %04f, %02d, %02d\n", &m_VoiceType, volume, sourceIndex, destinationIndex);
        m_IsNeedUpdateRenderer = true;
    }

    void RendererVoice::SetVoiceBiquadFilterParameter(int filterIndex) NN_NOEXCEPT
    {
        nn::audio::BiquadFilterParameter parameter; // TODO: BiquadFilterParameter をサポート
        nn::audio::SetVoiceBiquadFilterParameter(&m_VoiceType, filterIndex, parameter);
        NN_SDK_LOG("[atk2]SetVoiceBiquadFilterParameter 0x%08x, %04f, %02d, %02d\n", &m_VoiceType, filterIndex);
        m_IsNeedUpdateRenderer = true;
    }

    void RendererVoice::SetVoiceState(VoiceState state) NN_NOEXCEPT
    {
        m_VoiceState = state;
    }

    uint32_t RendererVoice::GetNodeId() NN_NOEXCEPT
    {
        return nn::audio::GetVoiceNodeId(&m_VoiceType);
    }

    bool RendererVoice::IsAvailable() const NN_NOEXCEPT
    {
        return m_IsAvailable;
    }

    void RendererVoice::SetAvailable(bool isAvailable) NN_NOEXCEPT
    {
        m_IsAvailable = isAvailable;
    }

    // private

    void RendererVoice::UpdateVoicePlayState(bool* pOutPlayWaveBufferFlag) NN_NOEXCEPT
    {
        nn::audio::VoiceType::PlayState currentVoicePlayState = nn::audio::GetVoicePlayState(&m_VoiceType);
        nn::audio::VoiceType::PlayState targetVoicePlayState = ConvertVoiceStateToVoicePlayState(m_VoiceState);

        if (currentVoicePlayState != targetVoicePlayState)
        {
            SetVoicePlayState(targetVoicePlayState);
        }

        if (m_VoiceState == VoiceState_Play)
        {
            if (currentVoicePlayState == nn::audio::VoiceType::PlayState_Play)
            {
                UpdatePlayPosition();
            }
            else if (currentVoicePlayState == nn::audio::VoiceType::PlayState_Stop)
            {
                *pOutPlayWaveBufferFlag = true;
            }
        }
    }

    void RendererVoice::UpdateWaveBuffer(bool playWaveBufferFlag) NN_NOEXCEPT
    {
        if (playWaveBufferFlag)
        {
            WaveBuffer* currentWaveBuffer = m_WaveBufferList.PeekFront();
            AppendWaveBuffer(currentWaveBuffer);
            currentWaveBuffer->state = WaveBufferState_Play;
        }

        FreeReleasedWaveBuffer();
        AppendStockedWaveBuffer();
    }

    void RendererVoice::AppendStockedWaveBuffer() NN_NOEXCEPT
    {
        while (m_LastAppendBuffer != nullptr
            && m_LastAppendBuffer->next != nullptr
            && nn::audio::GetWaveBufferCount(&m_VoiceType) < nn::audio::VoiceType::WaveBufferCountMax)
        {
            if (AppendWaveBuffer(m_LastAppendBuffer->next) != Result_Success)
            {
                NN_SDK_LOG("Falied to append waveBuffer\n");
                break;
            }
        }
    }

    void RendererVoice::FreeReleasedWaveBuffer() NN_NOEXCEPT
    {
        while (nn::audio::GetReleasedWaveBuffer(&m_VoiceType) != nullptr)
        {
            m_WaveBufferList.PeekFront()->state = WaveBufferState_Done;
            WaveBuffer* waveBuffer = m_WaveBufferList.PopFront();
            if (waveBuffer != nullptr)
            {
                waveBuffer->state = WaveBufferState_Play;
            }
        }
    }

    void RendererVoice::UpdatePlayPosition() NN_NOEXCEPT
    {
    }

    RendererVoice::Result RendererVoice::AppendWaveBuffer(WaveBuffer* pWaveBuffer) NN_NOEXCEPT
    {
        bool isSampleFormatAdpcm = (m_SampleFormat == SampleFormat_Adpcm);

        nn::audio::WaveBuffer rendererWaveBuffer;
        rendererWaveBuffer.buffer = pWaveBuffer->bufferAddress;
        rendererWaveBuffer.size = pWaveBuffer->bufferSize;
        rendererWaveBuffer.startSampleOffset = static_cast<int>(pWaveBuffer->sampleOffset);
        rendererWaveBuffer.endSampleOffset = static_cast<int>(pWaveBuffer->sampleLength);
        rendererWaveBuffer.loop = pWaveBuffer->loopFlag;
        rendererWaveBuffer.pContext = isSampleFormatAdpcm ? pWaveBuffer->pAdpcmContext : nullptr;
        rendererWaveBuffer.contextSize = isSampleFormatAdpcm ? sizeof(nn::audio::AdpcmContext) : 0;
        rendererWaveBuffer.isEndOfStream = true;

        if (!nn::audio::AppendWaveBuffer(&m_VoiceType, &rendererWaveBuffer))
        {
            NN_SDK_LOG("Falied to append waveBuffer\n");
            return Result_FailedToAppendWaveBuffer;
        }
        NN_SDK_LOG("[atk2]AppendWaveBuffer 0x%08x, 0x%08x\n", &m_VoiceType, rendererWaveBuffer.buffer);
        m_IsNeedUpdateRenderer = true;

        m_LastAppendBuffer = pWaveBuffer;
        return Result_Success;
    }

    void RendererVoice::SetVoicePlayState(nn::audio::VoiceType::PlayState state) NN_NOEXCEPT
    {
        nn::audio::SetVoicePlayState(&m_VoiceType, state);
        NN_SDK_LOG("[atk2]SetVoicePlayState 0x%08x, %02d\n", &m_VoiceType, state);
        m_IsNeedUpdateRenderer = true;
    }

}}}
