﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/util/util_BytePtr.h>
#include <nn/nn_SdkAssert.h>

#include <nn/atk2/detail/atk2_AudioFrameUpdater.h>
#include <nn/atk2/detail/atk2_AudioThread.h>

namespace nn { namespace atk2 { namespace detail {

    void AudioThread::GetDefaultInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        NN_UNUSED(arg);
        NN_UNUSED(platformArg);
    }

    size_t AudioThread::GetRequiredBufferSize(const InitArg& arg) const NN_NOEXCEPT
    {
        size_t result = 0;
        result += sizeof(uintptr_t) * arg.messageBufferSize;
        return result;
    }

    AudioThread::Result AudioThread::Initialize(const InitArg& arg, const PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_ALIGNED(platformArg.workBuffer, nn::os::ThreadStackAlignment);

        nn::util::BytePtr buffer(arg.workBuffer);
        nn::util::BytePtr platformBuffer(platformArg.workBuffer);
        NN_SDK_LOG("[atk2Debug]AudioThread 0x%08x %zd\n", arg.workBuffer, arg.workBufferSize);
        NN_SDK_LOG("[atk2Debug]AudioThreadPlatform 0x%08x %zd\n", platformArg.workBuffer, platformArg.workBufferSize);

        // スレッドの初期化
        fnd::Thread::RunArgs threadInitArg;
        threadInitArg.stack = platformBuffer.Get();
        threadInitArg.stackSize = arg.threadStackSize;
        buffer.Advance(arg.threadStackSize);

        threadInitArg.name = "nn::atk2::AudioThread";
        threadInitArg.priority = arg.threadPriority;
        threadInitArg.idealCoreNumber = arg.threadCoreNumber;
        threadInitArg.affinityMask = static_cast<fnd::Thread::AffinityMask>(arg.threadAffinityMask);
        threadInitArg.handler = this;
        threadInitArg.param = nullptr;

        if (!m_Thread.Run(threadInitArg))
        {
            NN_SDK_LOG("Failed to run audio thread.\n");
            return Result_FailedToRunThread;
        }

        // メッセージキューの初期化
        m_MsgBuffer = reinterpret_cast<uintptr_t*>(buffer.Get());
        buffer.Advance(sizeof(uintptr_t) * arg.messageBufferSize);
        nn::os::InitializeMessageQueue(&m_BlockingQueue, m_MsgBuffer, arg.messageBufferSize);

        m_ThreadUpdateIntervalMSec = arg.threadUpdateIntervalMSec;
        m_pAudioEngine = arg._pAudioEngine;
        m_pAudioFrameUpdater = arg._pAudioFrameUpdater;
        m_IsInitialized = true;

        return Result_Success;
    }

    void AudioThread::Finalize() NN_NOEXCEPT
    {
        if (!m_IsInitialized)
        {
            return;
        }

        nn::os::JamMessageQueue(&m_BlockingQueue, Message_Shutdown);

        m_Thread.WaitForExit();
        m_Thread.Release();
    }

    // プラットフォーム固有

    size_t AudioThread::GetRequiredPlatformBufferSize(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        size_t result = 0;
        result += arg.threadStackSize;
        return result;
    }

    AudioThread::~AudioThread() NN_NOEXCEPT
    {
    }

    // private

    uint32_t AudioThread::Run(void* pParam) NN_NOEXCEPT
    {
        NN_UNUSED(pParam);

        for (;;)
        {
            uintptr_t queueData;
            Message msg = static_cast<Message>(nn::os::TryReceiveMessageQueue(&queueData, &m_BlockingQueue) ? queueData & 0xf0000000 : 0x0);
            if (msg == Message_Shutdown)
            {
                break;
            }

            if (m_pAudioFrameUpdater != nullptr)
            {
                m_pAudioFrameUpdater->Update();
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(static_cast<int64_t>(m_ThreadUpdateIntervalMSec)));
        }

        return 0;
    }
}}}
