﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk2/atk2_SubMix.h>

#include <algorithm>
#include <nn/nn_SdkAssert.h>
#include <nn/audio/audio_SubMixApi.h>
#include <nn/util/util_BytePtr.h>
#include <nn/atk2/atk2_FinalMix.h>
#include <nn/atk2/detail/atk2_RendererManager.h>

namespace nn { namespace atk2 {


SubMix::SubMix() NN_NOEXCEPT
    : m_ReferenceCount( 0 )
    , m_pReceiver( nullptr )
    , m_IsInitialized( false )
{
}
size_t SubMix::GetRequiredMemorySize(int channelCount) NN_NOEXCEPT
{
    return GetRequiredMemorySize( 1, channelCount );
}
size_t SubMix::GetRequiredMemorySize(int busCount, int channelCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX( channelCount, 1, nn::audio::MixBufferCountMax );
    NN_SDK_REQUIRES_GREATER_EQUAL( busCount, 1 );
    NN_SDK_REQUIRES_LESS_EQUAL( channelCount * busCount, nn::audio::MixBufferCountMax );
    size_t result = 0;

    //  m_ppMixVolume のバッファ
    result += channelCount * busCount * sizeof(float*);
    result += channelCount * busCount * nn::audio::MixBufferCountMax * sizeof(float);

    //  BusData のバッファ
    result += sizeof(BusData) * busCount;

    return result;
}
bool SubMix::Initialize(detail::RendererManager* pRendererManager, int channelCount, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    return Initialize( pRendererManager, 1, channelCount, buffer, bufferSize );
}
bool SubMix::Initialize(detail::RendererManager* pRendererManager, int busCount, int channelCount, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( !m_IsInitialized );
    NN_SDK_REQUIRES_MINMAX( channelCount, 1, nn::audio::MixBufferCountMax );
    NN_SDK_REQUIRES_GREATER_EQUAL( busCount, 1 );
    NN_SDK_REQUIRES_LESS_EQUAL( channelCount * busCount, nn::audio::MixBufferCountMax );
    NN_SDK_REQUIRES_NOT_NULL( buffer );
    NN_SDK_REQUIRES_GREATER_EQUAL( bufferSize, GetRequiredMemorySize( busCount, channelCount ) );

    NN_UNUSED( bufferSize );

    nn::util::BytePtr ptr( buffer );

    m_ChannelCount = channelCount;
    m_BusCount = busCount;
    m_IsUnusedEffectChannelMuted = false;

    m_ppMixVolume = ptr.Get<float*>();
    ptr.Advance( m_ChannelCount * m_BusCount * sizeof(float*) );

    for(int srcIdx = 0 ; srcIdx < GetMixBufferCount(); srcIdx++)
    {
        // dst の MixBuffer の数は現段階では不明なので最大数である nn::audio::MixBufferCountMax だけ確保します
        m_ppMixVolume[srcIdx] = ptr.Get<float>();
        ptr.Advance( nn::audio::MixBufferCountMax * sizeof(float) );

        for(int dstIdx = 0; dstIdx < nn::audio::MixBufferCountMax; dstIdx++)
        {
            m_ppMixVolume[srcIdx][dstIdx] = 1.0f;
        }
    }

    m_pBusData = ptr.Get<BusData>();
    ptr.Advance( sizeof(BusData) * m_BusCount );
    for(int i = 0; i < m_BusCount; i++)
    {
        new( &m_pBusData[i] ) BusData();
        //m_pBusData[i].volume.InitValue( 1.0f );
        m_pBusData[i].isMute = false;
        m_pBusData[i].isPrevMute = false;
    }

    bool isSuccess = false;
    {
        pRendererManager->Lock();
        isSuccess = nn::audio::AcquireSubMix( &pRendererManager->GetAudioRendererConfig(), &m_SubMix, pRendererManager->GetAudioRendererParameter().sampleRate, GetMixBufferCount() );
        pRendererManager->Unlock();
    }

    m_IsInitialized = isSuccess;
    return isSuccess;
}
void SubMix::Finalize(detail::RendererManager* pRendererManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsRemovable() );

    if( m_IsInitialized )
    {
        m_IsInitialized = false;

        pRendererManager->Lock();
        nn::audio::ReleaseSubMix( &pRendererManager->GetAudioRendererConfig(), &m_SubMix );
        pRendererManager->Unlock();
    }

    m_pReceiver = nullptr;
    m_ppMixVolume = nullptr;
    m_pBusData = nullptr;
}
bool SubMix::IsRemovable() const NN_NOEXCEPT
{
    return m_ReferenceCount == 0u;
}
void SubMix::Update() NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsInitialized );
    for(int bus = 0; bus < m_BusCount; bus++)
    {
        //if( m_pBusData[bus].volume.IsFinished() )
        //{
        //    if( m_pBusData[bus].isMute != m_pBusData[bus].isPrevMute )
        //    {
        //        // ミュートの設定が変わったタイミングでボリュームを設定します
        //        UpdateBusVolume( bus );
        //        m_pBusData[bus].isPrevMute = m_pBusData[bus].isMute;
        //    }
        //}
        //else
        //{
        //    m_pBusData[bus].volume.Update();
        //    UpdateBusVolume( bus );
        //}
    }
}
void SubMix::UpdateBusVolume(int bus) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsInitialized );
    NN_SDK_ASSERT_NOT_NULL( m_pReceiver );
    NN_SDK_ASSERT_RANGE( bus, 0, m_BusCount );

    // TODO: pReceiver->GetBusCount() > 1 のときに対応する
    NN_SDK_REQUIRES_EQUAL( m_pReceiver->GetBusCount(), 1 );

    const int srcMixIdxBase = bus * m_ChannelCount;
    const int rootingChannelCount = std::min( m_ChannelCount, m_pReceiver->GetChannelCount() );

    switch( m_pReceiver->GetReceiverType() )
    {
        case ReceiverType::ReceiverType_SubMix:
        {
            nn::audio::SubMixType* pDestination = reinterpret_cast<SubMix*>( m_pReceiver )->GetAudioSubMixInstance();
            for(int i = 0; i < rootingChannelCount; i++)
            {
                const int srcMixIdx = srcMixIdxBase + i;
                const int dstMixIdx = i;
                //const float volume = m_pBusData[bus].isMute ? 0.0f : m_ppMixVolume[srcMixIdx][dstMixIdx] * m_pBusData[bus].volume.GetValue();
                const float volume = m_pBusData[bus].isMute ? 0.0f : m_ppMixVolume[srcMixIdx][dstMixIdx];
                nn::audio::SetSubMixMixVolume( &m_SubMix, pDestination, volume, srcMixIdx, dstMixIdx );
            }
            break;
        }
        case ReceiverType::ReceiverType_FinalMix:
        {
            nn::audio::FinalMixType* pDestination = reinterpret_cast<FinalMix*>( m_pReceiver )->GetAudioFinalMixInstance();
            for(int i = 0; i < rootingChannelCount; i++)
            {
                const int srcMixIdx = srcMixIdxBase + i;
                const int dstMixIdx = i;
                //const float volume = m_pBusData[bus].isMute ? 0.0f : m_ppMixVolume[srcMixIdx][dstMixIdx] * m_pBusData[bus].volume.GetValue();
                const float volume = m_pBusData[bus].isMute ? 0.0f : m_ppMixVolume[srcMixIdx][dstMixIdx];
                nn::audio::SetSubMixMixVolume( &m_SubMix, pDestination, volume, srcMixIdx, dstMixIdx );
            }
            break;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}
void SubMix::SetDestination(detail::RendererManager* pRendererManager, OutputReceiver* pReceiver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_NOT_NULL( pReceiver );

    // TODO: pReceiver->GetBusCount() > 1 のときに対応する
    NN_SDK_REQUIRES_EQUAL( pReceiver->GetBusCount(), 1 );

    m_pReceiver = pReceiver;
    switch( pReceiver->GetReceiverType() )
    {
        case ReceiverType::ReceiverType_SubMix:
        {
            nn::audio::SetSubMixDestination( &pRendererManager->GetAudioRendererConfig(), &m_SubMix, reinterpret_cast<SubMix*>( m_pReceiver )->GetAudioSubMixInstance() );
            break;
        }
        case ReceiverType::ReceiverType_FinalMix:
        {
            nn::audio::SetSubMixDestination( &pRendererManager->GetAudioRendererConfig(), &m_SubMix, reinterpret_cast<FinalMix*>( m_pReceiver )->GetAudioFinalMixInstance() );
            break;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    const int rootingChannelCount = std::min( m_ChannelCount, pReceiver->GetChannelCount() );
    for(int srcBus = 0; srcBus < m_BusCount; srcBus++)
    {
        for(int channel = 0; channel < rootingChannelCount; channel++)
        {
            //  pReceiver->GetBusCount() == 1 を仮定しているため、dstBus は 0 です。
            SetMixVolume( srcBus, channel, 0, channel, 1.0f );
        }
    }
}
void SubMix::SetBusVolume(int bus, float volume, int fadeFrame) NN_NOEXCEPT
{
    NN_UNUSED(bus);
    NN_UNUSED(volume);
    NN_UNUSED(fadeFrame);
    NN_SDK_ASSERT( m_IsInitialized );
    NN_SDK_ASSERT_RANGE( bus, 0, m_BusCount );
    NN_SDK_ASSERT_GREATER_EQUAL( fadeFrame, 0 );
    //m_pBusData[bus].volume.SetTarget( volume, fadeFrame );
}
float SubMix::GetBusVolume(int bus) const NN_NOEXCEPT
{
    NN_UNUSED(bus);
    NN_SDK_ASSERT( m_IsInitialized );
    NN_SDK_ASSERT_RANGE( bus, 0, m_BusCount );
    //return m_pBusData[bus].volume.GetValue();
    return 0.0f;
}
void SubMix::SetBusMute(int bus, bool isMute) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsInitialized );
    NN_SDK_ASSERT_RANGE( bus, 0, m_BusCount );
    m_pBusData[bus].isMute = isMute;
}
bool SubMix::IsBusMuted(int bus) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsInitialized );
    NN_SDK_ASSERT_RANGE( bus, 0, m_BusCount );
    return m_pBusData[bus].isMute;
}
void SubMix::SetMixVolume(int srcBus, int srcChannel, int dstBus, int dstChannel, float volume) NN_NOEXCEPT
{
    NN_UNUSED(srcBus);
    NN_UNUSED(dstBus);
    NN_UNUSED(srcChannel);
    NN_UNUSED(dstChannel);
    NN_UNUSED(srcChannel);
    NN_UNUSED(volume);
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_NOT_NULL( m_pReceiver );
    NN_SDK_REQUIRES_RANGE( srcBus, 0, m_BusCount );
    NN_SDK_REQUIRES_RANGE( srcChannel, 0, m_ChannelCount );
    NN_SDK_REQUIRES_RANGE( dstBus, 0, m_pReceiver->GetBusCount() );
    NN_SDK_REQUIRES_RANGE( dstChannel, 0, m_pReceiver->GetChannelCount() );

    //detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    //detail::DriverCommandSetMixVolume* command =
    //        cmdmgr.AllocCommand<detail::DriverCommandSetMixVolume>();

    //command->id = detail::DriverCommandId_SetMixVolume;
    //command->pReceiver = this;
    //command->volume = volume;
    //command->sourceBus = srcBus;
    //command->sourceChannel = srcChannel;
    //command->destinationBus = dstBus;
    //command->destinationChannel = dstChannel;

    //cmdmgr.PushCommand( command );
}
void SubMix::SetMixVolumeImpl(int srcBus, int srcChannel, int dstBus, int dstChannel, float volume) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsInitialized );
    NN_SDK_ASSERT_NOT_NULL( m_pReceiver );
    NN_SDK_ASSERT_RANGE( srcBus, 0, m_BusCount );
    NN_SDK_ASSERT_RANGE( srcChannel, 0, m_ChannelCount );
    NN_SDK_ASSERT_RANGE( dstBus, 0, m_pReceiver->GetBusCount() );
    NN_SDK_ASSERT_RANGE( dstChannel, 0, m_pReceiver->GetChannelCount() );

    const int srcIdx = m_ChannelCount * srcBus + srcChannel;
    const int dstIdx = m_pReceiver->GetChannelCount() * dstBus + dstChannel;

    m_ppMixVolume[srcIdx][dstIdx] = volume;
    //const float mixVolume = m_pBusData[srcBus].isMute ? 0.0f : volume * m_pBusData[srcBus].volume.GetValue();
    const float mixVolume = m_pBusData[srcBus].isMute ? 0.0f : volume;

    switch( m_pReceiver->GetReceiverType() )
    {
        case ReceiverType::ReceiverType_SubMix:
        {
            nn::audio::SetSubMixMixVolume( &m_SubMix, reinterpret_cast<SubMix*>( m_pReceiver )->GetAudioSubMixInstance(), mixVolume, srcIdx, dstIdx );
            break;
        }
        case ReceiverType::ReceiverType_FinalMix:
        {
            nn::audio::SetSubMixMixVolume( &m_SubMix, reinterpret_cast<FinalMix*>( m_pReceiver )->GetAudioFinalMixInstance(), mixVolume, srcIdx, dstIdx );
            break;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}
float SubMix::GetMixVolume(int srcBus, int srcChannel, int dstBus, int dstChannel) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_NOT_NULL( m_pReceiver );
    NN_SDK_REQUIRES_RANGE( srcBus, 0, m_BusCount );
    NN_SDK_REQUIRES_RANGE( srcChannel, 0, m_ChannelCount );
    NN_SDK_REQUIRES_RANGE( dstBus, 0, m_pReceiver->GetBusCount() );
    NN_SDK_REQUIRES_RANGE( dstChannel, 0, m_pReceiver->GetChannelCount() );

    const int srcIdx = m_ChannelCount * srcBus + srcChannel;
    const int dstIdx = m_pReceiver->GetChannelCount() * dstBus + dstChannel;
    return m_ppMixVolume[srcIdx][dstIdx];
}
void SubMix::SetMuteUnusedEffectChannel(bool isUnusedEffectChannelMuted) NN_NOEXCEPT
{
    m_IsUnusedEffectChannelMuted = isUnusedEffectChannelMuted;
}
bool SubMix::IsUnusedEffectChannelMuted() const NN_NOEXCEPT
{
    return m_IsUnusedEffectChannelMuted;
}
#if 0
void SubMix::SetupEffect(EffectBase* pEffect, int bus) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsInitialized );
    NN_SDK_ASSERT_NOT_NULL( pEffect );
    NN_SDK_ASSERT_RANGE( bus, 0, m_BusCount );

    if( m_IsUnusedEffectChannelMuted )
    {
        ChannelIndex effectChannelIndices[ChannelIndex_Count];
        for ( int channelIndex = 0; channelIndex < ChannelIndex_Count; ++channelIndex )
        {
            effectChannelIndices[channelIndex] = nn::atk::ChannelIndex_Count;
        }
        pEffect->GetChannelIndex( effectChannelIndices, pEffect->GetChannelSettingCountMax() );
        MuteUnusedEffectChannel( effectChannelIndices, pEffect->GetChannelSettingCountMax(), bus );
    }

    const int channelBase = bus * m_ChannelCount;
    int8_t inputOutputChannel[ChannelIndex_Count];
    for(int i = 0; i < m_ChannelCount; i++)
    {
        inputOutputChannel[i] = static_cast<int8_t>( i + channelBase );
    }
    pEffect->SetEffectInputOutput( inputOutputChannel, inputOutputChannel, m_ChannelCount, m_ChannelCount );
}
void SubMix::SetupEffect(EffectAux* pEffect, int bus) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsInitialized );
    NN_SDK_ASSERT_NOT_NULL( pEffect );
    NN_SDK_ASSERT_RANGE( bus, 0, m_BusCount );

    if( m_IsUnusedEffectChannelMuted )
    {
        ChannelIndex effectChannelIndices[ChannelIndex_Count];
        for ( int channelIndex = 0; channelIndex < ChannelIndex_Count; ++channelIndex )
        {
            effectChannelIndices[channelIndex] = nn::atk::ChannelIndex_Count;
        }
        pEffect->GetChannelIndex( effectChannelIndices, pEffect->GetChannelSettingCountMax() );
        MuteUnusedEffectChannel( effectChannelIndices, pEffect->GetChannelSettingCountMax(), bus );
    }

    const int channelBase = bus * m_ChannelCount;
    int8_t inputOutputChannel[ChannelIndex_Count];
    for(int i = 0; i < m_ChannelCount; i++)
    {
        inputOutputChannel[i] = static_cast<int8_t>( i + channelBase );
    }
    pEffect->SetEffectInputOutput( inputOutputChannel, inputOutputChannel, m_ChannelCount, m_ChannelCount );
}
void SubMix::MuteUnusedEffectChannel(ChannelIndex* effectChannelIndex, int effectChannelCount, int bus) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsInitialized );
    NN_SDK_ASSERT_NOT_NULL( effectChannelIndex );
    NN_SDK_ASSERT_RANGE( bus, 0, m_BusCount );

    // TODO: m_pReceiver->GetBusCount() > 1 のときに対応する
    NN_SDK_REQUIRES_EQUAL( m_pReceiver->GetBusCount(), 1 );

    for ( int channelIndex = 0; channelIndex < m_ChannelCount; ++channelIndex )
    {
        bool isExist = false;
        for ( auto i = 0; i < effectChannelCount; ++i )
        {
            if ( channelIndex == effectChannelIndex[i] )
            {
                isExist = true;
            }
        }

        // 有効なインデックスに含まれていなかったらミュート
        if ( !isExist )
        {
            //  pReceiver->GetBusCount() == 1 を仮定しているため、dstBus は 0 です。
            SetMixVolumeImpl( bus, channelIndex, 0, channelIndex, 0.0f );
        }
    }
}
#endif
OutputReceiver::ReceiverType SubMix::GetReceiverType() const NN_NOEXCEPT
{
    return ReceiverType::ReceiverType_SubMix;
}
int SubMix::GetChannelCount() const NN_NOEXCEPT
{
    return m_ChannelCount;
}
int SubMix::GetBusCount() const NN_NOEXCEPT
{
    return m_BusCount;
}
void SubMix::AddReferenceCount(int value) NN_NOEXCEPT
{
    m_ReferenceCount += value;
}

} // namespace nn::atk
} // namespace nn
