﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk2/atk2_AudioEngine.h>

#include <nn/nn_SdkLog.h>
#include <nn/util/util_BytePtr.h>
#include <nn/os/os_MemoryHeapCommon.h>

namespace nn { namespace atk2 {

    AudioEngine::AudioEngine() NN_NOEXCEPT
        : m_IsAudioThreadEnabled(false)
        , m_IsManualAudioFrameUpdateEnabled(false)
    {
    }

    void AudioEngine::GetDefaultInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        arg.renderingSampleRate = 48000;
        arg.audioThreadUpdateIntervalMSec = 5;
        arg.effectCount = 1;
        arg.voiceCount = 2;

        // オーディオスレッド
        arg.isAudioThreadEnabled = true;
        arg.audioThreadPriority = 0;
        arg.audioThreadCoreNumber = 0;
        arg.audioThreadMessageBufferSize = 32;
        arg.audioThreadStackSize = 16 * 1024;
        arg.audioThreadAffinityMask = 0;
        arg.isManualAudioFrameUpdateEnabled = false;

        // プラットフォーム固有
        platformArg.mixBufferCount = 6;
        platformArg.subMixCount = 0;
        platformArg.sinkCount = 1;
        platformArg.performanceFrameCount = 0;
        platformArg.waveBufferPacketCount = 512;
        platformArg.voiceCommandBufferSize = 512 * 1024;

        m_RendererManager.GetDefaultInitArg(arg._rendererInitArg, platformArg._rendererInitArg);
        m_RendererVoiceManager.GetDefaultInitArg(arg._voiceInitArg, platformArg._voiceInitArg);
        m_VoiceCommandManager.GetDefaultInitArg(arg._voiceCommandInitArg, platformArg._voiceCommandInitArg);
        m_VoiceReplyCommandManager.GetDefaultInitArg(arg._voiceReplyCommandInitArg, platformArg._voiceReplyCommandInitArg);
        m_AudioThread.GetDefaultInitArg(arg._audioThreadInitArg, platformArg._audioThreadInitArg);
    }

    size_t AudioEngine::GetRequiredBufferSize(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        size_t bufferSize = 0;
        SetupVoiceCommandInitArg(arg, platformArg);
        SetupVoiceReplyCommandInitArg(arg, platformArg);
        SetupAudioThreadInitArg(arg, platformArg);
        bufferSize += m_VoiceCommandManager.GetRequiredBufferSize(arg._voiceCommandInitArg, platformArg._voiceCommandInitArg);
        bufferSize += m_VoiceReplyCommandManager.GetRequiredBufferSize(arg._voiceReplyCommandInitArg, platformArg._voiceReplyCommandInitArg);
        bufferSize += (arg.isAudioThreadEnabled) ? m_AudioThread.GetRequiredBufferSize(arg._audioThreadInitArg) : 0;
        return bufferSize;
    }

    AudioEngine::Result AudioEngine::Initialize(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(arg.workBufferSize, GetRequiredBufferSize(arg, platformArg));
        NN_SDK_REQUIRES_GREATER_EQUAL(platformArg.workBufferSize, GetRequiredPlatformBufferSize(arg, platformArg));
        NN_SDK_REQUIRES_ALIGNED(platformArg.workBuffer, nn::os::MemoryPageSize);

        nn::util::BytePtr buffer(arg.workBuffer);
        nn::util::BytePtr platformBuffer(platformArg.workBuffer);
        nn::util::BytePtr platformBufferForMemoryPool(platformArg.workBufferForMemoryPool);
        NN_SDK_LOG("[atk2Debug]AudioEngine 0x%08x %zd\n", arg.workBuffer, arg.workBufferSize);
        NN_SDK_LOG("[atk2Debug]AudioEnginePlatform 0x%08x %zd\n", platformArg.workBuffer, platformArg.workBufferSize);

        // レンダラの初期化
        size_t rendererPlatformBufferSize = m_RendererManager.GetRequiredPlatformBufferSize(arg._rendererInitArg, platformArg._rendererInitArg);
        platformArg._rendererWorkBuffer = platformBuffer.Get();
        platformArg._rendererWorkBufferSize = rendererPlatformBufferSize;
        NN_SDK_LOG("[atk2Debug]RendererPlatform 0x%08x %zd\n", platformArg._rendererWorkBuffer, platformArg._rendererWorkBufferSize);
        SetupRendererInitArg(arg, platformArg);
        detail::RendererManager::Result resultForRenderer = m_RendererManager.Initialize(arg._rendererInitArg, platformArg._rendererInitArg);
        if (resultForRenderer != detail::RendererManager::Result_Success)
        {
            NN_SDK_LOG("Failed to Initialize RendererManager.\n");
            return Result_FailedToInitialize;
        }
        platformBuffer.Advance(rendererPlatformBufferSize);

        // ボイスの初期化
        size_t voiceBufferSize = m_RendererVoiceManager.GetRequiredBufferSize(arg._voiceInitArg, platformArg._voiceInitArg);
        platformArg._voiceWorkBuffer = buffer.Get();
        platformArg._voiceWorkBufferSize = voiceBufferSize;
        NN_SDK_LOG("[atk2Debug]VoicePlatform 0x%08x %zd\n", platformArg._voiceWorkBuffer, platformArg._voiceWorkBufferSize);
        size_t voiceBufferSizeForMemoryPool = m_RendererVoiceManager.GetRequiredBufferSizeForMemoryPool(arg._voiceInitArg, platformArg._voiceInitArg);
        platformArg._voiceWorkBufferForMemoryPool = platformBufferForMemoryPool.Get();
        platformArg._voiceWorkBufferSizeForMemoryPool = voiceBufferSizeForMemoryPool;
        NN_SDK_LOG("[atk2Debug]VoiceWorkBufferForMemoryPool 0x%08x %zd\n", platformArg._voiceWorkBufferForMemoryPool, platformArg._voiceWorkBufferSizeForMemoryPool);
        SetupVoiceInitArg(arg, platformArg);
        detail::RendererVoiceManager::Result resultForVoice = m_RendererVoiceManager.Initialize(arg._voiceInitArg, platformArg._voiceInitArg);
        if (resultForVoice != detail::RendererVoiceManager::Result_Success)
        {
            NN_SDK_LOG("Failed to Initialize RendererVoiceManager.\n");
            return Result_FailedToInitialize;
        }
        buffer.Advance(voiceBufferSize);
        platformBufferForMemoryPool.Advance(voiceBufferSizeForMemoryPool);

        // バーチャルボイスの初期化
        m_VirtualVoiceManager.Initialize();

        // ボイスコマンドの初期化
        size_t voiceCommandBufferSize = m_VoiceCommandManager.GetRequiredBufferSize(arg._voiceCommandInitArg, platformArg._voiceCommandInitArg);
        platformArg._voiceCommandWorkBuffer = buffer.Get();
        platformArg._voiceCommandWorkBufferSize = voiceCommandBufferSize;
        NN_SDK_LOG("[atk2Debug]VoiceCommandPlatform 0x%08x %zd\n", platformArg._voiceCommandWorkBuffer, platformArg._voiceCommandWorkBufferSize);
        size_t voiceCommandBufferSizeForMemoryPool = m_VoiceCommandManager.GetRequiredBufferSizeForMemoryPool(arg._voiceCommandInitArg, platformArg._voiceCommandInitArg);
        platformArg._voiceCommandWorkBufferForMemoryPool = platformBufferForMemoryPool.Get();
        platformArg._voiceCommandWorkBufferSizeForMemoryPool = voiceCommandBufferSizeForMemoryPool;
        NN_SDK_LOG("[atk2Debug]VoiceCommandWorkBufferForMemoryPool 0x%08x %zd\n", platformArg._voiceCommandWorkBufferForMemoryPool, platformArg._voiceCommandWorkBufferSizeForMemoryPool);
        SetupVoiceCommandInitArg(arg, platformArg);
        m_VoiceCommandManager.Initialize(arg._voiceCommandInitArg, platformArg._voiceCommandInitArg);
        buffer.Advance(voiceCommandBufferSize);
        platformBufferForMemoryPool.Advance(voiceCommandBufferSizeForMemoryPool);

        // ボイスリプライコマンドの初期化
        size_t voiceReplyCommandBufferSize = m_VoiceReplyCommandManager.GetRequiredBufferSize(arg._voiceReplyCommandInitArg, platformArg._voiceReplyCommandInitArg);
        platformArg._voiceReplyCommandWorkBuffer = buffer.Get();
        platformArg._voiceReplyCommandWorkBufferSize = voiceReplyCommandBufferSize;
        NN_SDK_LOG("[atk2Debug]VoiceReplyCommandPlatform 0x%08x %zd\n", platformArg._voiceReplyCommandWorkBuffer, platformArg._voiceReplyCommandWorkBufferSize);
        SetupVoiceReplyCommandInitArg(arg, platformArg);
        m_VoiceReplyCommandManager.Initialize(arg._voiceReplyCommandInitArg, platformArg._voiceReplyCommandInitArg);
        buffer.Advance(voiceReplyCommandBufferSize);

        // オーディオフレームアップデータの設定
        m_AudioFrameUpdater.RegisterAudioFrameUpdate(&m_RendererManager);
        m_AudioFrameUpdater.RegisterAudioFrameUpdate(&m_VoiceCommandManager);
        m_AudioFrameUpdater.RegisterAudioFrameUpdate(&m_RendererVoiceManager);
        m_AudioFrameUpdater.RegisterAudioFrameUpdate(&m_VirtualVoiceManager);

        // オーディオスレッドの初期化
        if (arg.isAudioThreadEnabled)
        {
            size_t audioThreadBufferSize = m_AudioThread.GetRequiredBufferSize(arg._audioThreadInitArg);
            arg._audioThreadWorkBuffer = buffer.Get();
            arg._audioThreadWorkBufferSize = audioThreadBufferSize;
            platformArg._audioThreadWorkBuffer = platformBuffer.Get();
            NN_SDK_LOG("[atk2Debug]AudioThread 0x%08x %zd\n", arg._audioThreadWorkBuffer, arg._audioThreadWorkBufferSize);
            SetupAudioThreadInitArg(arg, platformArg);
            detail::AudioThread::Result resultForAudioThread = m_AudioThread.Initialize(arg._audioThreadInitArg, platformArg._audioThreadInitArg);
            if (resultForAudioThread != detail::AudioThread::Result_Success)
            {
                NN_SDK_LOG("Failed to Initialize AudioThread.\n");
                return Result_FailedToInitialize;
            }
            buffer.Advance(audioThreadBufferSize);
        }
        m_IsAudioThreadEnabled = arg.isAudioThreadEnabled;
        m_IsManualAudioFrameUpdateEnabled = arg.isManualAudioFrameUpdateEnabled;

        return Result_Success;
    }

    void AudioEngine::Finalize() NN_NOEXCEPT
    {
        if (m_IsAudioThreadEnabled)
        {
            m_AudioThread.Finalize();
            m_IsAudioThreadEnabled = false;
        }

        m_AudioFrameUpdater.UnregisterAudioFrameUpdate(&m_VirtualVoiceManager);
        m_AudioFrameUpdater.UnregisterAudioFrameUpdate(&m_RendererVoiceManager);
        m_AudioFrameUpdater.UnregisterAudioFrameUpdate(&m_VoiceCommandManager);
        m_AudioFrameUpdater.UnregisterAudioFrameUpdate(&m_RendererManager);

        m_VoiceReplyCommandManager.Finalize();
        m_VoiceCommandManager.Finalize();
        m_RendererVoiceManager.Finalize();
        m_RendererManager.Finalize();

        m_IsManualAudioFrameUpdateEnabled = false;
    }

    AudioEngine::Result AudioEngine::Update() NN_NOEXCEPT
    {
        if (!m_IsAudioThreadEnabled && !m_IsManualAudioFrameUpdateEnabled)
        {
            if (m_AudioFrameUpdater.Update() != nn::atk2::detail::AudioFrameUpdater::Result_Success)
            {
                NN_SDK_LOG("Failed to update audio frame updater.\n");
                return Result_ErrorUnknown;
            }
        }

        return Result_Success;
    }

    detail::AudioThread& AudioEngine::GetAudioThread() NN_NOEXCEPT
    {
        return m_AudioThread;
    }

    detail::AudioFrameUpdater& AudioEngine::GetAudioFrameUpdater() NN_NOEXCEPT
    {
        return m_AudioFrameUpdater;
    }

    // プラットフォーム固有
    size_t AudioEngine::GetRequiredPlatformBufferSize(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        size_t bufferSize = 0;
        SetupRendererInitArg(arg, platformArg);
        SetupVoiceInitArg(arg, platformArg);
        SetupAudioThreadInitArg(arg, platformArg);
        bufferSize += m_RendererManager.GetRequiredPlatformBufferSize(arg._rendererInitArg, platformArg._rendererInitArg);
        bufferSize += m_RendererVoiceManager.GetRequiredPlatformBufferSize(arg._voiceInitArg, platformArg._voiceInitArg);
        bufferSize += m_AudioThread.GetRequiredPlatformBufferSize(arg._audioThreadInitArg, platformArg._audioThreadInitArg);
        return bufferSize;
    }

    size_t AudioEngine::GetRequiredBufferSizeForMemoryPool(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        size_t bufferSize = 0;
        SetupRendererInitArg(arg, platformArg);
        SetupVoiceInitArg(arg, platformArg);
        SetupVoiceCommandInitArg(arg, platformArg);
        bufferSize += m_RendererManager.GetRequiredBufferSizeForMemoryPool(arg._rendererInitArg, platformArg._rendererInitArg);
        bufferSize += m_RendererVoiceManager.GetRequiredBufferSizeForMemoryPool(arg._voiceInitArg, platformArg._voiceInitArg);
        bufferSize += m_VoiceCommandManager.GetRequiredBufferSizeForMemoryPool(arg._voiceCommandInitArg, platformArg._voiceCommandInitArg);
        return bufferSize;
    }

    AudioEngine::Result AudioEngine::SetupRendererInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        arg._rendererInitArg._pAudioEngine = this;
        platformArg._rendererInitArg.workBuffer = platformArg._rendererWorkBuffer;
        platformArg._rendererInitArg.workBufferSize = platformArg._rendererWorkBufferSize;
        platformArg._rendererInitArg.rendererParameter.voiceCount = arg.voiceCount;
        platformArg._rendererInitArg.rendererParameter.sampleRate = arg.renderingSampleRate;
        platformArg._rendererInitArg.rendererParameter.effectCount = arg.effectCount;
        platformArg._rendererInitArg.rendererParameter.sampleCount = arg.renderingSampleRate * arg.audioThreadUpdateIntervalMSec / 1000;
        platformArg._rendererInitArg.rendererParameter.mixBufferCount = platformArg.mixBufferCount + detail::RendererManager::ChannelCount;
        platformArg._rendererInitArg.rendererParameter.subMixCount = platformArg.subMixCount;
        platformArg._rendererInitArg.rendererParameter.sinkCount = platformArg.sinkCount;
        platformArg._rendererInitArg.rendererParameter.performanceFrameCount = platformArg.performanceFrameCount;
        return Result_Success;
    }

    AudioEngine::Result AudioEngine::SetupVoiceInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        NN_UNUSED(platformArg);
        arg._voiceInitArg._pAudioEngine = this;
        arg._voiceInitArg.voiceCount = arg.voiceCount;
        arg._voiceInitArg.workBuffer = platformArg._voiceWorkBuffer;
        arg._voiceInitArg.workBufferSize = platformArg._voiceWorkBufferSize;
        platformArg._voiceInitArg.workBufferForMemoryPool = platformArg.workBufferForMemoryPool;
        platformArg._voiceInitArg.workBufferSizeForMemoryPool = platformArg.workBufferSizeForMemoryPool;
        return Result_Success;
    }

    AudioEngine::Result AudioEngine::SetupVoiceCommandInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        arg._voiceCommandInitArg._pAudioEngine = this;
        arg._voiceCommandInitArg.commandBufferSize = platformArg.voiceCommandBufferSize;
        arg._voiceCommandInitArg.waveBufferPacketCount = platformArg.waveBufferPacketCount;
        arg._voiceCommandInitArg.workBuffer = platformArg._voiceCommandWorkBuffer;
        arg._voiceCommandInitArg.workBufferSize = platformArg._voiceCommandWorkBufferSize;
        platformArg._voiceCommandInitArg.workBufferForMemoryPool = platformArg.workBufferForMemoryPool;
        platformArg._voiceCommandInitArg.workBufferSizeForMemoryPool = platformArg.workBufferSizeForMemoryPool;
        return Result_Success;
    }

    AudioEngine::Result AudioEngine::SetupVoiceReplyCommandInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        NN_UNUSED(platformArg);
        arg._voiceReplyCommandInitArg._pAudioEngine = this;
        arg._voiceReplyCommandInitArg.commandBufferSize = platformArg.voiceCommandBufferSize;
        arg._voiceReplyCommandInitArg.workBuffer = platformArg._voiceReplyCommandWorkBuffer;
        arg._voiceReplyCommandInitArg.workBufferSize = platformArg._voiceReplyCommandWorkBufferSize;
        return Result_Success;
    }

    AudioEngine::Result AudioEngine::SetupAudioThreadInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT
    {
        NN_UNUSED(platformArg);
        arg._audioThreadInitArg._pAudioEngine = this;
        arg._audioThreadInitArg._pAudioFrameUpdater = &m_AudioFrameUpdater;
        arg._audioThreadInitArg.threadStackSize = arg.audioThreadStackSize;
        arg._audioThreadInitArg.messageBufferSize = arg.audioThreadMessageBufferSize;
        arg._audioThreadInitArg.threadPriority = arg.audioThreadPriority;
        arg._audioThreadInitArg.threadCoreNumber = arg.audioThreadCoreNumber;
        arg._audioThreadInitArg.threadAffinityMask = arg.audioThreadAffinityMask;
        arg._audioThreadInitArg.threadUpdateIntervalMSec = arg.audioThreadUpdateIntervalMSec;
        arg._audioThreadInitArg.workBuffer = arg._audioThreadWorkBuffer;
        arg._audioThreadInitArg.workBufferSize = arg._audioThreadWorkBufferSize;
        platformArg._audioThreadInitArg.workBuffer = platformArg._audioThreadWorkBuffer;
        platformArg._audioThreadInitArg.workBufferSize = platformArg._audioThreadWorkBufferSize;
        return Result_Success;
    }


    detail::RendererManager& AudioEngine::GetRendererManager() NN_NOEXCEPT
    {
        return m_RendererManager;
    }

    detail::RendererVoiceManager& AudioEngine::GetRendererVoiceManager() NN_NOEXCEPT
    {
        return m_RendererVoiceManager;
    }

    detail::VirtualVoiceManager& AudioEngine::GetVirtualVoiceManager() NN_NOEXCEPT
    {
        return m_VirtualVoiceManager;
    }

    detail::VoiceCommandManager& AudioEngine::GetVoiceCommandManager() NN_NOEXCEPT
    {
        return m_VoiceCommandManager;
    }

    detail::VoiceReplyCommandManager& AudioEngine::GetVoiceReplyCommandManager() NN_NOEXCEPT
    {
        return m_VoiceReplyCommandManager;
    }

}}
