﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/detail/res/atk_ResourceManager.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/atk/atk_DisposeCallbackManager.h>
#include <nn/atk/atk_SoundSystem.h>
#include <nn/atk/atk_Util.h>
#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/viewer/atk_Config.h>

#if !defined(NN_SDK_BUILD_RELEASE)
//#define NN_ATK_ENABLE_CACHE_DEBUG
#endif

namespace nn {
namespace atk {
namespace viewer {
namespace detail {

namespace
{

//----------------------------------------------------------
// データバッファを無効化します。
void InvalidateDataBuffer(CacheManager::Item& item) NN_NOEXCEPT
{
    atk::SoundSystem::LockSoundThread();

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO("[sndedit] InvalidateDataBuffer : + Dispose() Name=%s\n", item.GetName().GetName());
#endif

    // アイテムデータを利用中かもしれないので、解放する前に再生を強制停止しておきます。
    atk::detail::driver::DisposeCallbackManager::GetInstance().Dispose(
        item.GetData(),
        item.GetDataSize()
        );
    atk::detail::driver::DisposeCallbackManager::GetInstance().Dispose(
        item.GetPlaybackBuffer(),
        item.GetDataSize()
    );

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO("[sndedit] InvalidateDataBuffer : - Dispose() Name=%s\n", item.GetName().GetName());
#endif

    atk::SoundSystem::UnlockSoundThread();
}

//----------------------------------------------------------
// キャッシュを削除します。
void SafeRemoveCache(CacheManager& cacheManager, const char* name) NN_NOEXCEPT
{
    cacheManager.RemoveItem(name, InvalidateDataBuffer);
}

//----------------------------------------------------------
// キャッシュの参照カウントを安全にデクリメントします。
void SafeDecrementReferenceCount(CacheManager& cacheManager, const char* name) NN_NOEXCEPT
{
    CacheManager::Item* cache = cacheManager.GetItem(name);

    if(cache == nullptr)
    {
        return;
    }

    cache->DecrementReferenceCount();

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO("[sndedit] SafeDecrementReferenceCount : Count=%d, Name=%s\n", cache->GetReferenceCount(), name);
#endif

    if(cache->GetReferenceCount() == 0)
    {
        cacheManager.RemoveItem(name, InvalidateDataBuffer);
    }
}

//----------------------------------------------------------
// 削除中のキャッシュアイテムかどうか調べますします。
bool IsRemovingCacheItem(CacheManager::Item& item) NN_NOEXCEPT
{
    return item.GetUserParameter() == ResourceManager::ResourceState_Removing;
}

//----------------------------------------------------------
//! @brief  キャッシュのフリーズを解除する関数オブジェクトです。
class CacheUnfreezer
{
public:
    explicit CacheUnfreezer(ResourceManager::ResourceState resourceState) NN_NOEXCEPT :
      m_ResourceState(resourceState) { }

public:
    void operator()(CacheManager::Item& item) NN_NOEXCEPT
    {
        item.SetUserParameter(m_ResourceState);
        item.Unfreeze();
    }

private:
    ResourceManager::ResourceState m_ResourceState;
};

}

//----------------------------------------------------------
// ResourceManager::ErrorItem
//----------------------------------------------------------
ResourceManager::ErrorItem::ErrorItem() NN_NOEXCEPT
: m_ItemName(nullptr)
, m_ItemNameLength(0)
, m_Result()
, m_Reference()
{
    m_Reference.value = this;
}

ResourceManager::ErrorItem::~ErrorItem() NN_NOEXCEPT
{
    m_ItemNameLength = 0;
    m_ItemName = nullptr;
}

void ResourceManager::ErrorItem::SetItemName(const char* itemName) NN_NOEXCEPT
{
    if (itemName == nullptr)
    {
        return;
    }

    std::memcpy(m_ItemName, itemName, m_ItemNameLength);
}

char* ResourceManager::ErrorItem::AttachItemNameBuffer(char* itemNameBuffer, int itemNameLength) NN_NOEXCEPT
{
    char* lastBuffer = m_ItemName;
    m_ItemName = itemNameBuffer;
    m_ItemNameLength = itemNameLength;

    return lastBuffer;
}

//----------------------------------------------------------
NN_DEFINE_STATIC_CONSTANT( const size_t ResourceManager::ResourceAllocationThresholdSize );

//----------------------------------------------------------
ResourceManager::ResourceManager() NN_NOEXCEPT :
m_pMemoryForItemInfoManager(nullptr),
m_pMemoryForFileManager(nullptr),
m_pMemoryForErrorItemList(nullptr),
m_pMemoryForPlaybackHeap(nullptr),
m_IsPlaybackMemoryPoolAttached(false),
m_AllItemsState(ResourceState_Unknown),
m_MaxName(0),
m_IsItemInfoDictionaryExpandable(false),
m_IsFileDictionaryExpandable(false)
{
}

//----------------------------------------------------------
Result
ResourceManager::Initialize(
    atk::detail::fnd::FrameHeap& heap,
    uint32_t baseId,
    int maxItems,
    int maxFiles,
    int maxName,
    bool isMaxItemsExpandable,
    bool isMaxFilesExpandable) NN_NOEXCEPT
{
    // DataCacheHeap の初期化
    size_t dataCacheBufferLength = heap.GetFreeLength() / 2;
    void* dataCacheBuffer = heap.Alloc(dataCacheBufferLength);

    if(dataCacheBuffer == nullptr)
    {
        return Result(ResultType_OutOfMemory);
    }

    if (!atk::detail::Util::IsValidMemoryForDsp(dataCacheBuffer, dataCacheBufferLength))
    {
        NN_ATK_WARNING(
            "the memory area (0x%08x - 0x%08x %dbyte) provided cross a 512 MB segment.",
            dataCacheBuffer,
            util::BytePtr(dataCacheBuffer, dataCacheBufferLength).Get(),
            dataCacheBufferLength );
    }

    if(!m_DataCacheHeap.Initialize(dataCacheBuffer, dataCacheBufferLength))
    {
        // 拡張ヒープを利用可能な最小メモリサイズに満たない
        return Result(ResultType_OutOfMemory);
    }

    // 断片化を防ぐためにエラーアイテム用のヒープを用意
    size_t errorItemHeapSize = GetErrorItemHeapRequiredMemorySize(maxName);
    m_pMemoryForErrorItemList = m_DataCacheHeap.Alloc(errorItemHeapSize);
    if(m_pMemoryForErrorItemList == nullptr)
    {
        return Result(ResultType_OutOfMemory);
    }
    if(!m_ErrorItemHeap.Initialize(m_pMemoryForErrorItemList, errorItemHeapSize))
    {
        // 拡張ヒープを利用可能な最小メモリサイズに満たない
        return Result(ResultType_OutOfMemory);
    }

    // PlaybackHeap の初期化
    size_t playbackBufferLength = nn::util::align_down(heap.GetFreeLength(Alignments::AudioMemoryPool), nn::audio::MemoryPoolType::SizeGranularity);
    m_pMemoryForPlaybackHeap = heap.Alloc(playbackBufferLength, Alignments::AudioMemoryPool);

    if (m_pMemoryForPlaybackHeap == nullptr)
    {
        return Result(ResultType_OutOfMemory);
    }
    if(!m_PlaybackHeap.Initialize(m_pMemoryForPlaybackHeap, playbackBufferLength))
    {
        // 拡張ヒープを利用可能な最小メモリサイズに満たない
        return Result(ResultType_OutOfMemory);
    }

    nn::atk::SoundSystem::AttachMemoryPool(&m_PlaybackMemoryPool, m_pMemoryForPlaybackHeap, playbackBufferLength);
    m_IsPlaybackMemoryPoolAttached = true;

    // メモリの断片化を軽減するために、リソースサイズに応じて、
    // 拡張ヒープの割り当て先（ヒープの前から or 後ろから）を切り替える
    // アロケータを利用します。
    m_ResourceAllocator.Initialize(&m_DataCacheHeap, ResourceAllocationThresholdSize);
    m_PlaybackBufferAllocator.Initialize(&m_PlaybackHeap, ResourceAllocationThresholdSize);

    Result result = InitializeItemInfoManager(baseId, maxItems, maxName);

    if(result.IsFailed())
    {
        Finalize();
        return result;
    }

    result = InitializeFileManager(maxFiles);

    if(result.IsFailed())
    {
        Finalize();
        return result;
    }

    m_IsItemInfoDictionaryExpandable = isMaxItemsExpandable;
    m_IsFileDictionaryExpandable = isMaxFilesExpandable;

    return result;
}

//----------------------------------------------------------
void
ResourceManager::Finalize() NN_NOEXCEPT
{
    if(m_FileManager.IsInitialized())
    {
        // 再生中のバッファを安全に解放する(InvalidateDataBuffer() する)ために、
        // 明示的に RemoveAllItemFiles() を呼び出す。
        RemoveAllItemFiles();
    }

    m_ItemInfoManager.Finalize();
    m_FileManager.Finalize();

    if(m_ResourceAllocator.IsInitialized())
    {
        m_ResourceAllocator.Free(m_pMemoryForFileManager);
        m_ResourceAllocator.Free(m_pMemoryForItemInfoManager);
    }
    m_pMemoryForFileManager = nullptr;
    m_pMemoryForItemInfoManager = nullptr;

    if(m_IsPlaybackMemoryPoolAttached)
    {
        nn::atk::SoundSystem::DetachMemoryPool(&m_PlaybackMemoryPool);
        m_IsPlaybackMemoryPoolAttached = false;
    }

    m_PlaybackHeap.Finalize();
    m_pMemoryForPlaybackHeap = nullptr;

    DestroyAllErrorItems();
    m_ErrorItemHeap.Finalize();
    m_DataCacheHeap.Free(m_pMemoryForErrorItemList);
    m_pMemoryForErrorItemList = nullptr;

    m_PlaybackBufferAllocator.Finalize();
    m_ResourceAllocator.Finalize();
    m_DataCacheHeap.Finalize();

    m_IsItemInfoDictionaryExpandable = false;
    m_IsFileDictionaryExpandable = false;
}

//----------------------------------------------------------
size_t
ResourceManager::GetMemoryUsage() const NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return 0;
    }

    // アイテム情報キャッシュ管理領域のサイズ（固定長）
    size_t result = m_ItemInfoManager.GetRequiredMemorySize(m_ItemInfoManager.GetMaxItemCount());

    // アイテム情報キャッシュデータのサイズ（可変長）
    result += m_ItemInfoManager.GetMemoryUsage();

    // ファイルキャッシュ管理領域のサイズ（固定長）
    result += m_FileManager.GetRequiredMemorySize(m_FileManager.GetMaxItemCount());

    // ファイルキャッシュデータのサイズ（可変長）
    result += m_FileManager.GetMemoryUsage();

    return result;
}

//----------------------------------------------------------
bool
ResourceManager::TryGetEditItemInfo(int index, EditItemInfo* pItemInfo) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pItemInfo);
    const CacheManager::Item* pItem = m_ItemInfoManager.GetItem(index);

    if(pItem == nullptr)
    {
        return false;
    }

    pItemInfo->id = pItem->GetID();
    pItemInfo->name = pItem->GetName().GetName();
    pItemInfo->isOverrideOriginal = pItem->IsOverrideOriginal();

    switch(pItem->GetDataType())
    {
    case ResDataType_StreamSound:
        pItemInfo->itemType = EditItemType_StreamSound;
        break;

    case ResDataType_WaveSound:
        pItemInfo->itemType = EditItemType_WaveSound;
        break;

    case ResDataType_SequenceSound:
        pItemInfo->itemType = EditItemType_SequenceSound;
        break;

    case ResDataType_Bank:
        pItemInfo->itemType = EditItemType_Bank;
        break;

    default:
        pItemInfo->itemType = EditItemType_Unknown;
        break;
    }

    return true;
}

//----------------------------------------------------------
bool
ResourceManager::TryGetEditItemInfo(const char* name, EditItemInfo* pItemInfo) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(name);
    NN_SDK_ASSERT_NOT_NULL(pItemInfo);
    const CacheManager::Item* pItem = m_ItemInfoManager.GetItem(name);

    if(pItem == nullptr)
    {
        return false;
    }

    pItemInfo->id = pItem->GetID();
    pItemInfo->name = pItem->GetName().GetName();
    pItemInfo->isOverrideOriginal = pItem->IsOverrideOriginal();

    switch(pItem->GetDataType())
    {
    case ResDataType_StreamSound:
        pItemInfo->itemType = EditItemType_StreamSound;
        break;

    case ResDataType_WaveSound:
        pItemInfo->itemType = EditItemType_WaveSound;
        break;

    case ResDataType_SequenceSound:
        pItemInfo->itemType = EditItemType_SequenceSound;
        break;

    case ResDataType_Bank:
        pItemInfo->itemType = EditItemType_Bank;
        break;

    default:
        pItemInfo->itemType = EditItemType_Unknown;
        break;
    }

    return true;
}

//----------------------------------------------------------
bool
ResourceManager::TryGetEditItemInfoFromID(uint32_t id, EditItemInfo* pItemInfo) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pItemInfo);
    const CacheManager::Item* pItem = m_ItemInfoManager.GetItemFromID(id);

    if(pItem == nullptr)
    {
        return false;
    }

    pItemInfo->id = pItem->GetID();
    pItemInfo->name = pItem->GetName().GetName();
    pItemInfo->isOverrideOriginal = pItem->IsOverrideOriginal();

    switch(pItem->GetDataType())
    {
    case ResDataType_StreamSound:
        pItemInfo->itemType = EditItemType_StreamSound;
        break;

    case ResDataType_WaveSound:
        pItemInfo->itemType = EditItemType_WaveSound;
        break;

    case ResDataType_SequenceSound:
        pItemInfo->itemType = EditItemType_SequenceSound;
        break;

    case ResDataType_Bank:
        pItemInfo->itemType = EditItemType_Bank;
        break;

    default:
        pItemInfo->itemType = EditItemType_Unknown;
        break;
    }

    return true;
}

//----------------------------------------------------------
bool
ResourceManager::ContainsItemInfo(const char* name) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(name);
    NN_SDK_ASSERT(*name != '\0');

    return m_ItemInfoManager.GetItem(name) != nullptr;
}

//----------------------------------------------------------
bool
ResourceManager::TryGetItemID(const char* name, uint32_t* id) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(name);
    NN_SDK_ASSERT(*name != '\0');

    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    const CacheManager::Item* item =m_ItemInfoManager.GetItem(name);

    if(item == nullptr)
    {
        return false;
    }

    if(id != nullptr)
    {
        *id = item->GetID();
    }

    return true;
}

//----------------------------------------------------------
CacheState
ResourceManager::GetItemCacheState(const char* name) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(name);
    NN_SDK_ASSERT(*name != '\0');

    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    const CacheManager::Item* item =m_ItemInfoManager.GetItem(name);

    if(item == nullptr)
    {
        return CacheState_None;
    }

    return item->IsFreezed() ? CacheState_Cached : CacheState_Caching;
}

//----------------------------------------------------------
const void*
ResourceManager::GetItemInfo(
    const char* soundName,
    ResDataType* itemInfoDataType /*= nullptr*/,
    bool allowUnfreezedData /*= false*/) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(soundName);
    NN_SDK_ASSERT(*soundName != '\0');

    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    uint32_t dataType = ResDataType_Unknown;
    const void* result = m_ItemInfoManager.GetData(soundName, &dataType, allowUnfreezedData);

    if(itemInfoDataType != nullptr)
    {
        *itemInfoDataType = static_cast<ResDataType>(dataType);
    }

    return result;
}

//----------------------------------------------------------
ResDataType
ResourceManager::GetItemInfoDataType(const char* itemName, Result* result) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(itemName);
    NN_SDK_ASSERT(*itemName != '\0');

    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    const CacheManager::Item* item = m_ItemInfoManager.GetItem(itemName);

    if(item == nullptr)
    {
        if(result != nullptr)
        {
            *result = Result(ResultType_NameNotFound);
        }

        return ResDataType_Unknown;
    }

    if(result != nullptr)
    {
        *result = Result(ResultType_True);
    }

    return static_cast<ResDataType>(item->GetDataType());
}

//----------------------------------------------------------
Result
ResourceManager::SetItemInfo(
    const char* itemName,
    uint32_t originalId,
    ResDataType itemType,
    const void* itemInfo) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(itemName);
    NN_SDK_ASSERT_NOT_NULL(itemInfo);

    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    size_t dataSize = 0;

    switch(itemType)
    {
    case ResDataType_StreamSound:
        dataSize = ResStreamSoundInfo::GetRequiredSize(m_MaxName, Limits::MaxFilePath);
        break;

    case ResDataType_WaveSound:
        dataSize = ResWaveSoundInfo::GetRequiredSize(m_MaxName, Limits::MaxFilePath);
        break;

    case ResDataType_SequenceSound:
        dataSize = ResSequenceSoundInfo::GetRequiredSize(m_MaxName, Limits::MaxFilePath);
        break;

    case ResDataType_Bank:
        dataSize = ResBankInfo::GetRequiredSize(Limits::MaxFilePath);
        break;

    default:
        return Result(ResultType_Failed);
    }

    CacheManager::Item* item = m_ItemInfoManager.GetItem(itemName);

    // 既存のキャッシュがある場合は、データを上書きします。
    if(item != nullptr)
    {
        uint32_t cacheItemType = item->GetDataType();

        // 違う種類のデータをキャッシュすることはないようにする
        if(cacheItemType != ResDataType_Unknown &&
            cacheItemType != static_cast<uint32_t>(itemType))
        {
            NN_SDK_ASSERT(false, "[sndedit] invalid item data type.\n");
            return Result(ResultType_Failed);
        }

        void* itemData = item->GetData();
        NN_SDK_ASSERT_NOT_NULL(itemData);
        std::memcpy(itemData, itemInfo, dataSize);
        item->SetDataType(itemType);

        return Result(ResultType_True);
    }

    bool isOverrideOriginal = originalId != SoundArchive::InvalidId;
    uint32_t id;
    if (isOverrideOriginal)
    {
        id = originalId;
    }
    else
    {
        if(!m_ItemInfoManager.TryGetFreeID(&id))
        {
            return Result(ResultType_CacheOverFlow);
        }
    }

    return m_ItemInfoManager.AddItem(id, itemName, isOverrideOriginal, itemType, itemInfo, dataSize, false);
}

//----------------------------------------------------------
const void*
ResourceManager::GetFile(
    const char* filePath,
    ResDataType fileDataType,
    const Hash32** hashCode /*= nullptr*/) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(filePath);
    NN_SDK_ASSERT(*filePath != '\0');

    NN_SDK_ASSERT(IsInitialized(), "SoundArchiveFileEditor is not initialized.\n");

    // 該当キャッシュがあれば、それを返します。
    const CacheManager::Item* cache = m_FileManager.GetItem(filePath);

    if(cache == nullptr)
    {
        return nullptr;
    }

    if(cache->GetDataType() != static_cast<uint32_t>(fileDataType))
    {
        return nullptr;
    }

    if(hashCode != nullptr)
    {
        *hashCode = &cache->GetHashCode();
    }

    // ファイル情報を再生用バッファにコピー
    void* pPlaybackBuffer = const_cast<CacheManager::Item*>(cache)->GetPlaybackBuffer();
    const void* fileBuffer = util::ConstBytePtr(cache->GetData()).AlignUp(Alignments::FileIoBuffer).Get();
    std::memcpy(pPlaybackBuffer, fileBuffer, cache->GetDataSize());

    return pPlaybackBuffer;
}

//----------------------------------------------------------
Result
ResourceManager::IncrementFileReferenceCount(const char* filePath) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(filePath);
    NN_SDK_ASSERT(*filePath != '\0');

    CacheManager::Item* cache = m_FileManager.GetItem(filePath);

    if(cache == nullptr)
    {
        return Result(ResultType_NameNotFound);
    }

    cache->IncrementReferenceCount();

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO("[sndedit] ResourceManager::IncrementFileReferenceCount : Count=%d, Name=%s\n", cache->GetReferenceCount(), filePath);
#endif

    return Result(ResultType_True);
}

//----------------------------------------------------------
Result
ResourceManager::DecrementFileReferenceCount(const char* filePath) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(filePath);
    NN_SDK_ASSERT(*filePath != '\0');

    CacheManager::Item* cache = m_FileManager.GetItem(filePath);

    if(cache == nullptr)
    {
        return Result(ResultType_NameNotFound);
    }

    cache->DecrementReferenceCount();

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO("[sndedit] ResourceManager::DecrementReferenceCount : Count=%d, Name=%s\n", cache->GetReferenceCount(), filePath);
#endif

    if(cache->GetReferenceCount() == 0)
    {
        RemoveFile(filePath);
    }

    return Result(ResultType_True);
}

//----------------------------------------------------------
Result
ResourceManager::NewItemInfoEntry(const char* name, uint32_t originalId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(name);

    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    // アイテム数が上限に達した場合は、辞書を拡張します。
    if(m_ItemInfoManager.GetItemCount() == m_ItemInfoManager.GetMaxItemCount())
    {
        uint32_t newMaxItems = m_ItemInfoManager.GetItemCount() * 2;

        if(newMaxItems == 0)
        {
            newMaxItems = 32;
        }

        Result result = RebuildItemInfoDictionary(newMaxItems);

        if(result.IsFailed())
        {
            return result;
        }
        else if(result.IsFalse())
        {
            return Result(ResultType_CacheOverFlow);
        }
    }

    bool isOverrideOriginal = originalId != SoundArchive::InvalidId;
    uint32_t id;
    if (isOverrideOriginal)
    {
        id = originalId;
    }
    else
    {
        if(!m_ItemInfoManager.TryGetFreeID(&id))
        {
            return Result(ResultType_CacheOverFlow);
        }
    }

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO("[sndedit] ResourceManager::NewItemInfoEntry : OriginalId=%x, Id=%x\n", static_cast<int>(originalId), static_cast<int>(id));
#endif

    CacheManager::Item* newCache = nullptr;

    Result result = m_ItemInfoManager.CreateItemEntry(
        newCache,
        id,
        name,
        isOverrideOriginal,
        ResItemInfoUtility::GetMaxItemInfoSize(m_MaxName, Limits::MaxFilePath));

    if(result.IsFailed())
    {
        return result;
    }

    result = m_ItemInfoManager.AddItem(*newCache);

    if(result.IsFailed())
    {
        newCache->Dispose();
        return result;
    }

    newCache->SetUserParameter(ResourceState_Updating);

    return result;
}

//----------------------------------------------------------
Result
ResourceManager::NewFileEntry(
    const char* filePath,
    size_t fileSize,
    ResDataType fileDataType,
    void** filebuffer,
    Hash32* hashCode /*= nullptr*/) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(filePath);
    NN_SDK_ASSERT(*filePath != '\0');

    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    // ファイル数が上限に達した場合は、辞書を拡張します。
    if(m_FileManager.GetItemCount() == m_FileManager.GetMaxItemCount())
    {
        uint32_t newMaxFiles = m_FileManager.GetItemCount() * 2;

        if(newMaxFiles == 0)
        {
            newMaxFiles = 32;
        }

        Result result = RebuildFileDictionary(newMaxFiles);

        if(result.IsFailed())
        {
            return result;
        }
        else if(result.IsFalse())
        {
            return Result(ResultType_CacheOverFlow);
        }
    }

    CacheManager::Item* newCache = nullptr;

    Result result = m_FileManager.CreateFileEntry(newCache, 0, filePath, false, fileSize);

    // メモリ不足なら、未使用ファイルを削除（ガベージコレクション）して再試行します。
    if(result == ResultType_OutOfMemory)
    {
        RemoveAllGarbageItemFiles();

        result = m_FileManager.CreateFileEntry(newCache, 0, filePath, false, fileSize);
    }

    if(result.IsFailed())
    {
        return result;
    }

    newCache->SetDataType(fileDataType);

    if(hashCode != nullptr)
    {
        newCache->SetHashCode(*hashCode);
    }

    result = m_FileManager.AddItem(*newCache);

    if(result.IsFailed())
    {
        newCache->Dispose();
        return result;
    }

    if(filebuffer != nullptr)
    {
        *filebuffer = newCache->GetData();
    }

    return result;
}

//----------------------------------------------------------
void
ResourceManager::RemoveAllItemInfos() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    m_ItemInfoManager.RemoveAllItems();

    if(m_IsItemInfoDictionaryExpandable)
    {
        RebuildItemInfoDictionary(0);
    }
    else
    {
        UpdateItemInfoDictionary();
    }
}

//----------------------------------------------------------
void
ResourceManager::RemoveAllItemFiles() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    atk::SoundSystem::LockSoundThread();
    m_FileManager.RemoveAllItems(InvalidateDataBuffer);
    atk::SoundSystem::UnlockSoundThread();

    if(m_IsItemInfoDictionaryExpandable)
    {
        RebuildFileDictionary(0);
    }
    else
    {
        UpdateFileDictionary();
    }
}

//----------------------------------------------------------
void
ResourceManager::RemoveAllGarbageItemFiles() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    atk::SoundSystem::LockSoundThread();
    m_FileManager.RemoveGarbages();
    atk::SoundSystem::UnlockSoundThread();

    if(m_IsItemInfoDictionaryExpandable)
    {
        RebuildFileDictionary(0);
    }
    else
    {
        UpdateFileDictionary();
    }
}

//----------------------------------------------------------
void
ResourceManager::ClearAllItemFileReferenceCounts() NN_NOEXCEPT
{
    m_FileManager.ClearAllItemReferenceCounts();
}

//----------------------------------------------------------
Result
ResourceManager::RemoveItemData(const char* name) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    if(name == nullptr || name[0] == '\0')
    {
        return Result(ResultType_False);
    }

    // 関連するファイルキャッシュの参照カウントをデクリメントします。
    DecrementReferenceCountForItemFiles(name);

    // アイテム情報キャッシュを削除します。
    Result result = m_ItemInfoManager.RemoveItem(name);

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[sndedit] ResourceManager::RemoveItemData() : result=%s, name=%s\n",
        result.ToString(),
        name);
#endif

    return result;
}

//----------------------------------------------------------
Result
ResourceManager::RemoveItemFiles(const char* name) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    if(name == nullptr || name[0] == '\0')
    {
        return Result(ResultType_False);
    }

    // 関連するファイルキャッシュを削除します。
    Result result = ForeachFileCache(name, SafeRemoveCache);

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[sndedit] ResourceManager::RemoveItemFiles() : result=%s, name=%s\n",
        result.ToString(),
        name);
#endif

    return result;
}

//----------------------------------------------------------
Result
ResourceManager::DecrementReferenceCountForItemFiles(const char* name) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    if(name == nullptr || name[0] == '\0')
    {
        return Result(ResultType_False);
    }

    // 関連するファイルキャッシュの参照カウントをデクリメントします。
    Result result = ForeachFileCache(name, SafeDecrementReferenceCount);

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[sndedit] ResourceManager::DecrementReferenceCountForItemFiles() : result=%s, name=%s\n",
        result.ToString(),
        name);
#endif

    return result;
}

//----------------------------------------------------------
void
ResourceManager::RemoveFile(const char* filePath) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(filePath);
    NN_SDK_ASSERT(*filePath != '\0');

    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    m_FileManager.RemoveItem(filePath, InvalidateDataBuffer);
}

//----------------------------------------------------------
const char*
ResourceManager::GetFirstUnfreezedItemName(ResourceState* state /*= nullptr*/) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    // メモリ確保を優先するために削除中のアイテムを先に探します。
    const CacheManager::Item* item = m_ItemInfoManager.GetFirstUnfreezedItem(IsRemovingCacheItem);

    if(item != nullptr)
    {
        if(state != nullptr)
        {
            *state = static_cast<ResourceState>(item->GetUserParameter());
        }

        return item->GetName().GetName();
    }

    item = m_ItemInfoManager.GetFirstUnfreezedItem();

    if(item == nullptr)
    {
        if(state != nullptr)
        {
            *state = ResourceState_Unknown;
        }

        return nullptr;
    }

    if(state != nullptr)
    {
        *state = static_cast<ResourceState>(item->GetUserParameter());
    }

    return item->GetName().GetName();
}

//----------------------------------------------------------
Result
ResourceManager::FreezeItemInfo(const char* itemName, ResourceState* state /*= nullptr*/) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(itemName);
    NN_SDK_ASSERT(*itemName != '\0');

    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    CacheManager::Item* item = m_ItemInfoManager.GetItem(itemName);

    if(item == nullptr)
    {
        if(state != nullptr)
        {
            *state = ResourceState_Unknown;
        }

        return Result(ResultType_NameNotFound);
    }

    item->Freeze();

    if(state != nullptr)
    {
        *state = static_cast<ResourceState>(item->GetUserParameter());
    }

    return Result(ResultType_True);
}

//----------------------------------------------------------
Result
ResourceManager::UnfreezeAllItemInfos(ResourceState state) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    CacheUnfreezer unfreezer(state);
    m_ItemInfoManager.ForEach(unfreezer);

    return Result(ResultType_True);
}

//----------------------------------------------------------
Result
ResourceManager::UnfreezeItemInfo(const char* itemName, ResourceState state) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(itemName);
    NN_SDK_ASSERT(*itemName != '\0');

    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    CacheManager::Item* item = m_ItemInfoManager.GetItem(itemName);

    if(item == nullptr)
    {
        return Result(ResultType_NameNotFound);
    }

    item->SetUserParameter(state);
    item->Unfreeze();

    return Result(ResultType_True);
}

//----------------------------------------------------------
void
ResourceManager::UpdateItemInfoDictionary() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    m_ItemInfoManager.UpdateDictionary();
}

//----------------------------------------------------------
void
ResourceManager::UpdateFileDictionary() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "ResourceManager is not initialized.\n");

    m_FileManager.UpdateDictionary();
}

ResourceManager::ErrorItem* ResourceManager::CreateErrorItem(const char* itemName, nn::atk::viewer::Result result) NN_NOEXCEPT
{
    int itemLength = static_cast<int>(sizeof(char) * (std::strlen(itemName) + 1));

    void* itemNameBuffer = m_ErrorItemHeap.Alloc(itemLength);
    void* memory = m_ErrorItemHeap.Alloc(sizeof(ErrorItem));
    if (itemNameBuffer == nullptr || memory == nullptr)
    {
        return nullptr;
    }

    ErrorItem* item = new (memory) ErrorItem();
    char* lastBuffer = item->AttachItemNameBuffer(static_cast<char*>(itemNameBuffer), itemLength);
    if (lastBuffer != nullptr)
    {
        m_ErrorItemHeap.Free(lastBuffer);
    }

    item->SetResult(result);
    item->SetItemName(itemName);

    return item;
}

void ResourceManager::DestroyErrorItem(ErrorItem* item) NN_NOEXCEPT
{
    char* lastBuffer = item->AttachItemNameBuffer(nullptr, 0);
    if (lastBuffer != nullptr)
    {
        m_ErrorItemHeap.Free(lastBuffer);
    }
    item->~ErrorItem();
    m_ErrorItemHeap.Free(item);
    item = nullptr;
}

void ResourceManager::DestroyAllErrorItems() NN_NOEXCEPT
{
    ErrorItem* errorItem = PopErrorItem();
    while (errorItem != nullptr)
    {
        DestroyErrorItem(errorItem);
        errorItem = PopErrorItem();
    }
}

void ResourceManager::PushErrorItem(ErrorItem* item) NN_NOEXCEPT
{
    m_ErrorItemList.push_back(item->GetReference());
}

ResourceManager::ErrorItem* ResourceManager::PopErrorItem() NN_NOEXCEPT
{
    if (m_ErrorItemList.empty())
    {
        return nullptr;
    }

    ErrorItem* item = m_ErrorItemList.begin()->value;
    m_ErrorItemList.pop_front();
    return item;
}

size_t ResourceManager::GetErrorItemHeapRequiredMemorySize(int maxName) const NN_NOEXCEPT
{
    // ErrorItemのサイズ + 文字列バッファのサイズ + 1 ブロックの管理領域サイズ
    size_t oneItemSize = nn::util::align_up(
        sizeof(ErrorItem) + ( sizeof(char) * ( maxName + 1 ) ) + atk::detail::fnd::ExpandedHeap::BlockMdSize,
        atk::detail::fnd::MemoryTraits::DefaultAlignment );

    // ヒープ全体の管理領域のサイズ + 1 ErrorItem が使用する最大メモリサイズ * 同時に処理可能なエラー数
    return atk::detail::fnd::ExpandedHeap::HeapMdSize + oneItemSize * 64;
}

//----------------------------------------------------------
size_t
ResourceManager::GetItemInfoManagerRequiredMemorySize(int maxItems, int maxFiles, int maxName) const NN_NOEXCEPT
{
    NN_UNUSED(maxFiles);
    size_t result = m_ItemInfoManager.GetRequiredMemorySize(maxItems);

    result += GetItemInfoDataBufferLength(
        maxItems,
        maxName,
        ResItemInfoUtility::GetMaxItemInfoSize(maxName, Limits::MaxFilePath));

    return result;
}

//----------------------------------------------------------
Result
ResourceManager::InitializeItemInfoManager(uint32_t baseId, int maxEntries, int maxName) NN_NOEXCEPT
{
    size_t bufferLength = m_ItemInfoManager.GetRequiredMemorySize(maxEntries);
    m_pMemoryForItemInfoManager = bufferLength == 0 ? nullptr : m_ResourceAllocator.Alloc(bufferLength);

    if(bufferLength > 0 && m_pMemoryForItemInfoManager == nullptr)
    {
        return Result(ResultType_OutOfMemory);
    }

    CacheManager::InitializeArgs args;
    args.buffer = m_pMemoryForItemInfoManager;
    args.bufferLength = bufferLength;
    args.dataBufferAllocator = &m_ResourceAllocator;
    args.playbackBufferAllocator = &m_PlaybackBufferAllocator;
    args.baseId = baseId;
    args.maxEntries = maxEntries;
    args.maxName = maxName;

    return m_ItemInfoManager.Initialize(args);
}

//----------------------------------------------------------
Result
ResourceManager::InitializeFileManager(int maxEntries) NN_NOEXCEPT
{
    size_t bufferLength = m_FileManager.GetRequiredMemorySize(maxEntries);
    m_pMemoryForFileManager = bufferLength == 0 ? nullptr : m_ResourceAllocator.Alloc(bufferLength);

    if(bufferLength > 0 && m_pMemoryForFileManager == nullptr)
    {
        return Result(ResultType_OutOfMemory);
    }

    CacheManager::InitializeArgs args;
    args.buffer = m_pMemoryForFileManager;
    args.bufferLength = bufferLength;
    args.dataBufferAllocator = &m_ResourceAllocator;
    args.playbackBufferAllocator = &m_PlaybackBufferAllocator;
    args.maxEntries = maxEntries;
    args.maxName = Limits::MaxFilePath;

    return m_FileManager.Initialize(args);
}

//----------------------------------------------------------
Result
ResourceManager::RebuildItemInfoDictionary(int maxEntries) NN_NOEXCEPT
{
    if(!m_IsItemInfoDictionaryExpandable)
    {
        return Result(ResultType_False);
    }

    if(!m_ItemInfoManager.IsInitialized())
    {
        NN_SDK_ASSERT(false, "ItemInfoManager is not initialized.\n");
        return Result(ResultType_NotInitialized);
    }

    if (maxEntries > m_ItemInfoManager.GetMaxItemCount())
    {
        return Result(ResultType_False);
    }

    size_t bufferLength = m_ItemInfoManager.GetRequiredMemorySize(maxEntries);
    void* buffer = m_ResourceAllocator.Alloc(bufferLength);

    if(buffer == nullptr)
    {
        return Result(ResultType_OutOfMemory);
    }

    void* oldBuffer = nullptr;
    Result result = m_ItemInfoManager.RebuildDictionary(buffer, bufferLength, maxEntries, &oldBuffer);

    if(result.IsSucceeded() && oldBuffer != nullptr)
    {
        m_ResourceAllocator.Free(oldBuffer);
    }

    return result;
}

//----------------------------------------------------------
Result
ResourceManager::RebuildFileDictionary(int maxEntries) NN_NOEXCEPT
{
    if(!m_IsFileDictionaryExpandable)
    {
        return Result(ResultType_False);
    }

    if(!m_FileManager.IsInitialized())
    {
        NN_SDK_ASSERT(false, "FileManager is not initialized.\n");
        return Result(ResultType_NotInitialized);
    }

    if (maxEntries > m_FileManager.GetMaxItemCount())
    {
        return Result(ResultType_False);
    }

    size_t bufferLength = m_FileManager.GetRequiredMemorySize(maxEntries);
    void* buffer = m_ResourceAllocator.Alloc(bufferLength);

    if(buffer == nullptr)
    {
        return Result(ResultType_OutOfMemory);
    }

    void* oldBuffer = nullptr;
    Result result = m_FileManager.RebuildDictionary(buffer, bufferLength, maxEntries, &oldBuffer);

    if(result.IsSucceeded() && oldBuffer != nullptr)
    {
        m_ResourceAllocator.Free(oldBuffer);
    }

    return result;
}

//----------------------------------------------------------
size_t
ResourceManager::GetItemInfoDataBufferLength(int maxEntries, int maxName, size_t maxDataLength) const NN_NOEXCEPT
{
    // キャッシュサイズ × エントリー数
    return
        m_ItemInfoManager.GetRequiredMemorySizeForItem(maxName, maxDataLength) * maxEntries;
}

//----------------------------------------------------------
Result
ResourceManager::ForeachFileCache(const char* name, CacheAction action) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(name);
    NN_SDK_ASSERT_NOT_NULL(action);

    uint32_t cacheDataType = ResDataType_Unknown;

    const void* itemInfo = m_ItemInfoManager.GetData(name, &cacheDataType, true);

    if(itemInfo == nullptr)
    {
#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
        NN_DETAIL_ATK_INFO(
            "[sndedit] ResourceManager::ForeachFileCache() : itemInfo not found : name=%s\n",
            name);
#endif
        return Result(ResultType_False);
    }

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO("[sndedit] ResourceManager::ForeachFileCache() : name=%s\n", name);
#endif

    ResDataType itemDataType = static_cast<ResDataType>(cacheDataType);

    switch(itemDataType)
    {
    case ResDataType_WaveSound:
        {
            const ResSoundInfo* soundInfo = reinterpret_cast<const ResSoundInfo*>(itemInfo);

            const ResName* wsdFilePath =
                ResItemInfoUtility::GetFilePath(*soundInfo, itemDataType, ResDataType_WaveSoundFile);

            const ResName* warcFilePath =
                ResItemInfoUtility::GetFilePath(*soundInfo, itemDataType, ResDataType_WaveArchiveFile);

            if(wsdFilePath != nullptr)
            {
                action(m_FileManager, wsdFilePath->GetName());
            }

            if(warcFilePath != nullptr)
            {
                action(m_FileManager, warcFilePath->GetName());
            }
        }
        break;

    case ResDataType_SequenceSound:
        {
            const ResSoundInfo* soundInfo = reinterpret_cast<const ResSoundInfo*>(itemInfo);

            const ResName* seqFilePath =
                ResItemInfoUtility::GetFilePath(*soundInfo, itemDataType, ResDataType_SequenceFile);

            if(seqFilePath != nullptr)
            {
                action(m_FileManager, seqFilePath->GetName());
            }
        }
        break;

    case ResDataType_Bank:
        {
            const ResBankInfo* bankInfo = reinterpret_cast<const ResBankInfo*>(itemInfo);

            const ResName* bankFilePath =
                ResItemInfoUtility::GetFilePath(*bankInfo, ResDataType_BankFile);

            const ResName* warcFilePath =
                ResItemInfoUtility::GetFilePath(*bankInfo, ResDataType_WaveArchiveFile);

            if(bankFilePath != nullptr)
            {
                action(m_FileManager, bankFilePath->GetName());
            }

            if(warcFilePath != nullptr)
            {
                action(m_FileManager, warcFilePath->GetName());
            }
        }
        break;

    case ResDataType_StreamSound:
#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
        NN_DETAIL_ATK_INFO(
            "[sndedit] ResourceManager::ForeachFileCache() : skip stream sound : name=%s\n",
            name);
#endif
        break;

    default:
        NN_ATK_WARNING(
            "[sndedit] ResourceManager::ForeachFileCache() : unknown sound type : name=%s",
            name);
    }

    return Result(ResultType_True);
}

} // namespace nn::atk::viewer::detail
} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
